package ru.yandex.direct.web.core.entity.banner.converter;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.image.container.UploadedBannerImageInformation;
import ru.yandex.direct.core.entity.image.container.UploadedToMdsImageInformation;
import ru.yandex.direct.core.entity.image.model.BannerImageFormatNamespace;
import ru.yandex.direct.core.entity.image.model.ImageSizeMeta;
import ru.yandex.direct.core.entity.image.model.ImageSmartCenter;
import ru.yandex.direct.web.core.entity.banner.model.WebImageFormat;
import ru.yandex.direct.web.core.entity.banner.model.WebImageSize;
import ru.yandex.direct.web.core.entity.banner.model.WebImageSmartCenter;
import ru.yandex.direct.web.core.entity.banner.model.WebMdsNamespace;
import ru.yandex.direct.web.core.entity.banner.model.WebUploadedImage;
import ru.yandex.direct.web.core.entity.banner.model.WebUploadedToMdsImage;

import static ru.yandex.direct.core.entity.image.service.ImageConstants.SUPPORTED_FORMATS_BY_IMAGE_TYPE;
import static ru.yandex.direct.core.entity.image.service.ImageUtils.toDirectImagePath;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class BannerImageConverter {

    public static List<WebUploadedImage> toWebUploadedImages(
            List<UploadedBannerImageInformation> imageInformationList) {
        return mapList(imageInformationList, BannerImageConverter::toWebUploadedImage);
    }

    public static WebUploadedImage toWebUploadedImage(@Nullable UploadedBannerImageInformation imageInformation) {
        if (imageInformation == null) {
            return null;
        }

        List<WebImageFormat> webImageFormats = null;
        if (imageInformation.getMdsMeta() != null) {
            webImageFormats = EntryStream.of(imageInformation.getMdsMeta().getSizes())
                    .filterKeys(SUPPORTED_FORMATS_BY_IMAGE_TYPE.get(imageInformation.getImageType())::contains)
                    .values()
                    .map(BannerImageConverter::toWebImageFormat)
                    .toList();
        }

        return new WebUploadedImage()
                .withHash(imageInformation.getImageHash())
                .withMdsGroupId(imageInformation.getMdsGroupId())
                .withName(imageInformation.getName())
                .withWebMdsNamespace(toWebMdsNamespace(imageInformation.getNamespace()))
                .withWidth(imageInformation.getWidth())
                .withHeight(imageInformation.getHeight())
                .withFormats(webImageFormats);
    }

    private static WebImageFormat toWebImageFormat(ImageSizeMeta imageSizeMeta) {

        return new WebImageFormat()
                .withPath(toDirectImagePath(imageSizeMeta.getPath()))
                .withImageSize(toWebImageSize(imageSizeMeta.getHeight(), imageSizeMeta.getWidth()))
                .withSmartCenters(toWebImageSmartCenters(imageSizeMeta.getSmartCenters()));
    }

    private static WebImageSize toWebImageSize(int height, int width) {
        return new WebImageSize().withWidth(width)
                .withHeight(height);
    }


    @Nullable
    private static List<WebImageSmartCenter> toWebImageSmartCenters(
            @Nullable Map<String, ImageSmartCenter> smartCenters) {
        if (smartCenters == null) {
            return null;
        }

        return mapList(smartCenters.entrySet(), entry -> toWebImageSmartCenter(entry.getKey(), entry.getValue()));
    }

    private static WebImageSmartCenter toWebImageSmartCenter(String ratio, ImageSmartCenter imageSmartCenter) {
        return new WebImageSmartCenter()
                .withRatio(ratio)
                .withHeight(imageSmartCenter.getHeight())
                .withWidth(imageSmartCenter.getWidth())
                .withX(imageSmartCenter.getX())
                .withY(imageSmartCenter.getY());
    }

    @Nullable
    private static WebMdsNamespace toWebMdsNamespace(@Nullable BannerImageFormatNamespace namespace) {
        if (namespace == null) {
            return null;
        }
        switch (namespace) {
            case DIRECT:
                return WebMdsNamespace.DIRECT;
            case DIRECT_PICTURE:
                return WebMdsNamespace.DIRECT_PICTURE;
            default:
                throw new IllegalStateException("No such value: " + namespace);
        }
    }

    public static WebUploadedToMdsImage toWebUploadedToMdsFile(UploadedToMdsImageInformation fileInformation) {
        return new WebUploadedToMdsImage()
                .withName(fileInformation.getName())
                .withHash(fileInformation.getImageHash());
    }
}
