package ru.yandex.direct.web.core.entity.banner.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Objects;

import com.google.common.collect.ImmutableMap;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.service.FeatureHelper;
import ru.yandex.direct.core.entity.image.container.BannerImageType;
import ru.yandex.direct.core.entity.image.model.ImageUploadContainer;
import ru.yandex.direct.core.entity.image.service.ImageConstants;
import ru.yandex.direct.core.entity.image.service.validation.ImageConstraints;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Service
public class BannerImageValidationService {
    private static final Map<BannerImageType, Integer> MAX_IMAGE_SIZE_BY_BANNER_TYPE
            = ImmutableMap.<BannerImageType, Integer>builder()
            .put(BannerImageType.BANNER_MCBANNER, ImageConstants.MAX_IMAGE_FILE_SIZE_FOR_MC_BANNER)
            .put(BannerImageType.BANNER_TEXT, ImageConstants.MAX_IMAGE_FILE_SIZE_FOR_TEXT_BANNER)
            .put(BannerImageType.BANNER_IMAGE_AD, ImageConstants.MAX_IMAGE_FILE_SIZE_FOR_TEXT_IMAGE_BANNER)
            .put(BannerImageType.BANNER_INTERNAL, ImageConstants.MAX_IMAGE_FILE_SIZE_FOR_INTERNAL_BANNER)
            .put(BannerImageType.ASSET_LOGO, ImageConstants.MAX_IMAGE_FILE_SIZE_FOR_LOGO)
            .put(BannerImageType.ASSET_LOGO_EXTENDED, ImageConstants.MAX_IMAGE_FILE_SIZE_FOR_LOGO)
            .put(BannerImageType.ASSET_MULTICARD, ImageConstants.MAX_IMAGE_FILE_SIZE_FOR_MULTICARD)
            .put(BannerImageType.OFFER_IMAGE, ImageConstants.MAX_IMAGE_FILE_SIZE_FOR_OFFER_IMAGE)
            .build();

    public ValidationResult<String, Defect> validateImageUrl(String url) {
        return ItemValidationBuilder.of(url, Defect.class)
                .check(notNull())
                .check(notBlank())
                .getResult();
    }

    /**
     * Базовая валидация переданных файлов.
     * Проверяет BannerImageType, размер переданного файла и наличие ресурса для внутренней рекламы
     *
     * @param indexes                   - порядковые номера переданных файлов, которые будут использоваться в качестве id
     * @param containers                - контейнеры с файлами, ассоциированные с indexes
     * @param bannerType                - тип баннера
     * @param validationForInternalAd   - временный флаг для внутренней рекламы, можно выпилить после того,
     *                                как фронт перейдет на использование InternalAdImageController
     * @return
     */
    public ValidationResult<List<Integer>, Defect> validateImages(List<Integer> indexes,
                                                                  Map<Integer, ImageUploadContainer> containers,
                                                                  BannerImageType bannerType,
                                                                  boolean validationForInternalAd) {
        boolean allowProportionallyImages = FeatureHelper
                .feature(FeatureName.ALLOW_PROPORTIONALLY_LARGER_IMAGES)
                .enabled();

        ListValidationBuilder<Integer, Defect> lvb = ListValidationBuilder.of(indexes, Defect.class)
                .check(imageTypeSupported(bannerType));
        if (!lvb.getResult().hasAnyErrors()) {
            lvb
                    .checkEach(resourceExists(containers), When.isTrue(validationForInternalAd))
                    .checkEach(imageFileSizeIsValid(
                            containers,
                            getMaxImageSizeByBannerType(bannerType, allowProportionallyImages)
                    ));
        }
        return lvb.getResult();
    }

    private Constraint<Integer, Defect> imageFileSizeIsValid(Map<Integer, ImageUploadContainer> containers, int maxSize) {
        return index -> ImageConstraints.imageFileSizeIsValid((int) containers.get(index).getFile().getSize(), maxSize)
                .apply(index);
    }

    private Constraint<List<Integer>, Defect> imageTypeSupported(BannerImageType imageType) {
        return Constraint.fromPredicate(index -> MAX_IMAGE_SIZE_BY_BANNER_TYPE.containsKey(imageType),
                CommonDefects.invalidValue());
    }

    private Constraint<Integer, Defect> resourceExists(Map<Integer, ImageUploadContainer> containers) {
        return Constraint.fromPredicate(i -> Objects.nonNull(containers.get(i).getResource()), CommonDefects.requiredButEmpty());
    }

    private int getMaxImageSizeByBannerType(BannerImageType bannerType, boolean allowProportionallyImages) {

        if (bannerType == BannerImageType.BANNER_IMAGE_AD && allowProportionallyImages) {
            return ImageConstants.NEW_MAX_IMAGE_FILE_SIZE_FOR_TEXT_IMAGE_BANNER;
        }
        return MAX_IMAGE_SIZE_BY_BANNER_TYPE.get(bannerType);
    }
}
