package ru.yandex.direct.web.core.entity.bidmodifiers;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.bidmodifier.AbstractBidModifierRetargetingAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.AgeType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierABSegment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierABSegmentAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographicsAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnly;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnlyAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpression;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpressionAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierGeo;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobileAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierPerformanceTgo;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierPerformanceTgoAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRegionalAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargeting;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierSmartTV;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierSmartTVAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTablet;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTabletAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTrafaretPosition;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierVideo;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierVideoAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeather;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeatherAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeatherLiteral;
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier;
import ru.yandex.direct.core.entity.bidmodifier.GenderType;
import ru.yandex.direct.core.entity.bidmodifier.OperationType;
import ru.yandex.direct.core.entity.bidmodifier.OsType;
import ru.yandex.direct.core.entity.bidmodifier.TabletOsType;
import ru.yandex.direct.core.entity.bidmodifier.WeatherType;
import ru.yandex.direct.core.entity.bidmodifier.model.BidModifierExpressionLiteral;
import ru.yandex.direct.core.entity.bidmodifiers.Constants;
import ru.yandex.direct.core.entity.bidmodifiers.expression.ParameterType;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierExpressionFactory;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierABSegmentConditionWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierABSegmentWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierDemographyConditionWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierDemographyWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierExpressionAdjustmentWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierExpressionLiteralWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierExpressionWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierGeoRegionWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierGeoWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierMobileWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierRetargetingConditionWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierRetargetingWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierSingleWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierTabletWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierWeatherConditionWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierWeatherLiteralWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifierWeatherWeb;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.BidModifiersListWebResponse;
import ru.yandex.direct.web.core.entity.bidmodifiers.model.ComplexBidModifierWeb;
import ru.yandex.direct.web.core.exception.WebBadRequestException;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.web.core.entity.bidmodifiers.HelperConverter.booleanFromInt;
import static ru.yandex.direct.web.core.entity.bidmodifiers.HelperConverter.booleanToInt;

public class ComplexBidModifierConverter {
    // мапы для приведения lowercase строки значениям енумов
    private static Map<String, BidModifierType> defaultStringToTypeMap = lcToEnum(BidModifierType.values());

    private ComplexBidModifierConverter() {
    }

    private static <T extends Enum> Map<String, T> lcToEnum(T[] values) {
        return StreamEx.of(values).mapToEntry(t -> t.toString().toLowerCase(), Function.identity()).toImmutableMap();
    }

    public static ComplexBidModifier fromWeb(ComplexBidModifierWeb web) {
        ComplexBidModifier complexBidModifier = new ComplexBidModifier();
        if (web.getMobileModifier() != null) {
            BidModifierMobile bidModifier = new BidModifierMobile()
                    .withType(BidModifierType.MOBILE_MULTIPLIER)
                    .withEnabled(true)
                    .withMobileAdjustment(
                            new BidModifierMobileAdjustment()
                                    .withOsType(osTypeFromExternal(web.getMobileModifier().getOsType()))
                                    .withPercent(web.getMobileModifier().getPercent()));
            complexBidModifier.withMobileModifier(bidModifier);
        }
        if (web.getDesktopModifier() != null) {
            BidModifierDesktop bidModifier = new BidModifierDesktop()
                    .withType(BidModifierType.DESKTOP_MULTIPLIER)
                    .withEnabled(true)
                    .withDesktopAdjustment(
                            new BidModifierDesktopAdjustment()
                                    .withPercent(web.getDesktopModifier().getPercent()));
            complexBidModifier.withDesktopModifier(bidModifier);
        }
        if (web.getTabletModifier() != null) {
            BidModifierTablet bidModifier = new BidModifierTablet()
                    .withType(BidModifierType.TABLET_MULTIPLIER)
                    .withEnabled(true)
                    .withTabletAdjustment(
                            new BidModifierTabletAdjustment()
                                    .withOsType(tabletOsTypeFromExternal(web.getTabletModifier().getOsType()))
                                    .withPercent(web.getTabletModifier().getPercent()));
            complexBidModifier.withTabletModifier(bidModifier);
        }
        if (web.getDesktopOnlyModifier() != null) {
            BidModifierDesktopOnly bidModifier = new BidModifierDesktopOnly()
                    .withType(BidModifierType.DESKTOP_ONLY_MULTIPLIER)
                    .withEnabled(true)
                    .withDesktopOnlyAdjustment(
                            new BidModifierDesktopOnlyAdjustment()
                                    .withPercent(web.getDesktopOnlyModifier().getPercent()));
            complexBidModifier.withDesktopOnlyModifier(bidModifier);
        }
        if (web.getSmartTVModifier() != null) {
            BidModifierSmartTV bidModifier = new BidModifierSmartTV()
                    .withType(BidModifierType.SMARTTV_MULTIPLIER)
                    .withEnabled(true)
                    .withSmartTVAdjustment(
                            new BidModifierSmartTVAdjustment()
                                    .withPercent(web.getSmartTVModifier().getPercent()));
            complexBidModifier.withSmartTVModifier(bidModifier);
        }
        if (web.getVideoModifier() != null) {
            BidModifierVideo bidModifier = new BidModifierVideo()
                    .withType(BidModifierType.VIDEO_MULTIPLIER)
                    .withEnabled(true)
                    .withVideoAdjustment(
                            new BidModifierVideoAdjustment()
                                    .withPercent(web.getVideoModifier().getPercent()));
            complexBidModifier.withVideoModifier(bidModifier);
        }
        if (web.getPerformanceTgoModifier() != null) {
            BidModifierPerformanceTgo bidModifier = new BidModifierPerformanceTgo()
                    .withType(BidModifierType.PERFORMANCE_TGO_MULTIPLIER)
                    .withEnabled(true)
                    .withPerformanceTgoAdjustment(
                            new BidModifierPerformanceTgoAdjustment()
                                    .withPercent(web.getPerformanceTgoModifier().getPercent()));
            complexBidModifier.withPerformanceTgoModifier(bidModifier);
        }
        if (web.getGeoModifier() != null) {
            List<BidModifierRegionalAdjustment> regions =
                    web.getGeoModifier().getRegions().stream()
                            .map(it -> new BidModifierRegionalAdjustment()
                                    .withRegionId(it.getRegionId())
                                    .withPercent(it.getPercent())
                                    .withHidden(false))
                            .collect(toList());
            BidModifierGeo bidModifier = new BidModifierGeo()
                    .withType(BidModifierType.GEO_MULTIPLIER)
                    .withEnabled(booleanFromInt(web.getGeoModifier().isEnabled()))
                    .withRegionalAdjustments(regions);
            complexBidModifier.withGeoModifier(bidModifier);
        }
        if (web.getDemographyModifier() != null) {
            List<BidModifierDemographicsAdjustment> conditions =
                    web.getDemographyModifier().getConditions().stream()
                            .map(it -> new BidModifierDemographicsAdjustment()
                                    .withAge(ageFromExternal(it.getAge()))
                                    .withGender(genderFromExternal(it.getGender()))
                                    .withPercent(it.getPercent()))
                            .collect(toList());
            BidModifierDemographics bidModifier = new BidModifierDemographics()
                    .withType(BidModifierType.DEMOGRAPHY_MULTIPLIER)
                    .withEnabled(booleanFromInt(web.getDemographyModifier().isEnabled()))
                    .withDemographicsAdjustments(conditions);
            complexBidModifier.withDemographyModifier(bidModifier);
        }
        if (web.getWeatherModifier() != null) {
            List<BidModifierWeatherAdjustment> conditions =
                    web.getWeatherModifier().getConditions().stream()
                            .map(it -> new BidModifierWeatherAdjustment()
                                    .withExpression(expressionFromExternal(it.getConditionJson()))
                                    .withPercent(it.getPercent()))
                            .collect(toList());
            BidModifierWeather bidModifier = new BidModifierWeather()
                    .withType(BidModifierType.WEATHER_MULTIPLIER)
                    .withEnabled(booleanFromInt(web.getWeatherModifier().isEnabled()))
                    .withWeatherAdjustments(conditions);
            complexBidModifier.withWeatherModifier(bidModifier);
        }
        if (web.getRetargetingModifier() != null) {
            List<AbstractBidModifierRetargetingAdjustment> conditions =
                    EntryStream.of(web.getRetargetingModifier().getConditions())
                            .mapKeyValue((retCondIdStr, conditionWeb) -> {
                                Long retCondId = Long.valueOf(retCondIdStr);
                                var adjustment = new BidModifierRetargetingAdjustment()
                                        .withRetargetingConditionId(retCondId)
                                        .withPercent(conditionWeb.getPercent());
                                return (AbstractBidModifierRetargetingAdjustment) adjustment;
                            })
                            .toList();
            BidModifierRetargeting bidModifier = new BidModifierRetargeting()
                    .withType(BidModifierType.RETARGETING_MULTIPLIER)
                    .withEnabled(booleanFromInt(web.getRetargetingModifier().isEnabled()))
                    .withRetargetingAdjustments(conditions);
            complexBidModifier.withRetargetingModifier(bidModifier);
        }
        if (web.getAbSegmentModifier() != null) {
            List<BidModifierABSegmentAdjustment> conditions =
                    web.getAbSegmentModifier().getConditions().stream()
                            .map(cond -> new BidModifierABSegmentAdjustment()
                                    .withPercent(cond.getPercent())
                                    .withSectionId(cond.getSectionId())
                                    .withSegmentId(cond.getSegmentId())
                                    .withId(cond.getAbSegmentMultiplierValueId()))
                            .collect(toList());

            BidModifierABSegment bidModifier = new BidModifierABSegment()
                    .withType(BidModifierType.AB_SEGMENT_MULTIPLIER)
                    .withEnabled(web.getAbSegmentModifier().isEnabled())
                    .withAbSegmentAdjustments(conditions);
            complexBidModifier.withAbSegmentModifier(bidModifier);
        }

        if (web.getInventoryModifier() != null) {
            complexBidModifier.withInventoryModifier(InventoryModifierConverter.convert(web.getInventoryModifier()));
        }
        if (web.getBannerTypeModifier() != null) {
            complexBidModifier.withBannerTypeModifier(BannerTypeModifierConverter.convert(web.getBannerTypeModifier()));
        }
        if (web.getBidModifierExpression() != null && !web.getBidModifierExpression().isEmpty()) {
            List<BidModifierExpression> expressionModifiers = mapList(
                    web.getBidModifierExpression(),
                    ComplexBidModifierConverter::webExpressionBidModifierToCore
            );
            // В этой модели все типы универсальных корректировок должны указываться только по одному разу
            // Иначе непонятно, какой из указанных наборов нужно применить
            checkState(expressionModifiers.stream().map(BidModifier::getType).distinct().count()
                    == expressionModifiers.size());
            complexBidModifier.withExpressionModifiers(expressionModifiers);
        }
        if (web.getTrafaretPositionModifier() != null) {
            complexBidModifier
                    .withTrafaretPositionModifier(TrafaretPositionModifierConverter.convert(web.getTrafaretPositionModifier()));
        }

        return complexBidModifier;
    }

    public static BidModifierExpression webExpressionBidModifierToCore(BidModifierExpressionWeb bme) {
        BidModifierType type = parseBidModifierType(bme.getType());
        BidModifierExpression bidModifier = BidModifierExpressionFactory.createBidModifierByType(type);
        return bidModifier
                .withType(type)
                .withEnabled(bme.getEnabled() != 0)
                .withExpressionAdjustments(
                        mapList(bme.getAdjustments(), adj -> webExpressionBidModifierAdjustmentToCore(adj, type))
                );
    }


    private static BidModifierExpressionAdjustment webExpressionBidModifierAdjustmentToCore(
            BidModifierExpressionAdjustmentWeb s,
            BidModifierType type
    ) {
        BidModifierExpressionAdjustment adj = BidModifierExpressionFactory.createBidModifierAdjustmentByType(type);
        return adj
                .withPercent(s.getPercent())
                .withCondition(
                        mapList(s.getCondition(),
                                exp -> mapList(
                                        exp,
                                        lit -> webExpressionBidModifierLiteralToCore(lit, type)
                                ))
                );
    }

    private static BidModifierExpressionLiteral webExpressionBidModifierLiteralToCore(
            BidModifierExpressionLiteralWeb lit,
            BidModifierType type) {
        var param = lit.getParameter();
        BidModifierExpressionLiteral literal = new BidModifierExpressionLiteral()
                .withParameter(param)
                .withOperation(lit.getOperation());
        var parameterInfo = Constants.ALL_PARAMETERS.get(type).get(param);
        checkNotNull(parameterInfo);
        if (parameterInfo.getType() == ParameterType.INTEGER) {
            String value = lit.getValue();
            try {
                literal.setValueInteger(Integer.parseInt(value));
            } catch (NumberFormatException e) {
                throw new WebBadRequestException("Failed to parse %s as integer", value);
            }
        } else {
            literal.setValueString(lit.getValue());
        }

        return literal;
    }

    private static BidModifierType parseBidModifierType(String type) {
        if (!defaultStringToTypeMap.containsKey(type)) {
            throw new WebBadRequestException("Unknown BidModifier type '%s'", type);
        }
        BidModifierType coreType = defaultStringToTypeMap.get(type);
        if (BidModifierExpressionFactory.notExpressionBidModifierType(coreType)) {
            throw new WebBadRequestException("BidModifier type %s is not an expression bid modifier type", type);
        }
        return coreType;
    }

    public static BidModifiersListWebResponse convertToExternal(List<BidModifier> bidModifiers) {
        BidModifiersListWebResponse response = new BidModifiersListWebResponse();
        for (BidModifier bidModifier : bidModifiers) {

            if (bidModifier instanceof BidModifierExpression) {
                if (response.getBidModifierExpression() == null) {
                    response.setBidModifierExpression(new ArrayList<>());
                }
                response.getBidModifierExpression().add(
                        convertExpressBidModifierToExternal((BidModifierExpression) bidModifier)
                );
                continue;
            }

            switch (bidModifier.getType()) {
                case MOBILE_MULTIPLIER: {
                    BidModifierMobile modifier = (BidModifierMobile) bidModifier;
                    response.withBidModifierMobile(
                            new BidModifierMobileWeb()
                                    .withId(modifier.getId())
                                    .withOsType(osTypeToExternal(modifier.getMobileAdjustment().getOsType()))
                                    .withLastChange(modifier.getMobileAdjustment().getLastChange().toString())
                                    .withPercent(modifier.getMobileAdjustment().getPercent()));
                    break;
                }
                case TABLET_MULTIPLIER: {
                    BidModifierTablet modifier = (BidModifierTablet) bidModifier;
                    response.withBidModifierTablet(
                            new BidModifierTabletWeb()
                                    .withId(modifier.getId())
                                    .withOsType(osTypeToExternal(modifier.getTabletAdjustment().getOsType()))
                                    .withLastChange(modifier.getTabletAdjustment().getLastChange().toString())
                                    .withPercent(modifier.getTabletAdjustment().getPercent()));
                    break;
                }
                case DESKTOP_MULTIPLIER: {
                    BidModifierDesktop modifier = (BidModifierDesktop) bidModifier;
                    response.withBidModifierDesktop(
                            new BidModifierSingleWeb()
                                    .withId(modifier.getId())
                                    .withLastChange(modifier.getLastChange().toString())
                                    .withPercent(modifier.getDesktopAdjustment().getPercent()));
                    break;
                }
                case DESKTOP_ONLY_MULTIPLIER: {
                    BidModifierDesktopOnly modifier = (BidModifierDesktopOnly) bidModifier;
                    response.withBidModifierDesktopOnly(
                            new BidModifierSingleWeb()
                                    .withId(modifier.getId())
                                    .withLastChange(modifier.getLastChange().toString())
                                    .withPercent(modifier.getDesktopOnlyAdjustment().getPercent()));
                    break;
                }
                case SMARTTV_MULTIPLIER: {
                    BidModifierSmartTV modifier = (BidModifierSmartTV) bidModifier;
                    response.withBidModifierSmartTV(
                            new BidModifierSingleWeb()
                                    .withId(modifier.getId())
                                    .withLastChange(modifier.getLastChange().toString())
                                    .withPercent(modifier.getSmartTVAdjustment().getPercent()));
                    break;
                }
                case VIDEO_MULTIPLIER: {
                    BidModifierVideo modifier = (BidModifierVideo) bidModifier;
                    response.withBidModifierVideo(
                            new BidModifierSingleWeb()
                                    .withId(modifier.getId())
                                    .withPercent(modifier.getVideoAdjustment().getPercent()));
                    break;
                }
                case PERFORMANCE_TGO_MULTIPLIER: {
                    BidModifierPerformanceTgo modifier = (BidModifierPerformanceTgo) bidModifier;
                    response.withBidModifierPerformanceTgo(
                            new BidModifierSingleWeb()
                                    .withId(modifier.getId())
                                    .withPercent(modifier.getPerformanceTgoAdjustment().getPercent()));
                    break;
                }
                case DEMOGRAPHY_MULTIPLIER: {
                    BidModifierDemographics modifier = (BidModifierDemographics) bidModifier;
                    List<BidModifierDemographyConditionWeb> conditions =
                            modifier.getDemographicsAdjustments().stream()
                                    .map(it -> new BidModifierDemographyConditionWeb()
                                            .withId(it.getId())
                                            .withAge(ageToExternal(it.getAge()))
                                            .withGender(genderToExternal(it.getGender()))
                                            .withPercent(it.getPercent())
                                    ).collect(toList());
                    response.withBidModifierDemography(
                            new BidModifierDemographyWeb()
                                    .withHierarchicalMultiplierId(modifier.getId())
                                    .withEnabled(booleanToInt(modifier.getEnabled()))
                                    .withConditions(conditions));
                    break;
                }
                case WEATHER_MULTIPLIER: {
                    BidModifierWeather modifier = (BidModifierWeather) bidModifier;
                    List<BidModifierWeatherConditionWeb> conditions =
                            modifier.getWeatherAdjustments().stream()
                                    .map(it -> new BidModifierWeatherConditionWeb()
                                            .withId(it.getId())
                                            .withConditionJson(expressionToExternal(it.getExpression()))
                                            .withPercent(it.getPercent())
                                    ).collect(toList());
                    response.withBidModifierWeather(
                            new BidModifierWeatherWeb()
                                    .withHierarchicalMultiplierId(modifier.getId())
                                    .withEnabled(booleanToInt(modifier.getEnabled()))
                                    .withConditions(conditions));
                    break;
                }
                case GEO_MULTIPLIER: {
                    BidModifierGeo modifier = (BidModifierGeo) bidModifier;
                    List<BidModifierGeoRegionWeb> regions = modifier.getRegionalAdjustments().stream()
                            .filter(it -> !it.getHidden())
                            .map(it -> new BidModifierGeoRegionWeb()
                                    .withId(it.getId())
                                    .withRegionId(it.getRegionId())
                                    .withPercent(it.getPercent()))
                            .collect(toList());
                    response.withBidModifierGeo(
                            new BidModifierGeoWeb()
                                    .withHierarchicalMultiplierId(modifier.getId())
                                    .withRegions(regions)
                                    .withEnabled(booleanToInt(modifier.getEnabled())));
                    break;
                }
                case RETARGETING_MULTIPLIER: {
                    BidModifierRetargeting modifier = (BidModifierRetargeting) bidModifier;
                    Map<String, BidModifierRetargetingConditionWeb> conditions =
                            modifier.getRetargetingAdjustments().stream()
                                    .collect(toMap(it -> String.valueOf(it.getRetargetingConditionId()),
                                            it -> new BidModifierRetargetingConditionWeb()
                                                    .withId(it.getId())
                                                    .withPercent(it.getPercent())));
                    response.withBidModifierRetargeting(
                            new BidModifierRetargetingWeb()
                                    .withHierarchicalMultiplierId(modifier.getId())
                                    .withEnabled(booleanToInt(modifier.getEnabled()))
                                    .withConditions(conditions));
                    break;
                }
                case AB_SEGMENT_MULTIPLIER: {
                    BidModifierABSegment modifier = (BidModifierABSegment) bidModifier;

                    List<BidModifierABSegmentConditionWeb> conditions =
                            mapList(modifier.getAbSegmentAdjustments(),
                                    it -> new BidModifierABSegmentConditionWeb()
                                            .withPercent(it.getPercent())
                                            .withSectionId(it.getSectionId())
                                            .withSegmentId(it.getSegmentId())
                                            .withAbSegmentMultiplierValueId(it.getId()));

                    response.withBidModifierABSegment(
                            new BidModifierABSegmentWeb()
                                    .withEnabled(modifier.getEnabled())
                                    .withConditions(conditions));
                    break;
                }
                case BANNER_TYPE_MULTIPLIER: {
                    response.withBidModifierBannerTypeWeb(
                            BannerTypeModifierConverter.convertToWeb((BidModifierBannerType) bidModifier));
                    break;
                }
                case INVENTORY_MULTIPLIER: {
                    response.withBidModifierInventory(
                            InventoryModifierConverter.convertToWeb((BidModifierInventory) bidModifier));
                    break;
                }
                case TRAFARET_POSITION_MULTIPLIER: {
                    response.withBidModifierTrafaretPosition(
                            TrafaretPositionModifierConverter.convertToWeb((BidModifierTrafaretPosition) bidModifier));
                    break;
                }
                default: {
                    throw new IllegalStateException("Unknown bid modifier type: " + bidModifier.getType());
                }
            }
        }

        return response;
    }

    private static BidModifierExpressionWeb convertExpressBidModifierToExternal(BidModifierExpression bidModifier) {
        return new BidModifierExpressionWeb()
                .withType(bidModifier.getType().toString().toLowerCase())
                .withHierarchicalMultiplierId(bidModifier.getId())
                .withEnabled(bidModifier.getEnabled() ? 1 : 0)
                .withAdjustments(mapList(
                        bidModifier.getExpressionAdjustments(),
                        adj -> convertExpressAdjustmentToExternal(adj, bidModifier.getType())
                ));
    }

    private static BidModifierExpressionAdjustmentWeb
    convertExpressAdjustmentToExternal(BidModifierExpressionAdjustment adjustment, BidModifierType type) {
        return new BidModifierExpressionAdjustmentWeb()
                .withId(adjustment.getId())
                .withPercent(adjustment.getPercent())
                .withCondition(mapList(
                        adjustment.getCondition(),
                        list -> mapList(list, lit -> convertExpressLiteralToExternal(lit, type))
                ));
    }

    private static BidModifierExpressionLiteralWeb
    convertExpressLiteralToExternal(BidModifierExpressionLiteral literal, BidModifierType type) {
        var webLiteral = new BidModifierExpressionLiteralWeb()
                .withParameter(literal.getParameter())
                .withOperation(literal.getOperation());

        var param = literal.getParameter();
        var parameterInfo = Constants.ALL_PARAMETERS.get(type).get(param);
        checkNotNull(parameterInfo);
        if (parameterInfo.getType() == ParameterType.INTEGER) {
            webLiteral.withValue(literal.getValueInteger().toString());
        } else {
            webLiteral.withValue(literal.getValueString());
        }
        return webLiteral;
    }

    private static OsType osTypeFromExternal(String osType) {
        if (osType == null) {
            return null;
        }
        switch (osType) {
            case "ios":
                return OsType.IOS;
            case "android":
                return OsType.ANDROID;
            default:
                throw new IllegalStateException("Unknown osType: " + osType);
        }
    }

    private static TabletOsType tabletOsTypeFromExternal(String osType) {
        if (osType == null) {
            return null;
        }
        switch (osType) {
            case "ios":
                return TabletOsType.IOS;
            case "android":
                return TabletOsType.ANDROID;
            default:
                throw new IllegalStateException("Unknown osType: " + osType);
        }
    }

    private static String osTypeToExternal(OsType osType) {
        if (osType == null) {
            return null;
        }
        switch (osType) {
            case IOS:
                return "ios";
            case ANDROID:
                return "android";
            default:
                throw new IllegalStateException("Unknown osType: " + osType);
        }
    }

    private static String osTypeToExternal(TabletOsType osType) {
        if (osType == null) {
            return null;
        }
        switch (osType) {
            case IOS:
                return "ios";
            case ANDROID:
                return "android";
            default:
                throw new IllegalStateException("Unknown osType: " + osType);
        }
    }

    private static String ageToExternal(AgeType age) {
        if (age == null) {
            return null;
        }
        switch (age) {
            case _0_17:
                return "0-17";
            case _18_24:
                return "18-24";
            case _25_34:
                return "25-34";
            case _35_44:
                return "35-44";
            case _45_54:
                return "45-54";
            case _45_:
                return "45-";
            case _55_:
                return "55-";
            case UNKNOWN:
                return "unknown";
            default: {
                throw new IllegalStateException("Unknown age: " + age);
            }
        }
    }

    private static String parameterToExternal(WeatherType parameter) {
        switch (parameter) {
            case TEMP:
                return "temp";
            case PREC_STRENGTH:
                return "prec_strength";
            case CLOUDNESS:
                return "cloudness";
            default: {
                throw new IllegalStateException("Unknown parameter: " + parameter);
            }
        }
    }

    private static String operationToExternal(OperationType operation) {
        switch (operation) {
            case GE:
                return "ge";
            case LE:
                return "le";
            case EQ:
                return "eq";
            default: {
                throw new IllegalStateException("Unknown operation: " + operation);
            }
        }
    }

    private static List<List<BidModifierWeatherLiteralWeb>> expressionToExternal(
            List<List<BidModifierWeatherLiteral>> expressions) {
        if (expressions == null) {
            return null;
        }
        return mapList(expressions,
                listOfDisjunctions -> mapList(listOfDisjunctions,
                        ComplexBidModifierConverter::toBidModifierWeatherLiteralWeb));
    }

    private static BidModifierWeatherLiteralWeb toBidModifierWeatherLiteralWeb(BidModifierWeatherLiteral internal) {
        return new BidModifierWeatherLiteralWeb()
                .withParameter(parameterToExternal(internal.getParameter()))
                .withOperation(operationToExternal(internal.getOperation()))
                .withValue(internal.getValue());
    }

    private static String genderToExternal(GenderType gender) {
        if (gender == null) {
            return null;
        }
        switch (gender) {
            case MALE:
                return "male";
            case FEMALE:
                return "female";
            default: {
                throw new IllegalStateException("Unknown gender: " + gender);
            }
        }
    }


    private static GenderType genderFromExternal(String genderString) {
        if (genderString == null) {
            return null;
        }

        switch (genderString) {
            case "male":
                return GenderType.MALE;
            case "female":
                return GenderType.FEMALE;
            case "all":
                return null;
            default: {
                throw new IllegalStateException("Unknown gender: " + genderString);
            }
        }
    }

    private static WeatherType parameterFromExternal(String parameterString) {
        if (parameterString == null) {
            return null;
        }

        switch (parameterString) {
            case "temp":
                return WeatherType.TEMP;
            case "prec_strength":
                return WeatherType.PREC_STRENGTH;
            case "cloudness":
                return WeatherType.CLOUDNESS;
            default: {
                throw new IllegalStateException("Unknown parameter: " + parameterString);
            }
        }
    }

    private static OperationType operationFromExternal(String operationString) {
        if (operationString == null) {
            return null;
        }

        switch (operationString) {
            case "ge":
                return OperationType.GE;
            case "le":
                return OperationType.LE;
            case "eq":
                return OperationType.EQ;
            default: {
                throw new IllegalStateException("Unknown operation: " + operationString);
            }
        }
    }

    private static List<List<BidModifierWeatherLiteral>> expressionFromExternal(
            List<List<BidModifierWeatherLiteralWeb>> expressions) {
        if (expressions == null) {
            return null;
        }
        return mapList(expressions,
                listOfDisjunctions -> mapList(listOfDisjunctions,
                        ComplexBidModifierConverter::toBidModifierWeatherLiteral));
    }

    private static BidModifierWeatherLiteral toBidModifierWeatherLiteral(BidModifierWeatherLiteralWeb external) {
        return new BidModifierWeatherLiteral()
                .withParameter(parameterFromExternal(external.getParameter()))
                .withOperation(operationFromExternal(external.getOperation()))
                .withValue(external.getValue());
    }

    private static AgeType ageFromExternal(String ageString) {
        if (ageString == null) {
            return null;
        }

        switch (ageString) {
            case "0-17":
                return AgeType._0_17;
            case "18-24":
                return AgeType._18_24;
            case "25-34":
                return AgeType._25_34;
            case "35-44":
                return AgeType._35_44;
            case "45-54":
                return AgeType._45_54;
            case "45-":
                return AgeType._45_;
            case "55-":
                return AgeType._55_;
            case "unknown":
                return AgeType.UNKNOWN;
            case "all":
                return null;
            default: {
                throw new IllegalStateException("Unknown age: " + ageString);
            }
        }
    }
}
