package ru.yandex.direct.web.core.entity.inventori.service;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.brandsafety.translation.BrandSafetyTranslations;
import ru.yandex.direct.core.entity.contentsegment.translation.ContentCategoryTranslations;
import ru.yandex.direct.core.entity.contentsegment.translation.ContentGenreTranslations;
import ru.yandex.direct.core.entity.crypta.AudienceType;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalScope;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.service.common.GoalUtilsService;
import ru.yandex.direct.core.util.ReflectionTranslator;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.i18n.bundle.TranslationBundle;
import ru.yandex.direct.web.core.model.retargeting.CryptaGoalAvailableGroupType;
import ru.yandex.direct.web.core.model.retargeting.CryptaGoalWeb;
import ru.yandex.direct.web.core.model.retargeting.CryptaInterestTypeWeb;
import ru.yandex.direct.web.core.model.retargeting.WebGoalType;

import static java.lang.CharSequence.compare;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.crypta.utils.CryptaSegmentBrandSafetyUtils.getBrandSafetyAdditionalCategoriesForClient;
import static ru.yandex.direct.feature.FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_CHILDREN;
import static ru.yandex.direct.feature.FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_CHILDREN_BANNER;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.web.core.model.retargeting.CryptaGoalAvailableGroupType.CPC;
import static ru.yandex.direct.web.core.model.retargeting.CryptaGoalAvailableGroupType.CPM_BANNER;
import static ru.yandex.direct.web.core.model.retargeting.CryptaGoalAvailableGroupType.CPM_VIDEO;

@Service
public class CryptaService {
    private static final Logger logger = LoggerFactory.getLogger(CryptaService.class);

    /**
     * Переводы сегментам берутся из танкера крипты (используются только подтвержденные)
     * <a href="https://tanker-beta.yandex-team.ru/project/crypta/keyset/profile?branch=master">profile</a>.
     * <p>
     * Скачиваются в Директ они автоматически sandbox-таской, либо ручным запуском {@code bin/tanker.sh}
     */

    private static final String CINEMA_GENRES_NAME_KEY = "content_genre_cinema_genres_name";
    private static final String CONTENT_CATEGORY_REST_NAME_KEY = "content_genre_rest_name";

    private static final Map<Long, Map<FeatureName, List<CryptaGoalAvailableGroupType>>> CONTENT_CATEGORY_FEATURES_BY_GOAL = Map.of(
            4294968297L, Map.of(FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_BUSINESS, List.of(CPM_VIDEO, CPC),
                    FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_BUSINESS_BANNER, List.of(CPM_BANNER)),
            4294968317L, Map.of(FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_INTERNET_TELECOM, List.of(CPM_VIDEO, CPC),
                    FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_INTERNET_TELECOM_BANNER, List.of(CPM_BANNER)),
            4294968341L, Map.of(FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_BUSINESS_BANKING_FINANCE,
                    List.of(CPM_VIDEO, CPC),
                    FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_BUSINESS_BANKING_FINANCE_BANNER, List.of(CPM_BANNER)),
            4294968333L, Map.of(FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_JOB, List.of(CPM_VIDEO, CPC),
                    FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_JOB_BANNER, List.of(CPM_BANNER)),
            4294968313L, Map.of(FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_LAW, List.of(CPM_VIDEO, CPC),
                    FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_LAW_BANNER, List.of(CPM_BANNER)),
            4294968328L, Map.of(FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_SOCIETY, List.of(CPM_VIDEO, CPC),
                    FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_SOCIETY_BANNER, List.of(CPM_BANNER)),
            4294968332L, Map.of(FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_NEWS, List.of(CPM_VIDEO, CPC),
                    FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_NEWS_BANNER, List.of(CPM_BANNER)),
            4294968342L, Map.of(FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_HOBBY, List.of(CPM_VIDEO, CPC),
                    FeatureName.CONTENT_CATEGORY_TARGETING_SHOW_HOBBY_BANNER, List.of(CPM_BANNER)),
            4294968377L, Map.of(FeatureName.ALLOW_ADULT_CONTENT, List.of(CPM_VIDEO, CPM_BANNER))
    );

    private static final String CRYPTA_GOAL_TRANSLATIONS_BUNDLE_NAME = "ru.yandex.direct.core.entity.crypta" +
            ".CryptaGoalTranslations";
    private static final Set<GoalType> CAN_SELECT_ALL_TYPES = Set.of(
            GoalType.INTERESTS, GoalType.INTERNAL, GoalType.AUDIO_GENRES, GoalType.CONTENT_CATEGORY,
            GoalType.CONTENT_GENRE);
    private static final Map<GoalType, TranslationBundle> TRANSLATION_BUNDLE_BY_GOAL_TYPE = Map.of(
            GoalType.AUDIO_GENRES, AudioGenresTranslations.INSTANCE,
            GoalType.BRANDSAFETY, BrandSafetyTranslations.INSTANCE,
            GoalType.CONTENT_CATEGORY, ContentCategoryTranslations.INSTANCE,
            GoalType.CONTENT_GENRE, ContentGenreTranslations.INSTANCE);

    private final CryptaSegmentRepository cryptaSegmentRepository;
    private final TranslationService translationService;
    private final ReflectionTranslator reflectionTranslator;
    private final FeatureService featureService;

    @Autowired
    public CryptaService(
            CryptaSegmentRepository cryptaSegmentRepository,
            TranslationService translationService,
            ReflectionTranslator reflectionTranslator,
            FeatureService featureService
    ) {
        this.cryptaSegmentRepository = cryptaSegmentRepository;
        this.translationService = translationService;
        this.reflectionTranslator = reflectionTranslator;
        this.featureService = featureService;
    }

    /**
     * Получение сегментов крипты, доступные "обычным" клиентам. Не выдаёт сегменты доступные для внутренней рекламы
     * <p>
     * Отдаются все сегменты крипты, кроме Brand Safety, потому что brand safety - не являются сегментами крипты, а
     * являются разметкой контента.
     */
    public List<CryptaGoalWeb> getSegments() {
        return getSegments(CryptaGoalScope.COMMON);
    }

    /**
     * Получение сегментов крипты для фронта.
     * <p>
     * Из сегментов, загруженных с базы, фильтруются те категории контента "Остальное", которые в единственном числе
     * составляют группу сегментов.
     * К жанрам контента добавляется фиктивный "Жанры кино" - родительский для сегментов типа `content_genre`.
     * Применяется только на фронте, в базу не заносим, поэтому добавляем его вне репозитория.
     * Жанры кино назначается родителем для всех жанров контента.
     *
     * @param scope видимость сегментов, например CryptaGoalScope.INTERNAL_AD -- сегменты, видимые
     *              для внутренней рекламы, при этом они могут быть недоступны для "обычных" пользователей.
     */
    public List<CryptaGoalWeb> getSegments(CryptaGoalScope scope) {
        List<CryptaGoalWeb> segmentsFromDb = getSegmentsFromDb(scope);
        return addCinemaGenres(setOrderKey(filterSingleSegments(segmentsFromDb)));
    }

    /**
     * Получение сегментов крипты для фронта с фильтрацией по фичам про отображение категорий контента
     */
    public List<CryptaGoalWeb> getSegments(CryptaGoalScope scope, ClientId clientId) {
        var segments = getSegments(scope);

        fillAvailableGroups(segments, clientId);

        return segments;
    }

    /**
     * Получение сегментов крипты.
     * <p>
     * Отдаются все сегменты крипты, кроме Brand Safety, потому что brand safety - не являются сегментами крипты, а
     * являются разметкой контента, получение категорий Brand Safety реализовано в этой ручке:
     * {@link ClientConstantsGraphQlService#getAllowedBrandSafetyCategoriesByClient(GridGraphQLContext, GdClientConstants)}
     *
     * @param scope видимость сегментов, например CryptaGoalScope.INTERNAL_AD -- сегменты, видимые
     *              для внутренней рекламы, при этом они могут быть недоступны для "обычных" пользователей.
     */
    private List<CryptaGoalWeb> getSegmentsFromDb(CryptaGoalScope scope) {
        Map<Long, Set<Long>> mutuallyExclusiveGoals = GoalUtilsService.getMutuallyExclusiveGoals();
        return cryptaSegmentRepository.getWithoutBrandSafety(scope)
                .values()
                .stream()
                .map(goal -> convertToCryptaGoalWeb(mutuallyExclusiveGoals, goal))
                .collect(toList());
    }

    /**
     * Получение сегментов крипты для intapi.
     * <p>
     * Brand Safety будет работать только в новом интерфейсе, поэтому то же что и для
     * {@link CryptaService#getSegments()}
     */
    public List<CryptaGoalWeb> getSegmentsIntapi() {
        return addCinemaGenres(setOrderKey(cryptaSegmentRepository.getWithoutBrandSafety(CryptaGoalScope.COMMON).values().stream()
                .map(this::convertToBaseCryptaGoalWeb)
                .collect(toList())));
    }

    public List<CryptaGoalWeb> getBrandSafety() {
        return cryptaSegmentRepository.getBrandSafety()
                .values()
                .stream()
                .map(this::convertToBaseCryptaGoalWeb)
                .collect(toList());
    }

    public List<CryptaGoalWeb> getAdditionalBrandSafetyCategoriesForClient(ClientId clientId) {
        List<Long> categoryIds = getBrandSafetyAdditionalCategoriesForClient(
                featureService.getEnabledForClientId(clientId));

        return cryptaSegmentRepository.getByIds(categoryIds)
                .values()
                .stream()
                .map(this::convertToBaseCryptaGoalWeb)
                .collect(toList());
    }

    @Nullable
    public Boolean canSelectAll(Goal goal) {
        if (!(Objects.equals(goal.getParentId(), 0L) || Objects.isNull(goal.getParentId()))) {
            return null;
        }

        if (CAN_SELECT_ALL_TYPES.contains(goal.getType())) {
            return true;
        }

        try {
            return AudienceType.fromTypedValue(goal.getId()).isAllValuesAllowed();
        } catch (IllegalArgumentException e) {
            logger.error("Crypta segment group not added to AudienceType enum", e);
            return true;
        }
    }

    public Map<Long, Goal> getSegmentMap() {
        return cryptaSegmentRepository.getAll();
    }

    private void fillAvailableGroups(List<CryptaGoalWeb> segments, ClientId clientId) {
        var showChildrenForCpmVideo = featureService.isEnabledForClientId(clientId,
                CONTENT_CATEGORY_TARGETING_SHOW_CHILDREN);
        var showChildrenForCpmBanner = featureService.isEnabledForClientId(clientId,
                CONTENT_CATEGORY_TARGETING_SHOW_CHILDREN_BANNER);
        var defaultGroups = new HashSet<>(List.of(CPM_VIDEO, CPM_BANNER, CPC));
        var defaultGenreGroups = new HashSet<>(List.of(CPM_VIDEO, CPC));

        for (var segment : segments) {
            if (WebGoalType.content_genre.equals(segment.getType())) {
                segment.withAvailableGroups(defaultGenreGroups);
                continue;
            }

            if (!WebGoalType.content_category.equals(segment.getType())) {
                continue;
            }

            var availableGroups = new HashSet<CryptaGoalAvailableGroupType>();

            Map<FeatureName, List<CryptaGoalAvailableGroupType>> featuresMap = null;
            if (CONTENT_CATEGORY_FEATURES_BY_GOAL.containsKey(segment.getId())) {
                featuresMap = CONTENT_CATEGORY_FEATURES_BY_GOAL.get(segment.getId());
            }

            if (CONTENT_CATEGORY_FEATURES_BY_GOAL.containsKey(segment.getParentId())) {
                featuresMap = CONTENT_CATEGORY_FEATURES_BY_GOAL.get(segment.getParentId());
            }

            if (featuresMap != null) {
                for (var feature : featuresMap.entrySet()) {
                    if (featureService.isEnabledForClientId(clientId, feature.getKey())) {
                        availableGroups.addAll(feature.getValue());
                    }
                }
            } else {
                availableGroups = new HashSet<>(defaultGroups);
            }

            if (segment.getParentId() != null && segment.getParentId() != 0) {
                if (!showChildrenForCpmVideo) {
                    availableGroups.remove(CPM_VIDEO);
                    availableGroups.remove(CPC);
                }

                if (!showChildrenForCpmBanner) {
                    availableGroups.remove(CPM_BANNER);
                }
            }

            segment.withAvailableGroups(availableGroups);
        }
    }

    private String translateGoal(@Nullable String key, GoalType goalType) {
        if (TRANSLATION_BUNDLE_BY_GOAL_TYPE.keySet().contains(goalType)) {
            return reflectionTranslator.translate(key, TRANSLATION_BUNDLE_BY_GOAL_TYPE.get(goalType));
        }

        return translationService.translate(CRYPTA_GOAL_TRANSLATIONS_BUNDLE_NAME, key);
    }

    /**
     * Для внутренней рекламы, если нет перевода в COMMON, возвращается передаваемое значение
     *
     * @param key              Ключ для перевода
     * @param goal             Цель, из которой берётся ключ
     * @param defaultTranslate Перевод по-умолчанию
     * @return перевод (если он есть) / перевод по-умолчанию
     */
    private String goalTranslateWithDefault(@Nullable String key, Goal goal,
                                            String defaultTranslate) {
        if (!goal.getCryptaScope().contains(CryptaGoalScope.COMMON)) {
            return defaultTranslate;
        }

        return translateGoal(key, goal.getType());
    }

    private CryptaGoalWeb convertToBaseCryptaGoalWeb(Goal goal) {
        return new CryptaGoalWeb()
                .withId(goal.getId())
                .withParentId(goal.getParentId())
                .withName(goalTranslateWithDefault(goal.getTankerNameKey(), goal,
                        goal.getName()))
                .withType(WebGoalType.fromCoreType(goal.getType()))
                .withDescription(goalTranslateWithDefault(goal.getTankerDescriptionKey(), goal,
                        goal.getTankerDescriptionKey()))
                .withInterestType(CryptaInterestTypeWeb.fromCoreType(goal.getInterestType()))
                .withTankerNameKey(goal.getTankerNameKey())
                .withKeywordValue(goal.getKeywordValue());
    }

    private CryptaGoalWeb convertToCryptaGoalWeb(Map<Long, Set<Long>> mutuallyExclusiveGoals, Goal goal) {
        return convertToBaseCryptaGoalWeb(goal)
                .withKeyword(goal.getKeyword())
                .withKeywordValue(goal.getKeywordValue())
                .withMutuallyExclusiveIds(mutuallyExclusiveGoals.getOrDefault(goal.getId(), emptySet()))
                .withCanSelectAll(canSelectAll(goal));
    }

    private List<CryptaGoalWeb> filterSingleSegments(List<CryptaGoalWeb> segmentsFromDb) {
        Set<Long> segmentIdsForIgnore = findSingleSegmentIds(segmentsFromDb);
        return filterList(segmentsFromDb, goal -> !segmentIdsForIgnore.contains(goal.getId()));
    }

    private List<CryptaGoalWeb> addCinemaGenres(List<CryptaGoalWeb> segmentsFromDb) {
        List<CryptaGoalWeb> segments = mapList(segmentsFromDb,
                CryptaService::setContentGenresParentIdToCinemaGenreGoalId);
        segments.add(createCinemaGenresSegment());
        return segments;
    }

    private CryptaGoalWeb createCinemaGenresSegment() {
        return new CryptaGoalWeb()
                .withId(Goal.CINEMA_GENRES_GOAL_ID)
                .withParentId(0L)
                .withType(WebGoalType.content_genre)
                .withInterestType(CryptaInterestTypeWeb.all)
                .withTankerNameKey(CINEMA_GENRES_NAME_KEY)
                .withName(translateGoal(CINEMA_GENRES_NAME_KEY, GoalType.CONTENT_GENRE))
                .withDescription(translateGoal("content_genre_cinema_genres_description", GoalType.CONTENT_GENRE))
                .withKeyword("content_category")
                .withKeywordValue(Goal.CINEMA_GENRES_GOAL_ID.toString())
                .withMutuallyExclusiveIds(emptySet())
                .withCanSelectAll(true)
                // флаг canNotCheck - true только для фиктивного "Жанры кино"
                .withCanNotCheck(true)
                // жанры должны быть всегда в конце DIRECT-133024
                .withOrder((long) Integer.MAX_VALUE);
    }

    private static CryptaGoalWeb setContentGenresParentIdToCinemaGenreGoalId(CryptaGoalWeb goal) {
        if (WebGoalType.content_genre.equals(goal.getType()) && !goal.getId().equals(Goal.CINEMA_GENRES_GOAL_ID)) {
            goal.setParentId(Goal.CINEMA_GENRES_GOAL_ID);
        }
        return goal;
    }

    private static Set<Long> findSingleSegmentIds(List<CryptaGoalWeb> segments) {
        return segments
                .stream()
                .filter(goal -> WebGoalType.content_category.equals(goal.getType()) &&
                        !Objects.isNull(goal.getParentId()))
                .collect(groupingBy(CryptaGoalWeb::getParentId))
                .values()
                .stream()
                .filter(goals -> goals.size() == 1)
                .map(goals -> goals.get(0).getId())
                .collect(toSet());
    }

    private List<CryptaGoalWeb> setOrderKey(List<CryptaGoalWeb> cryptaGoals) {
        var orderedGoals = cryptaGoals.stream()
                .map(g -> new CryptaGoalWeb()
                        .withId(g.getId())
                        .withTankerNameKey(g.getTankerNameKey())
                        .withName(g.getName()))
                .sorted((g1, g2) -> {
                    if (g1.getName() == null) {
                        return g2.getName() == null ? 0 : -1;
                    }

                    if (g2.getName() == null) {
                        return g1.getName() == null ? 0 : 1;
                    }

                    // "Остальное" должно быть всегда в конце
                    if (CONTENT_CATEGORY_REST_NAME_KEY.equals(g1.getTankerNameKey())) {
                        return 1;
                    }

                    if (CONTENT_CATEGORY_REST_NAME_KEY.equals(g2.getTankerNameKey())) {
                        return -1;
                    }

                    return compare(g1.getName(), g2.getName());
                })
                .collect(toList());

        long order = 0;
        for (var goal : orderedGoals) {
            goal.setOrder(order++);
        }

        var orderMap = listToMap(orderedGoals, CryptaGoalWeb::getId, CryptaGoalWeb::getOrder);
        cryptaGoals.forEach(g -> g.setOrder(orderMap.get(g.getId())));

        return cryptaGoals;
    }
}
