package ru.yandex.direct.web.core.entity.inventori.service;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.TargetTagEnum;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.currency.service.CurrencyRateService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.inventori.service.CampaignInfoCollector;
import ru.yandex.direct.core.entity.inventori.service.InventoriServiceCore;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.inventori.model.request.CampaignParameters;
import ru.yandex.direct.inventori.model.request.CampaignParametersCorrections;
import ru.yandex.direct.inventori.model.request.CampaignParametersRf;
import ru.yandex.direct.inventori.model.request.CampaignParametersSchedule;
import ru.yandex.direct.inventori.model.request.CampaignPredictionRequest;
import ru.yandex.direct.inventori.model.request.GroupType;
import ru.yandex.direct.inventori.model.request.StrategyType;
import ru.yandex.direct.inventori.model.request.Target;
import ru.yandex.direct.inventori.model.request.TrafficTypeCorrections;
import ru.yandex.direct.inventori.model.response.GeneralCampaignPredictionResponse;
import ru.yandex.direct.inventori.model.response.GeneralRecommendationResponse;
import ru.yandex.direct.inventori.model.response.error.PredictionResponseError;
import ru.yandex.direct.inventori.model.response.error.PredictionResponseInternalError;
import ru.yandex.direct.inventori.model.response.error.PredictionResponseInvalidBudgetError;
import ru.yandex.direct.inventori.model.response.error.PredictionResponseInvalidCpmError;
import ru.yandex.direct.inventori.model.response.error.PredictionResponseInvalidDatesError;
import ru.yandex.direct.inventori.model.response.error.PredictionResponseInvalidRequestError;
import ru.yandex.direct.inventori.model.response.error.PredictionResponseInvalidRfError;
import ru.yandex.direct.inventori.model.response.error.PredictionResponseNoGroupsError;
import ru.yandex.direct.inventori.model.response.error.PredictionResponseUnknownSegmentsError;
import ru.yandex.direct.inventori.model.response.error.PredictionResponseUnsupportedError;
import ru.yandex.direct.inventori.model.response.error.PredictionResponseUnsupportedSegmentsError;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.defect.params.NumberDefectParams;
import ru.yandex.direct.validation.defect.params.StringDefectParams;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.web.core.entity.inventori.model.AdGroupGeo;
import ru.yandex.direct.web.core.entity.inventori.model.CampaignStrategy;
import ru.yandex.direct.web.core.entity.inventori.model.CpmCampaignType;
import ru.yandex.direct.web.core.entity.inventori.model.CpmForecastRequest;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCampaignPredictionSuccessResult;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCpmRecommendationSuccessResult;
import ru.yandex.direct.web.core.entity.inventori.model.TrafficTypeCorrectionsWeb;
import ru.yandex.direct.web.core.entity.inventori.validation.InventoriDefectIds;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.split;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPV;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_MAX_IMPRESSIONS;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_MAX_REACH;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.PERIOD_FIX_BID;
import static ru.yandex.direct.core.entity.crypta.utils.CryptaSegmentBrandSafetyUtils.makeBrandSafetyCategories;
import static ru.yandex.direct.core.entity.inventori.service.InventoriServiceCore.getInventoriCampaignType;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.web.core.entity.inventori.validation.InventoriDefectIds.Number.LOW_REACH;

/**
 * Сервис для получения запросов в InventORI (CPM-прогнозатор).
 */
@ParametersAreNonnullByDefault
@Service
public class InventoriService {

    public static final Set<StrategyName> VALID_STRATEGIES = ImmutableSet.of(
            AUTOBUDGET_MAX_REACH,
            AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD,
            AUTOBUDGET_MAX_IMPRESSIONS,
            AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD,
            AUTOBUDGET_AVG_CPV,
            AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD,
            PERIOD_FIX_BID);

    public static final Set<CampaignType> VALID_CAMPAIGN_TYPES = ImmutableSet.of(
            CampaignType.CPM_BANNER,
            CampaignType.CPM_YNDX_FRONTPAGE,
            CampaignType.CPM_PRICE);

    public static final Set<AdGroupType> INVALID_ADGROUPS_TYPES = ImmutableSet.of(
            AdGroupType.CPM_GEOPRODUCT,
            AdGroupType.CPM_OUTDOOR,
            AdGroupType.CPM_INDOOR,
            AdGroupType.CPM_GEO_PIN);

    private final CampaignInfoCollector infoCollector;
    private final ShardHelper shardHelper;
    private final InventoriServiceCore inventoriServiceCore;
    private final PricePackageService pricePackageService;
    private final CryptaSegmentRepository cryptaSegmentRepository;
    private final FeatureService featureService;
    private final CurrencyRateService currencyRateService;
    private final CampaignRepository campaignRepository;

    @Autowired
    public InventoriService(
            CurrencyRateService currencyRateService,
            CampaignInfoCollector infoCollector,
            ShardHelper shardHelper,
            InventoriServiceCore inventoriServiceCore,
            PricePackageService pricePackageService,
            CryptaSegmentRepository cryptaSegmentRepository,
            CampaignRepository campaignRepository,
            FeatureService featureService) {
        this.currencyRateService = currencyRateService;
        this.infoCollector = infoCollector;
        this.shardHelper = shardHelper;
        this.inventoriServiceCore = inventoriServiceCore;
        this.pricePackageService = pricePackageService;
        this.cryptaSegmentRepository = cryptaSegmentRepository;
        this.campaignRepository = campaignRepository;
        this.featureService = featureService;
    }

    /**
     * Получить параметр rfReset из стратегии. Если rf = 0, то возвращаем 0. Сделано из-за того, что при отключении
     * ограничения показов контроллер saveCamp сохраняет 0 только в rf, а rfReset остается прежним.
     *
     * @param campaignStrategy стратегия кампании
     * @return значение rfReset
     */
    private int getRfReset(CampaignStrategy campaignStrategy) {
        //noinspection ConstantConditions
        if (campaignStrategy.getImpressionLimit().getImpressions().intValue() == 0) {
            return 0;
        }

        return campaignStrategy.getImpressionLimit().getDays().intValue();
    }

    @NotNull
    public CampaignParametersRf getRf(CampaignStrategy strategy) {
        return new CampaignParametersRf(strategy.getImpressionLimit().getImpressions().intValue(),
                getRfReset(strategy));
    }

    /**
     * Конвертирует запрос в CPM-прогнозатор.
     *
     * @param operatorUid uid оператора
     * @param clientId    id клиента
     * @param request     запрос
     * @param currency    валюта клиента/кампании
     * @return объект запроса в CPM-прогнозатор
     */
    public CampaignPredictionRequest convertForecastRequest(Long operatorUid, ClientId clientId,
                                                            CpmForecastRequest request, CurrencyCode currency) {
        CampaignPredictionRequest.Builder builder = CampaignPredictionRequest.builder();

        int shard = shardHelper.getShardByClientId(clientId);
        CampaignStrategy strategy = request.getStrategy();
        PricePackage pricePackage = null;

        if (request.getNewCampaignExampleType() != null) {
            builder.withDefaultTargetId(request.getNewCampaignExampleType())
                    .withTargets(campaignLevelTarget(request));
        } else if (request.getCampaignId() != null) {
            Map<Long, Set<Integer>> adGroupGeoByAdGroupId = StreamEx.of(nvl(request.getAdGroupGeos(), emptyList()))
                    .toMap(AdGroupGeo::getAdGroupId, adGroupGeo -> convertGeo(adGroupGeo.getGeo()));

            TrafficTypeCorrections corrections = getTrafficTypeCorrections(shard, request);
            Map<Long, TrafficTypeCorrections> correctionsMap = corrections != null ?
                    Map.of(request.getCampaignId(), corrections)
                    : null;

            Pair<List<Target>, Boolean> info =
                    infoCollector.collectCampaignsInfoWithClientIdAndUid(null, false, shard, operatorUid, clientId,
                            singletonList(request.getCampaignId()), true, correctionsMap).get(request.getCampaignId());

            pricePackage = pricePackageService
                    .getPricePackageByCampaignIds(shard, List.of(request.getCampaignId()))
                    .get(request.getCampaignId());

            var brandLifts = campaignRepository.getBrandSurveyIdsForCampaigns(shard, List.of(request.getCampaignId()));

            builder.withBrandlift(brandLifts.get(request.getCampaignId()) != null);
            correctTargetsGeo(info.getLeft(), adGroupGeoByAdGroupId);
            correctTargetsTags(info.getLeft(), request.getTargetTags());

            builder.withCampaignId(request.getCampaignId())
                    .withTargets(StreamEx.of(info.getLeft()).filter(Target::hasCreatives).toList())
                    .containsKeywordGroup(info.getRight());
        }

        //noinspection ConstantConditions
        StrategyType strategyType = StrategyType.parse(strategy.getType());
        CampaignParameters.Builder parametersBuilder = CampaignParameters.builder()
                .withSchedule(getSchedule(currency, strategy, pricePackage, strategyType, request.getCampaignId()))
                .withRf(getRf(strategy));

        TrafficTypeCorrections trafficTypeCorrections = getTrafficTypeCorrections(shard, request);
        if (trafficTypeCorrections != null) {
            parametersBuilder.withCorrections(new CampaignParametersCorrections(trafficTypeCorrections));
        }

        CampaignType campaignType = CampaignType.CPM_BANNER;
        if (request.getCampaignId() != null) {
            campaignType = campaignRepository.getCampaignsTypeMap(shard, List.of(request.getCampaignId()))
                    .getOrDefault(request.getCampaignId(), CampaignType.CPM_BANNER);
        } else if (request.getCpmCampaignType() != null) {
            campaignType = convertCpmCampaignTypeToCampaignType(request.getCpmCampaignType());
        }
        builder.withCampaignType(getInventoriCampaignType(campaignType, null));

        builder.withParameters(parametersBuilder.build());
        builder.withIsSocial(featureService.isEnabledForClientId(clientId, FeatureName.SOCIAL_ADVERTISING_BY_LAW));

        return builder.build();
    }

    CampaignType convertCpmCampaignTypeToCampaignType(CpmCampaignType cpmCampaignType) {
        switch (cpmCampaignType) {
            case CPM_YNDX_FRONTPAGE:
                return CampaignType.CPM_YNDX_FRONTPAGE;
            case CPM_BANNER:
                return CampaignType.CPM_BANNER;
            case CPM_DEALS:
                return CampaignType.CPM_DEALS;
            default:
                return CampaignType.CPM_BANNER;
        }
    }
    @NotNull
    public CampaignParametersSchedule getSchedule(CurrencyCode currency, CampaignStrategy strategy,
                                                  PricePackage pricePackage, StrategyType strategyType, Long campaignId) {
        return CampaignParametersSchedule.builder()
                .withStrategyType(strategyType)
                .withBudget(inventoriServiceCore.convertFromClientsCurrency(strategy.getBudget(), currency))
                .withStartDate(strategy.getStartDate())
                .withEndDate(strategy.getEndDate())
                .withCpm(inventoriServiceCore.calcAvgCpm(strategyType, strategy.getCpm(), currency, pricePackage, campaignId))
                .withCpv(inventoriServiceCore.calcAvgCpv(ifNotNull(strategy.getCpv(), BigDecimal::valueOf), currency))
                .withIsBooking(pricePackage == null ? null : pricePackage.isFrontpagePackage())
                .build();
    }

    /**
     * Заменяем тэги из базы на фронтовые, актуально для морды
     */
    private void correctTargetsTags(@Nullable List<Target> targets, @Nullable List<TargetTagEnum> targetTags) {
        if (targets == null) {
            return;
        }

        if (targetTags != null) {
            var frontendTags = mapList(targetTags, TargetTagEnum::getTypedValue);
            for (var target : targets) {
                target.withTargetTags(frontendTags);
            }
        }
    }

    /**
     * Заменяет старое гео таргетов на обновленное.
     *
     * @param targets               таргеты
     * @param adGroupGeoByAdGroupId мап id группы -> обновленное гео группы
     */
    private void correctTargetsGeo(@Nullable List<Target> targets, Map<Long, Set<Integer>> adGroupGeoByAdGroupId) {
        if (targets == null) {
            return;
        }

        for (Target target : targets) {
            Set<Integer> adGroupGeo = adGroupGeoByAdGroupId.get(target.getAdGroupId());

            if (adGroupGeo != null) {
                target.withRegions(adGroupGeo);
            }
        }
    }

    /**
     * Для вновь создаваемой кампании проставляет таргетинг, который задаётся на уровне РК.
     *
     * @param request запрос
     * @return таргеты. Null, если нет таргетов уровня РК
     */
    private List<Target> campaignLevelTarget(CpmForecastRequest request) {
        Target group = null;
        if (request.getBrandsafety() != null && nvl(request.getBrandsafety().getEnabled(), false)) {
            //таргетинг по брендсейфити
            group = new Target().withExcludedBsCategories(
                    makeExcludedBsCategories(request.getBrandsafety().getAdditionalCategories()));
        }
        if (request.getPageBlocks() != null) {
            group = nvl(group, new Target());
            //таргетинг по белому списку PageID
            group.withPageBlocks(request.getPageBlocks());
        }
        if (group != null) {
            group.withExcludedPageBlocks(emptyList());

            return List.of(group.withGroupType(GroupType.VIDEO));
        }
        return null;
    }

    private List<String> makeExcludedBsCategories(List<Long> additionalCategories) {
        var brandSafetyCategories = makeBrandSafetyCategories(true, nvl(additionalCategories, emptyList()));
        var cryptaGoals = cryptaSegmentRepository.getBrandSafety();
        return StreamEx.of(brandSafetyCategories)
                .filter(id -> cryptaGoals.containsKey(id))
                .map(id -> cryptaGoals.get(id))
                .map(goal -> goal.getKeyword() + ":" + goal.getKeywordValue())
                .toList();
    }

    /**
     * Конвертирует запрос в CPM-прогнозатор.
     *
     * @param operatorUid uid оператора
     * @param clientId    id клиента
     * @param request     запрос
     * @param currency    валюта клиента/кампании
     * @return объект запроса в CPM-прогнозатор
     */
    public CampaignPredictionRequest convertTrafficLightPredictionRequest(
            Long operatorUid,
            ClientId clientId,
            CpmForecastRequest request,
            CurrencyCode currency) {
        CampaignPredictionRequest inventoriRequest;

        if (request.getCampaignId() != null) {
            var campaignIds = singletonList(request.getCampaignId());
            ListValidationBuilder<Long, Defect> listValidationBuilder =
                    ListValidationBuilder.of(campaignIds, Defect.class);
            inventoriRequest = inventoriServiceCore.getCampaignPredictionRequestForCampaigns(
                    false,
                    operatorUid,
                    clientId,
                    campaignIds,
                    true,
                    request.getStrategy() == null,
                    false,
                    false,
                    listValidationBuilder.getResult())
                    .get(0);

            Map<Long, Set<Integer>> adGroupGeoByAdGroupId = StreamEx.of(nvl(request.getAdGroupGeos(), emptyList()))
                    .toMap(AdGroupGeo::getAdGroupId, adGroupGeo -> convertGeo(adGroupGeo.getGeo()));

            correctTargetsGeo(inventoriRequest.getTargets(), adGroupGeoByAdGroupId);
        } else {
            inventoriRequest = new CampaignPredictionRequest();
            inventoriRequest.setDefaultTargetId(request.getNewCampaignExampleType());
            inventoriRequest.setTargets(campaignLevelTarget(request));
        }

        int shard = shardHelper.getShardByClientId(clientId);

        if (request.getStrategy() != null) {
            inventoriRequest.setParameters(convertParameters(shard, request, currency));
        }

        correctTargetsTags(inventoriRequest.getTargets(), request.getTargetTags());

        // traffic corrections
        addTrafficCorrectionsToRequest(inventoriRequest, getTrafficTypeCorrections(shard, request));

        return inventoriRequest;
    }

    private static void addTrafficCorrectionsToRequest(CampaignPredictionRequest inventoriRequest,
                                                       @Nullable TrafficTypeCorrections trafficTypeCorrections) {
        if (trafficTypeCorrections == null) {
            return;
        }
        inventoriRequest.getParameters().setCorrections(new CampaignParametersCorrections(trafficTypeCorrections));
    }

    private TrafficTypeCorrections getTrafficTypeCorrections(int shard, CpmForecastRequest request) {
        TrafficTypeCorrectionsWeb correctionsWeb = request.getTrafficTypeCorrections();
        if (correctionsWeb == null) {
            if (request.getCampaignId() != null) {
                return infoCollector.collectCampaignsCorrections(shard, singletonList(request.getCampaignId()))
                        .get(request.getCampaignId());
            }
            return null;
        }
        return new TrafficTypeCorrections(correctionsWeb.getBanner(), correctionsWeb.getVideoInpage(),
                correctionsWeb.getVideoInstream(), correctionsWeb.getVideoInterstitial(),
                correctionsWeb.getVideoInbanner(), correctionsWeb.getVideoRewarded());
    }

    /**
     * Конвертирует данные о стратегии для запроса в CPM-прогнозатор.
     *
     * @param shard    номер шарда
     * @param request  данные о запросе
     * @param currency валюта клиента/кампании
     * @return объект CampaignParameters
     */
    public CampaignParameters convertParameters(int shard, CpmForecastRequest request, CurrencyCode currency) {
        CampaignStrategy strategy = request.getStrategy();
        PricePackage pricePackage = null;
        if (request.getCampaignId() != null) {
            pricePackage = pricePackageService
                    .getPricePackageByCampaignIds(shard, List.of(request.getCampaignId()))
                    .get(request.getCampaignId());
        }

        //noinspection ConstantConditions
        StrategyType strategyType = StrategyType.parse(strategy.getType());
        return new CampaignParameters(
                CampaignParametersSchedule.builder()
                        .withStrategyType(strategyType)
                        .withBudget(inventoriServiceCore.convertFromClientsCurrency(strategy.getBudget(), currency))
                        .withStartDate(strategy.getStartDate())
                        .withEndDate(strategy.getEndDate())
                        .withCpm(inventoriServiceCore.calcAvgCpm(strategyType, strategy.getCpm(), currency,
                                pricePackage, request.getCampaignId()))
                        .withCpv(inventoriServiceCore.calcAvgCpv(ifNotNull(strategy.getCpv(), BigDecimal::valueOf),
                                currency))
                        .withAutoProlongation(strategy.getAutoProlongation() == null ? null :
                                strategy.getAutoProlongation() == 1)
                        .withIsBooking(pricePackage == null ? null : pricePackage.isFrontpagePackage())
                        .build(),
                getRf(strategy));
    }

    private Set<Integer> convertGeo(String geo) {
        return isEmpty(geo) ? null : StreamEx.of(split(geo, ","))
                .map(String::trim)
                .map(Integer::valueOf)
                .toSet();
    }

    public Defect convertError(PredictionResponseError error) {
        if (error instanceof PredictionResponseInternalError) {
            return new Defect<>(InventoriDefectIds.Gen.INTERNAL_ERROR);
        }

        if (error instanceof PredictionResponseInvalidBudgetError) {
            PredictionResponseInvalidBudgetError typedError = (PredictionResponseInvalidBudgetError) error;
            return new Defect<>(InventoriDefectIds.Number.INVALID_BUDGET, new NumberDefectParams()
                    .withMax(typedError.getInvalidBudget()));
        }

        if (error instanceof PredictionResponseInvalidCpmError) {
            PredictionResponseInvalidCpmError typedError = (PredictionResponseInvalidCpmError) error;
            return new Defect<>(InventoriDefectIds.Number.INVALID_CPM, new NumberDefectParams()
                    .withMax(typedError.getInvalidCpm()));
        }

        if (error instanceof PredictionResponseInvalidDatesError) {
            PredictionResponseInvalidDatesError typedError = (PredictionResponseInvalidDatesError) error;
            return new Defect<>(InventoriDefectIds.String.INVALID_DATES, new StringDefectParams()
                    .withInvalidSubstrings(singletonList(typedError.getInvalidDates())));
        }

        if (error instanceof PredictionResponseInvalidRequestError) {
            PredictionResponseInvalidRequestError typedError = (PredictionResponseInvalidRequestError) error;
            return new Defect<>(InventoriDefectIds.String.INVALID_REQUEST, new StringDefectParams()
                    .withInvalidSubstrings(singletonList(typedError.getMessage())));
        }

        if (error instanceof PredictionResponseInvalidRfError) {
            PredictionResponseInvalidRfError typedError = (PredictionResponseInvalidRfError) error;
            return new Defect<>(InventoriDefectIds.String.INVALID_RF, new StringDefectParams()
                    .withInvalidSubstrings(singletonList(typedError.getInvalidRf())));
        }

        if (error instanceof PredictionResponseUnknownSegmentsError) {
            PredictionResponseUnknownSegmentsError typedError = (PredictionResponseUnknownSegmentsError) error;
            return new Defect<>(InventoriDefectIds.String.UNKNOWN_SEGMENTS, new StringDefectParams()
                    .withInvalidSubstrings(typedError.getSegmentIds()));
        }

        if (error instanceof PredictionResponseUnsupportedSegmentsError) {
            PredictionResponseUnsupportedSegmentsError typedError =
                    (PredictionResponseUnsupportedSegmentsError) error;
            return new Defect<>(InventoriDefectIds.String.UNSUPPORTED_SEGMENTS, new StringDefectParams()
                    .withInvalidSubstrings(typedError.getSegmentIds()));
        }

        if (error instanceof PredictionResponseNoGroupsError) {
            return new Defect<>(InventoriDefectIds.Gen.NO_GROUPS);
        }

        if (error instanceof PredictionResponseUnsupportedError) {
            PredictionResponseUnsupportedError typedError =
                    (PredictionResponseUnsupportedError) error;
            return new Defect<>(InventoriDefectIds.Gen.UNSUPPORTED_ERROR, typedError.getProperties());
        }

        throw new UnsupportedOperationException("Unknown defect type.");
    }

    public static Defect lowReach(long reach) {
        return new Defect<>(LOW_REACH, new NumberDefectParams().withMin(reach));
    }

    public double convertToClientsCurrency(long microrub, CurrencyCode target) {
        return currencyRateService.convertMoney(Money.valueOfMicros(microrub, CurrencyCode.RUB), target)
                .bigDecimalValue().doubleValue();
    }

    public GeneralCpmRecommendationSuccessResult convertResponseToSuccessResult(
            GeneralRecommendationResponse response,
            CurrencyCode currency, StrategyType strategyType) {

        Long recommended;

        if (strategyType == StrategyType.MAX_AVG_CPV_CUSTOM_PERIOD || strategyType == StrategyType.MAX_AVG_CPV) {
            recommended = response.getRecommendedMaxAvgCpv();
        } else {
            recommended = response.getRecommendedMaxAvgCpm();
        }

        if (recommended == null) {
            recommended = nvl(response.getRecommendedMaxAvgCpm(), response.getRecommendedMaxAvgCpv());
        }

        Long targetEventsLeftBorder = nvl(response.getPredictedImpressionsLeftBorder(),
                response.getPredictedTrueViewsLeftBorder());
        Long targetEventsRightBorder = nvl(response.getPredictedImpressionsRightBorder(),
                response.getPredictedTrueViewsRightBorder());
        Long reachLeftBorder = response.getPredictedReachLeftBorder();
        Long reachRightBorder = response.getPredictedReachRightBorder();

        checkNotNull(recommended);
        checkNotNull(targetEventsLeftBorder);
        checkNotNull(targetEventsRightBorder);
        checkNotNull(reachLeftBorder);
        checkNotNull(reachRightBorder);

        return new GeneralCpmRecommendationSuccessResult(
                response.getTrafficLightColourId(),
                convertToClientsCurrency(recommended, currency),
                targetEventsLeftBorder,
                targetEventsRightBorder,
                reachLeftBorder,
                reachRightBorder
        );
    }

    public GeneralCampaignPredictionSuccessResult convertGeneralCampaignPredictionResponseToSuccessResult(
            GeneralCampaignPredictionResponse response,
            CurrencyCode currency, StrategyType strategyType) {

        Long recommendedPrice;

        if (strategyType == StrategyType.MAX_AVG_CPV_CUSTOM_PERIOD || strategyType == StrategyType.MAX_AVG_CPV) {
            recommendedPrice = response.getRecommendedMaxAvgCpv();
        } else {
            recommendedPrice = response.getRecommendedMaxAvgCpm();
        }

        if (recommendedPrice == null) {
            recommendedPrice = nvl(response.getRecommendedMaxAvgCpm(), nvl(response.getRecommendedMaxAvgCpv(), 0L));
        }

        Long targetReach = response.getTargetReach();
        Long forecastReach = response.getForecastReach();
        Long forecastShows = nvl(response.getForecastImpressions(), nvl(response.getForecastTrueViews(), 0L));
        Double sovByShows = nvl(response.getSovByImpressions(), nvl(response.getSovByTrueViews(), 0D));
        Double sovByReach = response.getSovByReach();
        Long forecastBudget = response.getForecastBudget();

        checkNotNull(recommendedPrice);
        checkNotNull(targetReach);
        checkNotNull(forecastReach);
        checkNotNull(forecastBudget);
        checkNotNull(sovByReach);

        return new GeneralCampaignPredictionSuccessResult(
                targetReach,
                forecastReach,
                forecastShows,
                sovByShows,
                sovByReach,
                convertToClientsCurrency(response.getForecastBudget(), currency),
                convertToClientsCurrency(recommendedPrice, currency),
                response.getTrafficLightColour()
        );
    }

}
