package ru.yandex.direct.web.core.entity.inventori.service;

import java.math.BigDecimal;
import java.math.MathContext;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import javax.annotation.Nullable;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.asynchttpclient.Param;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignWithBrandSafetyService;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.inventori.service.CampaignInfoCollector;
import ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter;
import ru.yandex.direct.core.entity.inventori.service.type.FrontendData;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.uac.model.DeviceType;
import ru.yandex.direct.core.entity.uac.model.InventoryType;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.inventori.InventoriClient;
import ru.yandex.direct.inventori.model.request.AudioCreative;
import ru.yandex.direct.inventori.model.request.BlockSize;
import ru.yandex.direct.inventori.model.request.CampaignParameters;
import ru.yandex.direct.inventori.model.request.CampaignParametersCorrections;
import ru.yandex.direct.inventori.model.request.CampaignParametersRf;
import ru.yandex.direct.inventori.model.request.CampaignParametersSchedule;
import ru.yandex.direct.inventori.model.request.CampaignPredictionRequest;
import ru.yandex.direct.inventori.model.request.GroupType;
import ru.yandex.direct.inventori.model.request.InventoriCampaignType;
import ru.yandex.direct.inventori.model.request.MobileOsType;
import ru.yandex.direct.inventori.model.request.PageBlock;
import ru.yandex.direct.inventori.model.request.PlatformCorrections;
import ru.yandex.direct.inventori.model.request.ProfileCorrection;
import ru.yandex.direct.inventori.model.request.StrategyType;
import ru.yandex.direct.inventori.model.request.Target;
import ru.yandex.direct.inventori.model.request.TrafficTypeCorrections;
import ru.yandex.direct.inventori.model.request.VideoCreative;
import ru.yandex.direct.inventori.model.response.CampaignPredictionLowReachResponse;
import ru.yandex.direct.inventori.model.response.CampaignPredictionResponse;
import ru.yandex.direct.inventori.model.response.ForecastResponse;
import ru.yandex.direct.inventori.model.response.GeneralCampaignPredictionResponse;
import ru.yandex.direct.inventori.model.response.GeneralForecastResponse;
import ru.yandex.direct.inventori.model.response.GeneralRecommendationResponse;
import ru.yandex.direct.inventori.model.response.IndoorPredictionResponse;
import ru.yandex.direct.inventori.model.response.MultiBudgetsPredictionResponse;
import ru.yandex.direct.inventori.model.response.OutdoorPredictionResponse;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.web.core.entity.inventori.model.BannerFormatIncreasePercent;
import ru.yandex.direct.web.core.entity.inventori.model.BidModifierDemographicWeb;
import ru.yandex.direct.web.core.entity.inventori.model.CampaignStrategy;
import ru.yandex.direct.web.core.entity.inventori.model.Condition;
import ru.yandex.direct.web.core.entity.inventori.model.Error;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCampaignPredictionResult;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCampaignPredictionSuccessResult;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCpmRecommendationRequest;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCpmRecommendationResult;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCpmRecommendationSuccessResult;
import ru.yandex.direct.web.core.entity.inventori.model.PageBlockWeb;
import ru.yandex.direct.web.core.entity.inventori.model.ReachIndoorRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ReachIndoorResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachInfo;
import ru.yandex.direct.web.core.entity.inventori.model.ReachMultiBudgetResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachOutdoorRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ReachOutdoorResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachRecommendationResult;
import ru.yandex.direct.web.core.entity.inventori.model.ReachRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ReachResult;
import ru.yandex.direct.web.core.model.retargeting.CryptaInterestTypeWeb;
import ru.yandex.direct.web.core.model.retargeting.RetargetingConditionRuleType;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections4.CollectionUtils.containsAny;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.inventori.service.InventoriServiceCore.getInventoriCampaignType;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.VIDEO_PROPORTION_16_9;
import static ru.yandex.direct.core.entity.inventori.service.type.AdGroupDataConverter.getAllowedBlockSizes;
import static ru.yandex.direct.inventori.InventoriClient.DEFAULT_REACH_LESS_THAN;
import static ru.yandex.direct.inventori.model.request.GroupType.INDOOR;
import static ru.yandex.direct.inventori.model.request.GroupType.OUTDOOR;
import static ru.yandex.direct.inventori.model.request.StrategyType.MIN_CPM;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.web.core.entity.inventori.service.InventoriService.lowReach;

@Service
public class InventoriWebService {

    public static final String UNDEFINED_KEYWORD = "0";
    public static final String BUDGETS_PARAM = "budgets";
    public static final Set<GroupType> INVALID_GROUP_TYPES = ImmutableSet.of(
            GroupType.INDOOR,
            GroupType.OUTDOOR,
            GroupType.GEOPRODUCT,
            GroupType.GEO_PIN,
            GroupType.BANNER_IN_GEO_APPS,
            GroupType.BANNER_IN_METRO);
    public static final GeneralCampaignPredictionSuccessResult DEFAULT_GENERAL_CAMPAIGN_PREDICTION_SUCCES_RESULT =
            new GeneralCampaignPredictionSuccessResult(0L, 0L,  0L, 0.0, 0.0, 0.0, 0.0, 0);

    public static final GeneralCpmRecommendationSuccessResult DEFAULT_GENERAL_CPM_RECOMMENDATION_RESULT =
            new GeneralCpmRecommendationSuccessResult(0, 0.0,  0L, 0L, 0L, 0L);

    public static final MultiBudgetsPredictionResponse DEFAULT_MULTI_BUDGETS_PREDICTION_RESPONCE =
            new MultiBudgetsPredictionResponse(emptyList(), 0L, emptyList());

    private final ShardHelper shardHelper;
    private final InventoriClient inventoriClient;
    private final CryptaService cryptaService;
    private final DirectWebAuthenticationSource authenticationSource;
    private final UserService userService;
    private final CampaignInfoCollector collector;
    private final InventoriService inventoriService;
    private final PricePackageService pricePackageService;
    private final CampaignRepository campaignRepository;
    private final FeatureService featureService;
    private final RetargetingConditionRepository retargetingConditionRepository;
    private final CampaignWithBrandSafetyService campaignWithBrandSafetyService;
    private final AdGroupRepository adGroupRepository;
    private final CryptaSegmentRepository cryptaSegmentRepository;

    @Autowired
    public InventoriWebService(
            ShardHelper shardHelper,
            InventoriClient inventoriClient,
            CryptaService cryptaService,
            DirectWebAuthenticationSource authenticationSource,
            UserService userService,
            CampaignInfoCollector collector,
            InventoriService inventoriService,
            CampaignRepository campaignRepository,
            PricePackageService pricePackageService,
            FeatureService featureService,
            RetargetingConditionRepository retargetingConditionRepository,
            CampaignWithBrandSafetyService campaignWithBrandSafetyService,
            AdGroupRepository adGroupRepository,
            CryptaSegmentRepository cryptaSegmentRepository) {
        this.shardHelper = shardHelper;
        this.inventoriClient = inventoriClient;
        this.cryptaService = cryptaService;
        this.authenticationSource = authenticationSource;
        this.userService = userService;
        this.collector = collector;
        this.inventoriService = inventoriService;
        this.campaignRepository = campaignRepository;
        this.pricePackageService = pricePackageService;
        this.featureService = featureService;
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.campaignWithBrandSafetyService = campaignWithBrandSafetyService;
        this.adGroupRepository = adGroupRepository;
        this.cryptaSegmentRepository = cryptaSegmentRepository;
    }

    @Nullable
    private static List<Error> getErrors(Map<Long, Goal> cryptaGoalMap, ForecastResponse detailedForecast) {
        List<Error> errors;
        Map<String, Long> keywordLongMap = cryptaGoalMap.values().stream()
                .filter(g -> !StringUtils.isEmpty(g.getKeyword()) && !UNDEFINED_KEYWORD.equals(g.getKeyword()))
                .collect(toMap(g -> g.getKeyword() + ":" + g.getKeywordValue(), Goal::getId));

        Map<String, Long> keywordShortMap = cryptaGoalMap.values().stream()
                .filter(g -> !StringUtils.isEmpty(g.getKeywordShort()) && !UNDEFINED_KEYWORD
                        .equals(g.getKeywordShort()))
                .collect(toMap(g -> g.getKeywordShort() + ":" + g.getKeywordValueShort(), Goal::getId));

        Map<String, Long> keywordMap = new HashMap<>();
        keywordMap.putAll(keywordLongMap);
        keywordMap.putAll(keywordShortMap);

        errors = Optional.ofNullable(detailedForecast.getErrors())
                .map(error -> error.stream()
                        .map(e -> e.getSegmentIds().stream()
                                .filter(Objects::nonNull)
                                // audience id comes as is, crypta segement id matches pattern 'keyword:keyword_value'
                                .map(id -> new Error(StringUtils.isNumeric(id) ? Long.valueOf(id) : keywordMap.get(id),
                                        e.getType()))
                                // collapse short and long keywords
                                .collect(toSet()))
                        .flatMap(Collection::stream)
                        .collect(toList()))
                .orElse(null);
        return errors;
    }

    /**
     * Собирает объект {@link Target} для запроса охвата в Inventori.
     *
     * @param request  запрос охвата
     * @param detailed является ли запрос охвата общим (basic) или детализированным (detailed)
     * @return {@link Target} для запроса охвата в Inventori
     */
    public Target createTarget(ReachRequest request, boolean detailed) {
        final User subClient = authenticationSource.getAuthentication().getSubjectUser();
        final ClientId clientId = subClient.getClientId();

        Long campaignId = request.getCampaignId();
        // у clientId больше шансов оказаться в кеше шардов
        int shard = shardHelper.getShardByClientId(clientId);

        // Данные с фронта
        FrontendData frontendData = convertToFrontendData(request);

        PlatformCorrections platformCorrections = fetchPlatformCorrections(shard, request);
        if (platformCorrections != null) {
            frontendData.setPlatformCorrections(platformCorrections);
        }
        if (campaignId == null) {
            campaignId = 0L;
        }
        Map<Long, Pair<List<Target>, Boolean>> targetsByCampaignIds = collector.collectCampaignsInfoWithClientIdAndUid(
                frontendData, false, shard, subClient.getUid(), clientId, singletonList(campaignId), true, null);

        List<Target> targets = targetsByCampaignIds.get(campaignId) != null ? targetsByCampaignIds.get(campaignId).getLeft() : emptyList();
        Target target = targets.isEmpty() ? null : targets.get(0);

        if (!detailed && target != null) {
            List<BlockSize> allBlockSizes = (AdGroupDataConverter.allowedAddBlockSizes(target)) ?
                    new ArrayList<>(getAllowedBlockSizes(target.getGroupType(), target.getMainPageTrafficType())) :
                    null;
            target = new Target()
                    .withAdGroupId(target.getAdGroupId())
                    .withGroupType(target.getGroupType())
                    .withBlockSizes(allBlockSizes)
                    .withDomains(target.getDomains())
                    .withExcludedDomains(target.getExcludedDomains())
                    .withSsp(target.getSsp())
                    .withExcludedSsp(target.getExcludedSsp())
                    .withRegions(target.getRegions())
                    .withCorrections(target.getCorrections())
                    .withPageBlocks(target.getPageBlocks())
                    .withExcludedPageBlocks(target.getExcludedPageBlocks())
                    .withTargetTags(target.getTargetTags())
                    .withPlatformCorrections(target.getPlatformCorrections())
                    .withMainPageTrafficType(target.getMainPageTrafficType())
                    .withAudienceGroups(target.getAudienceGroups())
                    .withCryptaGroups(target.getCryptaGroups())
                    .withOrderTags(target.getOrderTags())
                    .withExcludedBsCategories(request.getExcludedBsCategories());
        }

        return target;
    }

    private PlatformCorrections fetchPlatformCorrections(int shard, ReachRequest request) {

        PricePackage pricePackage = pricePackageService
                .getPricePackageByCampaignIds(shard, singleton(request.getCampaignId()))
                .get(request.getCampaignId());

        if (pricePackage == null) {
            // у кампании нет пакета
            return null;
        }
        List<BidModifier> bidModifiers = pricePackage.getBidModifiers();
        if (bidModifiers.isEmpty()) {
            // у пакета нет корректировок
            return null;
        }

        //На пакете выставляются галочки на то, где надо показывать, как следствие, где галочек нет - показывать не
        // надо.
        // То есть если выставлена галочка Desktop, то не надо показывать на всех мобильных, то есть выставить им
        // корректировку равную 0
        // корректировка, равная null - значет отсутствие корректировки, то есть показывать можно.
        PlatformCorrections.Builder builder = PlatformCorrections.builder()
                .withDesktop(0)
                .withMobile(0);
        bidModifiers.forEach(
                bidModifier -> {
                    switch (bidModifier.getType()) {
                        case DESKTOP_MULTIPLIER:
                            builder.withDesktop(null);
                            break;
                        case MOBILE_MULTIPLIER:
                            switch (((BidModifierMobile) bidModifier).getMobileAdjustment().getOsType()) {
                                case ANDROID:
                                    builder.withMobile(0);
                                    builder.withMobileOsType(MobileOsType.IOS);
                                    break;
                                case IOS:
                                    builder.withMobile(0);
                                    builder.withMobileOsType(MobileOsType.ANDROID);
                                    break;
                                default:
                                    builder.withMobile(null);
                                    builder.withMobileOsType(null);
                            }
                            break;
                    }
                }
        );
        return builder.build();
    }

    private FrontendData convertToFrontendData(ReachRequest request) {
        return new FrontendData()
                .withGroupId(request.getAdgroupId())
                .withGroupType(request.getGroupType())
                .withGeo(request.getGeo())
                .withBlockSizes(mapList(request.getBlockSizes(), InventoriWebService::toBlockSize))
                .withVideoCreatives(mapList(request.getVideoCreatives(), InventoriWebService::toVideoCreative))
                .withAudioCreatives(mapList(request.getAudioCreatives(),
                        c -> new AudioCreative(c.getDuration() * 1000,
                                ifNotNull(c.getBlockSize(), InventoriWebService::toBlockSize))))
                .withHasAdaptiveCreative(request.getHasAdaptiveCreative())
                .withConditions(ifNotNull(request.getConditions(),
                        conditions -> singletonList(toRetargetingCondition(request.getConditions()))))
                .withPlatformCorrections(ifNotNull(request.getPlatfromCorrectionsWeb(),
                        p -> PlatformCorrections.builder()
                                .withDesktop(p.getDesktop())
                                .withMobile(p.getMobile())
                                .withMobileOsType(ifNotNull(p.getMobileOsType(),
                                        type -> MobileOsType.valueOf(type.name())))
                                .build()))
                .withTargetTags(request.getTargetTags())
                .withExcludedBsCategories(request.getExcludedBsCategories());
    }

    static RetargetingCondition toRetargetingCondition(List<Condition> conditions) {
        return (RetargetingCondition) new RetargetingCondition().withRules(mapList(conditions,
                InventoriWebService::toRule));
    }

    static Rule toRule(ru.yandex.direct.web.core.entity.inventori.model.Condition c) {
        return new Rule()
                .withInterestType(ifNotNull(c.getInterestType(),
                        CryptaInterestTypeWeb::toCoreType))
                .withType(ifNotNull(c.getType(),
                        RetargetingConditionRuleType::getCoreMappedValue))
                .withGoals(mapList(c.getGoals(),
                        g -> (Goal) new Goal().withId(g.getId()).withTime(g.getTime())));
    }

    @NotNull
    static BlockSize toBlockSize(ru.yandex.direct.web.core.entity.inventori.model.BlockSize b) {
        return new BlockSize(b.getWidth(), b.getHeight());
    }

    @NotNull
    static VideoCreative toVideoCreative(ru.yandex.direct.web.core.entity.inventori.model.VideoCreativeWeb c) {
        return new VideoCreative(c.getDuration() * 1000, null, singleton(VIDEO_PROPORTION_16_9));
    }

    /**
     * Метод расчета охвата аудитории группы объявлений с учетом выбранных аудиторных условий
     *
     * @param request
     * <a href="https://wiki.yandex-team.ru/users/aliho/projects/direct/cryptainventory/#zapros">запрос</a>
     * @return <a href="https://wiki.yandex-team.ru/users/aliho/projects/direct/cryptainventory/#otvet">ответ</a>
     * @see <a href="https://wiki.yandex-team.ru/users/aliho/projects/direct/cryptainventory/#osobennosti">WIKI</a>
     */
    public ReachResult getReachForecast(ReachRequest request) {
        final User subClient = authenticationSource.getAuthentication().getSubjectUser();
        final User operator = authenticationSource.getAuthentication().getOperator();

        final Long campaignId = request.getCampaignId();
        final Long adgroupId = request.getAdgroupId();

        final String requestId = generateRequestId();
        final String clientLogin = getClientLogin(subClient);
        final String operatorLogin = getOperatorLogin(operator);

        final Target basicTarget = createTarget(request, false);

        if (basicTarget != null && INVALID_GROUP_TYPES.contains(basicTarget.getGroupType())) {
            return new ReachResult(requestId,
                    new ReachInfo(DEFAULT_REACH_LESS_THAN, null, null),
                    null);
        }
        final ForecastResponse basicForecast =
                inventoriClient.getForecast(requestId, basicTarget, campaignId, adgroupId, clientLogin, operatorLogin);

        // if basic forecastFake is less than tolerance threshold then detailed forecastFake will be even less
        if (basicForecast.getReachLessThan() != null) {
            return new ReachResult(requestId,
                    new ReachInfo(basicForecast.getReachLessThan(),
                            basicForecast.getReach(),
                            null),
                    null);
        }

        final Target detailedTarget = createTarget(request, true);
        // Не отправлять запрос, если нет ни одного валидного креатива
        final ForecastResponse detailedForecast = (detailedTarget != null && detailedTarget.hasCreatives()) ?
                inventoriClient.getForecast(
                        requestId,
                        detailedTarget,
                        campaignId,
                        adgroupId,
                        clientLogin,
                        operatorLogin) :
                new ForecastResponse();

        // local cache of crypta segments for reuse in errors processing
        final Map<Long, Goal> cryptaGoalMap = EntryStream.of(cryptaService.getSegmentMap())
                // DIRECT-100365: [audio][inventori] Фильтровать жанры при отправке запроса в инвентори
                // Нужно фильтровать жанры на стороне бекенда и НЕ передавать их в крипту пока она их не научится.
                .filterValues(g -> Goal.computeType(g.getId()) != GoalType.AUDIO_GENRES)
                .toMap();

        // errors are expected only in detailed forecast
        List<Error> errors = null;
        if (detailedForecast.getErrors() != null) {
            errors = getErrors(cryptaGoalMap, detailedForecast);
        }
        return new ReachResult(requestId,
                new ReachInfo(basicForecast.getReachLessThan(),
                        basicForecast.getReach(), null),
                new ReachInfo(detailedForecast.getReachLessThan(),
                        detailedForecast.getReach(), errors));
    }

    public ReachResult getGeneralReachForecast(ReachRequest request) {
        final User subClient = authenticationSource.getAuthentication().getSubjectUser();
        final User operator = authenticationSource.getAuthentication().getOperator();

        final Long campaignId = request.getCampaignId();
        final Long adgroupId = request.getAdgroupId();

        final String requestId = generateRequestId();
        final String clientLogin = getClientLogin(subClient);
        final String operatorLogin = getOperatorLogin(operator);

        Map<Long, Goal> segmentMap = cryptaService.getSegmentMap();

        GroupType groupType = request.getGroupType() == GroupType.VIDEO_NON_SKIPPABLE ? GroupType.VIDEO :
                request.getGroupType();

        Target target = new Target()
                .withGroupType(groupType)
                .withRegions(request.getGeo())
                .withBlockSizes(mapList(request.getBlockSizes(), InventoriWebService::toBlockSize))
                .withVideoCreatives(mapList(request.getVideoCreatives(), InventoriWebService::toVideoCreative))
                .withGenresAndCategories(getGenresAndCategories(request, campaignId));

        if (request.getConditions() != null) {
            target.withCryptaGroups(ifNotNull(singletonList(toRetargetingCondition(request.getConditions())),
                    conditions -> AdGroupDataConverter
                            .getCryptaGroups(conditions, segmentMap)));
        }

        if (target != null && INVALID_GROUP_TYPES.contains(target.getGroupType())) {
            return new ReachResult(requestId,
                    new ReachInfo(DEFAULT_REACH_LESS_THAN, null, null),
                    new ReachInfo(DEFAULT_REACH_LESS_THAN, null, null));
        }

        if (request.getGroupType() == GroupType.VIDEO) {
            target.withEnableNonSkippableVideo(false);
        } else if (request.getGroupType() == GroupType.VIDEO_NON_SKIPPABLE) {
            target.withEnableNonSkippableVideo(true);
        }

        final GeneralForecastResponse response =
                inventoriClient.getGeneralForecast(requestId, target, campaignId, adgroupId, clientLogin,
                        operatorLogin);

        // local cache of crypta segments for reuse in errors processing
        final Map<Long, Goal> cryptaGoalMap = EntryStream.of(segmentMap)
                // DIRECT-100365: [audio][inventori] Фильтровать жанры при отправке запроса в инвентори
                // Нужно фильтровать жанры на стороне бекенда и НЕ передавать их в крипту пока она их не научится.
                .filterValues(g -> Goal.computeType(g.getId()) != GoalType.AUDIO_GENRES)
                .toMap();

        ReachInfo basicReachInfo = ifNotNull(response.getBasicReachPrediction(),
                basic -> new ReachInfo(basic.getReachLessThan(), basic.getReach(), null));
        ReachInfo detailedReachInfo = ifNotNull(response.getDetailedReachPrediction(),
                detailed -> new ReachInfo(detailed.getReachLessThan(), detailed.getReach(),
                        getErrors(cryptaGoalMap, detailed)));
        return new ReachResult(requestId, basicReachInfo, detailedReachInfo);
    }

    private List<String> getGenresAndCategories(ReachRequest request, Long campaignId) {
        RetargetingCondition bsRetargetingCondition = null;
        Long adGroupId = request.getAdgroupId();
        FrontendData frontendData = convertToFrontendData(request);
        List<RetargetingCondition> retargetingConditions = emptyList();
        AdGroup adGroup = null;
        if (campaignId != null) {
            int shard = shardHelper.getShardByCampaignId(campaignId);
            var brandSafetyRetargetingsByCampaignId = campaignWithBrandSafetyService.getRetargetingConditions(List.of(campaignId));
            bsRetargetingCondition = brandSafetyRetargetingsByCampaignId.get(campaignId);
            if (adGroupId != null) {
                Map<Long, List<RetargetingCondition>> retargetingConditionsByAdGroupId =
                        retargetingConditionRepository.getRetConditionsByAdGroupIds(shard, List.of(adGroupId));
                retargetingConditions = retargetingConditionsByAdGroupId.get(adGroupId);
                adGroup = adGroupRepository.getAdGroups(shard, List.of(adGroupId)).get(0);
            }
        }

        var retargetings = new ArrayList<RetargetingCondition>();
        if (retargetingConditions != null) {
            retargetings.addAll(retargetingConditions);
        }

        if (bsRetargetingCondition != null) {
            retargetings.add(bsRetargetingCondition);
        }
        return new AdGroupDataConverter()
                .getGenresAndCategories(request.getGroupType(),
                        frontendData,
                        cryptaSegmentRepository.getAll(),
                        retargetings,
                        adGroup == null ? emptyList() : adGroup.getContentCategoriesRetargetingConditionRules());
    }

    /**
     * Метод расчета охвата по нескольким бюджетам
     *
     * @param request Тело запроса
     * @param budgets Предлагаемые бюджеты
     */
    public ReachMultiBudgetResult getReachMultiBudgetsForecast(ReachRequest request, List<Long> budgets) {
        User subClient = authenticationSource.getAuthentication().getSubjectUser();
        ClientId clientId = subClient.getClientId();
        User operator = authenticationSource.getAuthentication().getOperator();
        Long campaignId = request.getCampaignId();
        String requestId = generateRequestId();
        String clientLogin = getClientLogin(subClient);
        String operatorLogin = getOperatorLogin(operator);
        int shard = shardHelper.getShardByClientId(clientId);
        Target target = createTarget(request, true);

        if (request.getGroupType() != null && INVALID_GROUP_TYPES.contains(request.getGroupType())) {
            return new ReachMultiBudgetResult(requestId, DEFAULT_MULTI_BUDGETS_PREDICTION_RESPONCE, null);
        }

        var brandLifts = campaignRepository.getBrandSurveyIdsForCampaigns(shard, List.of(campaignId));
        InventoriCampaignType inventoriCampaignType = InventoriCampaignType.MEDIA_RSYA;

        if (campaignId != null) {
            var campaignType = campaignRepository.getCampaignsTypeMap(shard, List.of(campaignId))
                    .getOrDefault(campaignId, CampaignType.CPM_BANNER);
            inventoriCampaignType = getInventoriCampaignType(campaignType, null);
        } else if (GroupType.MAIN_PAGE_AND_NTP.equals(request.getGroupType())){
            inventoriCampaignType = InventoriCampaignType.MAIN_PAGE_AND_NTP;
        }

        CampaignPredictionRequest campaignPredictionRequest = CampaignPredictionRequest.builder()
                .withCampaignId(campaignId)
                .withCampaignType(inventoriCampaignType)
                .withTargets(singletonList(target))
                .withParameters(defaultCampaignParameters())
                .withIsSocial(featureService.isEnabledForClientId(clientId, FeatureName.SOCIAL_ADVERTISING_BY_LAW))
                .withBrandlift(brandLifts.get(request.getCampaignId()) != null)
                .build();

        List<Param> params = singletonList(new Param(BUDGETS_PARAM, StringUtils.join(budgets, ",")));

        CampaignPredictionResponse response;
        try {
            response = inventoriClient.getParametrisedCampaignPrediction(requestId, operatorLogin, clientLogin,
                    campaignPredictionRequest, params);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not deserialize object from json", e);
        }

        List<Defect> errors = mapList(response.getErrors(), inventoriService::convertError);

        if (response instanceof CampaignPredictionLowReachResponse) {
            CampaignPredictionLowReachResponse typedResponse = (CampaignPredictionLowReachResponse) response;
            errors.add(lowReach(typedResponse.getReachLessThan()));
        }

        MultiBudgetsPredictionResponse forecast = errors.isEmpty() ? (MultiBudgetsPredictionResponse) response : null;

        return new ReachMultiBudgetResult(requestId, forecast, errors);
    }

    /**
     * Метод расчета охвата аудитории outdoor группы объявлений
     *
     * @param request Запрос прогноза охвата
     * @return Прогноз
     */
    public ReachOutdoorResult getReachOutdoorForecast(ReachOutdoorRequest request) {
        User subClient = authenticationSource.getAuthentication().getSubjectUser();
        User operator = authenticationSource.getAuthentication().getOperator();
        ClientId clientId = subClient.getClientId();
        int shard = shardHelper.getShardByClientId(clientId);

        Long campaignId = request.getCampaignId();
        Long adgroupId = request.getAdgroupId();

        String requestId = generateRequestId();
        String clientLogin = getClientLogin(subClient);
        String operatorLogin = getOperatorLogin(operator);

        List<PageBlock> pageBlocks = convertToInventoriPageBlocks(request.getPageBlocks());

        // Данные с фронта
        FrontendData frontendData = new FrontendData()
                .withGroupId(adgroupId)
                .withGroupType(OUTDOOR)
                .withVideoCreativeIds(request.getVideoCreativeIds())
                .withPageBlocks(pageBlocks);

        Map<Long, Pair<List<Target>, Boolean>> targetsbyCampaignIds
                = collector.collectCampaignsInfoWithClientIdAndUid(frontendData, false, shard,
                subClient.getUid(), clientId, singletonList(campaignId), true, null);

        Target target = targetsbyCampaignIds.get(campaignId).getLeft().get(0);

        OutdoorPredictionResponse forecast = inventoriClient
                .getOutdoorPrediction(requestId, target, campaignId, adgroupId, clientLogin, operatorLogin);
        checkState(isEmpty(forecast.getErrors()), "Inventori service responded with unexpected errors, requestId: %s",
                requestId);

        return new ReachOutdoorResult(requestId,
                forecast.getReach(),
                forecast.getOtsCapacity(),
                forecast.getReachLessThan());
    }

    /**
     * Метод расчета охвата аудитории indoor группы объявлений
     *
     * @param request Запрос прогноза охвата
     * @return Прогноз
     */
    public ReachIndoorResult getReachIndoorForecast(ReachIndoorRequest request) {
        User subClient = authenticationSource.getAuthentication().getSubjectUser();
        User operator = authenticationSource.getAuthentication().getOperator();
        ClientId clientId = subClient.getClientId();
        int shard = shardHelper.getShardByClientId(clientId);

        Long campaignId = request.getCampaignId();
        Long adgroupId = request.getAdgroupId();

        String requestId = generateRequestId();
        String clientLogin = getClientLogin(subClient);
        String operatorLogin = getOperatorLogin(operator);

        List<PageBlock> pageBlocks = convertToInventoriPageBlocks(request.getPageBlocks());
        List<ProfileCorrection> profileCorrections =
                convertToInventoriProfileCorrections(request.getBidModifierDemographics());

        // Данные с фронта
        FrontendData frontendData = new FrontendData()
                .withGroupId(adgroupId)
                .withGroupType(INDOOR)
                .withVideoCreativeIds(request.getVideoCreativeIds())
                .withPageBlocks(pageBlocks)
                .withProfileCorrections(profileCorrections);

        Map<Long, Pair<List<Target>, Boolean>> targetsbyCampaignIds =
                collector.collectCampaignsInfoWithClientIdAndUid(frontendData, false, shard,
                        subClient.getUid(), clientId, singletonList(campaignId), true, null);

        Target target = targetsbyCampaignIds.get(campaignId).getLeft().get(0);

        IndoorPredictionResponse forecast = inventoriClient
                .getIndoorPrediction(requestId, target, campaignId, adgroupId, clientLogin, operatorLogin);
        checkState(isEmpty(forecast.getErrors()),
                "Inventori service responded with unexpected errors, requestId: %s", requestId);

        return new ReachIndoorResult(requestId,
                forecast.getReach(),
                forecast.getOtsCapacity(),
                forecast.getReachLessThan());
    }

    public ReachRecommendationResult getReachRecommendation(ReachRequest request) {
        final User subClient = authenticationSource.getAuthentication().getSubjectUser();
        final User operator = authenticationSource.getAuthentication().getOperator();

        Long campaignId = request.getCampaignId();
        Long adgroupId = request.getAdgroupId();

        final String requestId = generateRequestId();
        final String clientLogin = getClientLogin(subClient);
        final String operatorLogin = getOperatorLogin(operator);

        final Target target = createTarget(request, true);

        if (target == null || INVALID_GROUP_TYPES.contains(target.getGroupType())) {
            return new ReachRecommendationResult(requestId, null);
        }

        List<BlockSize> blockSizes = target.getBlockSizes();
        final Set<BlockSize> allowedBlockSizes = getAllowedBlockSizes(target.getGroupType(),
                target.getMainPageTrafficType());

        final ForecastResponse detailedForecast =
                inventoriClient.getForecast(
                        requestId,
                        target,
                        campaignId,
                        adgroupId,
                        clientLogin,
                        operatorLogin);

        // there no reason to continue with invalid forecast request
        if (detailedForecast.getErrors() != null || detailedForecast.getReach() == null) {
            return new ReachRecommendationResult(requestId, null);
        }

        final Map<BlockSize, ForecastResponse> blockSizeForecastResponseMap = allowedBlockSizes.stream()
                .filter(size -> (blockSizes != null && !blockSizes.contains(size)))
                .collect(toMap(identity(),
                        size -> {
                            List<BlockSize> blockSizeList = StreamEx.of(blockSizes).append(size).collect(toList());
                            final Target innerTarget = new Target()
                                    .withAdGroupId(target.getAdGroupId())
                                    .withBlockSizes(blockSizeList)
                                    .withVideoCreatives(target.getVideoCreatives())
                                    .withAudioCreatives(target.getAudioCreatives())
                                    .withGenresAndCategories(target.getGenresAndCategories())
                                    .withExcludedBsCategories(target.getExcludedBsCategories())
                                    .withDomains(target.getDomains())
                                    .withExcludedDomains(target.getExcludedDomains())
                                    .withSsp(target.getSsp())
                                    .withExcludedSsp(target.getExcludedSsp())
                                    .withCryptaGroups(target.getCryptaGroups())
                                    .withAudienceGroups(target.getAudienceGroups())
                                    .withRegions(target.getRegions())
                                    .withPlatformCorrections(target.getPlatformCorrections())
                                    .withCorrections(target.getCorrections())
                                    .withGroupType(target.getGroupType())
                                    .withPageBlocks(target.getPageBlocks())
                                    .withExcludedPageBlocks(target.getExcludedPageBlocks())
                                    .withMainPageTrafficType(target.getMainPageTrafficType());

                            return inventoriClient.getForecast(
                                    requestId,
                                    innerTarget,
                                    campaignId,
                                    adgroupId,
                                    clientLogin,
                                    operatorLogin);
                        }));

        List<BannerFormatIncreasePercent> percents = blockSizeForecastResponseMap.entrySet().stream()
                .filter(pair -> pair.getValue().getErrors() == null && pair.getValue().getReach() != null)
                .map(pair -> new BannerFormatIncreasePercent()
                        .withFormat(String.format("%dx%d", pair.getKey().getWidth(), pair.getKey().getHeight()))
                        .withIncreasePercent(
                                BigDecimal.valueOf((pair.getValue().getReach() - detailedForecast.getReach()) * 100)
                                        .divide(BigDecimal.valueOf(detailedForecast.getReach()),
                                                MathContext.DECIMAL128)))
                .collect(toList());
        return new ReachRecommendationResult(requestId, percents);
    }

    @Nullable
    private List<PageBlock> convertToInventoriPageBlocks(@Nullable List<PageBlockWeb> pageBlocks) {
        return mapList(pageBlocks, pb -> new PageBlock(pb.getPageId(), pb.getBlockIds()));
    }

    @Nullable
    private List<ProfileCorrection> convertToInventoriProfileCorrections(@Nullable List<BidModifierDemographicWeb> bidModifierDemographics) {
        if (isEmpty(bidModifierDemographics)) {
            return null;
        }

        return mapList(bidModifierDemographics, bidModifierDemographic -> ProfileCorrection.builder()
                .withGender(convertToInventoriProfileCorrectionGender(bidModifierDemographic.getGender()))
                .withAge(convertToInventoriProfileCorrectionAge(bidModifierDemographic.getAge()))
                .withCorrection(bidModifierDemographic.getMultiplier())
                .build());
    }

    @Nullable
    private ProfileCorrection.Gender convertToInventoriProfileCorrectionGender(@Nullable String gender) {
        if (gender == null) {
            return null;
        }

        switch (gender) {
            case "all":
                return null;
            case "male":
                return ProfileCorrection.Gender.MALE;
            case "female":
                return ProfileCorrection.Gender.FEMALE;
            default:
                throw new IllegalStateException("Unsupported gender value: " + gender);
        }
    }

    @Nullable
    private ProfileCorrection.Age convertToInventoriProfileCorrectionAge(@Nullable String age) {
        if (age == null) {
            return null;
        }

        switch (age) {
            case "all":
                return null;
            case "0-17":
                return ProfileCorrection.Age._0_17;
            case "18-24":
                return ProfileCorrection.Age._18_24;
            case "25-34":
                return ProfileCorrection.Age._25_34;
            case "35-44":
                return ProfileCorrection.Age._35_44;
            case "45-54":
                return ProfileCorrection.Age._45_54;
            case "55-":
                return ProfileCorrection.Age._55_;
            default:
                throw new IllegalStateException("Unsupported age value: " + age);
        }
    }

    // Инвентори ожидает такие параметры по дефолту, если кампания еще не создана
    private CampaignParameters defaultCampaignParameters() {
        return CampaignParameters.builder()
                .withSchedule(CampaignParametersSchedule.builder()
                        .withStrategyType(MIN_CPM)
                        .withBudget(10_000_000)
                        .withStartDate(LocalDate.now())
                        .withEndDate(LocalDate.now().plusWeeks(1))
                        .withCpm(3_000_000_000L)
                        .withAutoProlongation(true)
                        .build())
                .withRf(new CampaignParametersRf(0, 0))
                .build();
    }

    private static String generateRequestId() {
        return UUID.randomUUID().toString().toUpperCase();
    }

    private String getClientLogin(User client) {
        ClientId clientId = client.getClientId();
        return client.getChiefUid() == null || client.getChiefUid().equals(client.getUid())
                ? client.getLogin()
                : userService.getChiefsLoginsByClientIds(singleton(clientId)).get(clientId);
    }

    private String getOperatorLogin(@Nullable User operator) {
        return Optional.ofNullable(operator).map(User::getLogin).orElse(null);
    }

    public GeneralCpmRecommendationResult forecast(GeneralCpmRecommendationRequest request, CurrencyCode workCurrency) {
        User subClient = authenticationSource.getAuthentication().getSubjectUser();
        User operator = authenticationSource.getAuthentication().getOperator();
        String requestId = generateRequestId();
        String clientLogin = getClientLogin(subClient);
        String operatorLogin = getOperatorLogin(operator);

        if (request.getGroupType() != null && INVALID_GROUP_TYPES.contains(request.getGroupType())) {
            return new GeneralCpmRecommendationResult(requestId, null, DEFAULT_GENERAL_CPM_RECOMMENDATION_RESULT, emptyList());
        }

        CampaignPredictionRequest inventoriRequest = convertForecastRequest(request, workCurrency);



        CampaignPredictionResponse response;
        try {
            response = inventoriClient.getGeneralRecommendation(requestId,
                    operatorLogin,
                    clientLogin,
                    inventoriRequest);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not deserialize object from json", e);
        }

        GeneralCpmRecommendationSuccessResult result = null;
        StreamEx<Defect> errors = StreamEx.of(response.getErrors()).map(inventoriService::convertError);

        if (response instanceof CampaignPredictionLowReachResponse) {
            CampaignPredictionLowReachResponse typedResponse = (CampaignPredictionLowReachResponse) response;
            errors = errors.append(lowReach(typedResponse.getReachLessThan()));
        }

        StrategyType strategyType = null;

        if (request.getStrategy() != null) {
            strategyType = StrategyType.parse(request.getStrategy().getType());
        }


        if (response instanceof GeneralRecommendationResponse) {
            GeneralRecommendationResponse typedResponse = (GeneralRecommendationResponse) response;
            if (typedResponse.getRecommendedMaxAvgCpm() != null || typedResponse.getRecommendedMaxAvgCpv() != null) {
                result = inventoriService.convertResponseToSuccessResult(typedResponse, workCurrency, strategyType);
            }
        }
        return new GeneralCpmRecommendationResult(requestId, request, result, errors.toList());
    }

    public GeneralCampaignPredictionResult getGeneralCampaignPrediction(GeneralCpmRecommendationRequest request,
                                                                        CurrencyCode workCurrency) {
        User subClient = authenticationSource.getAuthentication().getSubjectUser();
        User operator = authenticationSource.getAuthentication().getOperator();
        String requestId = generateRequestId();
        String clientLogin = getClientLogin(subClient);
        String operatorLogin = getOperatorLogin(operator);

        if (request.getGroupType() != null && INVALID_GROUP_TYPES.contains(request.getGroupType())) {
            return new GeneralCampaignPredictionResult(requestId, null, DEFAULT_GENERAL_CAMPAIGN_PREDICTION_SUCCES_RESULT, emptyList());
        }

        CampaignPredictionRequest inventoriRequest = convertForecastRequest(request, workCurrency);

        CampaignPredictionResponse response;
        try {
            response = inventoriClient.getGeneralCampaignPrediction(requestId,
                    operatorLogin,
                    clientLogin,
                    inventoriRequest);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not deserialize object from json", e);
        }

        GeneralCampaignPredictionSuccessResult result = null;
        StreamEx<Defect> errors = StreamEx.of(response.getErrors()).map(inventoriService::convertError);

        if (response instanceof CampaignPredictionLowReachResponse) {
            CampaignPredictionLowReachResponse typedResponse = (CampaignPredictionLowReachResponse) response;
            errors = errors.append(lowReach(typedResponse.getReachLessThan()));
        }

        StrategyType strategyType = null;

        if (request.getStrategy() != null) {
            strategyType = StrategyType.parse(request.getStrategy().getType());
        }


        if (response instanceof GeneralCampaignPredictionResponse) {
            GeneralCampaignPredictionResponse typedResponse = (GeneralCampaignPredictionResponse) response;
            if (typedResponse.getRecommendedMaxAvgCpm() != null || typedResponse.getRecommendedMaxAvgCpv() != null) {
                result = inventoriService.convertGeneralCampaignPredictionResponseToSuccessResult(
                        typedResponse, workCurrency, strategyType);
            }
        }
        return new GeneralCampaignPredictionResult(requestId, inventoriRequest, result, errors.toList());
    }

    /**
     * Конвертирует запрос в CPM-прогнозатор.
     *
     * @param request  запрос
     * @param currency валюта клиента/кампании
     * @return объект запроса в CPM-прогнозатор
     */
    public CampaignPredictionRequest convertForecastRequest(GeneralCpmRecommendationRequest request,
                                                            CurrencyCode currency) {
        CampaignPredictionRequest.Builder builder = CampaignPredictionRequest.builder();

        CampaignStrategy strategy = request.getStrategy();
        if (request.getCampaignId() != null) {
            builder.withCampaignId(request.getCampaignId());
        }

        CampaignParametersCorrections campaignParametersCorrections = getCampaignParametersCorrections(request);

        builder.withTargets(List.of(
                createTarget(convertGeneralCpmRecommendationRequestToReachRequest(request), true)
                        .withCorrections(campaignParametersCorrections)
                        .withPlatformCorrections(getPlatformCorrections(request))));

        StrategyType strategyType = StrategyType.parse(strategy.getType());
        CampaignParameters.Builder parametersBuilder = CampaignParameters.builder()
                .withSchedule(inventoriService.getSchedule(currency, strategy, null, strategyType, null))
                .withRf(inventoriService.getRf(strategy))
                .withCorrections(campaignParametersCorrections);

        builder.withCampaignType(getInventoriCampaignType(
                inventoriService.convertCpmCampaignTypeToCampaignType(request.getCpmCampaignType()), null));
        builder.withParameters(parametersBuilder.build());
        builder.withBrandlift(request.getHasBrandlift());
        return builder.build();
    }

    private CampaignParametersCorrections getCampaignParametersCorrections(GeneralCpmRecommendationRequest request) {
        var inventoryTypes = request.getInventoryTypes();
        if (inventoryTypes == null || inventoryTypes.isEmpty()) {
            return null;
        }
        return new CampaignParametersCorrections(
                new TrafficTypeCorrections(100,
                        (inventoryTypes.contains(InventoryType.INPAGE)) ? 100 : 0,
                        (inventoryTypes.contains(InventoryType.INSTREAM)) ? 100 : 0,
                        (inventoryTypes.contains(InventoryType.INAPP)) ? 100 : 0,
                        //inbanner всегда идет вместе с inapp
                        (inventoryTypes.contains(InventoryType.INAPP)) ? 100 : 0,
                        (inventoryTypes.contains(InventoryType.REWARDED)) ? 100 : 0));
    }

    private PlatformCorrections getPlatformCorrections(GeneralCpmRecommendationRequest request) {
        var deviceTypes = request.getDeviceTypes();
        if (deviceTypes == null || deviceTypes.isEmpty()) {
            return null;
        }
        MobileOsType osType = null;
        boolean isMobile = containsAny(deviceTypes, List.of(DeviceType.PHONE_ANDROID, DeviceType.PHONE_IOS, DeviceType.PHONE));

        if (deviceTypes.containsAll(List.of(DeviceType.PHONE_ANDROID, DeviceType.PHONE_IOS))) {
            // если выбрано обе ОС, то считаем, что все смартфоны (мобильные телефоны)
            osType = null;
        } else if (deviceTypes.contains(DeviceType.PHONE_ANDROID)) {
            osType = MobileOsType.ANDROID;
        } else if (deviceTypes.contains(DeviceType.PHONE_IOS)) {
            osType = MobileOsType.IOS;
        }
        //SmartTv инвентори пока не поддерживает

        return PlatformCorrections.builder()
                .withDesktop((deviceTypes.contains(DeviceType.DESKTOP)) ? 100 : 0)
                .withMobile((isMobile) ? 100 : 0)
                .withMobileOsType(osType)
                .build();
    }

    private ReachRequest convertGeneralCpmRecommendationRequestToReachRequest(GeneralCpmRecommendationRequest request) {
        return new ReachRequest()
                .withCampaignId(request.getCampaignId())
                .withGroupType(request.getGroupType())
                .withGeo(request.getGeo())
                .withBlockSizes(request.getBlockSizes())
                .withVideoCreatives(request.getVideoCreatives())
                .withConditions(request.getConditions())
                .withExcludedDomains(request.getExcludedDomains())
                .withExcludedBsCategories(request.getExcludedBsCategories());
    }
}
