package ru.yandex.direct.web.core.entity.inventori.service;

import java.util.List;
import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.entity.inventori.model.Condition;
import ru.yandex.direct.web.core.entity.inventori.model.GeneralCpmRecommendationRequest;
import ru.yandex.direct.web.core.entity.inventori.model.Goal;
import ru.yandex.direct.web.core.entity.inventori.model.PageBlockWeb;
import ru.yandex.direct.web.core.entity.inventori.model.ReachIndoorRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ReachOutdoorRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ReachRequest;
import ru.yandex.direct.web.core.entity.inventori.validation.InventoriConstraints;

import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeConstraints.isClientHasCreative;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Service
public class InventoriWebValidationService {

    private final CreativeRepository creativeRepository;
    private final ShardHelper shardHelper;
    private final CampaignSubObjectAccessCheckerFactory accessCheckerFactory;

    @Autowired
    public InventoriWebValidationService(CreativeRepository creativeRepository, ShardHelper shardHelper,
                                         CampaignSubObjectAccessCheckerFactory accessCheckerFactory) {
        this.creativeRepository = creativeRepository;
        this.shardHelper = shardHelper;
        this.accessCheckerFactory = accessCheckerFactory;
    }

    /**
     * Валидирует запрос на получение прогноза по кампании.
     *
     * @param request     запрос
     * @param clientId    id клиента
     * @param operatorUid uid оператора
     * @return результат валидации
     */
    public ValidationResult<GeneralCpmRecommendationRequest, Defect> validate(
            GeneralCpmRecommendationRequest request,
            long operatorUid,
            ClientId clientId) {

        ItemValidationBuilder<GeneralCpmRecommendationRequest, Defect> vb = ItemValidationBuilder.of(request);

        vb.item(request.getCampaignId(), "campaign_id")
                .check(validId(), When.notNull());

        vb.item(request.getStrategy(), "strategy")
                .check(notNull())
                .checkBy(strategy -> InventoriConstraints.validateStrategy(strategy, false), When.isValid());

        if (request.getCampaignId() != null) {
            CampaignSubObjectAccessChecker checker = accessCheckerFactory.newCampaignChecker(
                    operatorUid, clientId, singletonList(request.getCampaignId()));
            vb.item(request.getCampaignId(), "campaign_id")
                    .checkBy(checker.createValidator(CampaignAccessType.READ), When.isValid());
        }

        return vb.getResult();
    }

    public ValidationResult<ReachRequest, Defect> validate(ReachRequest request) {
        ItemValidationBuilder<ReachRequest, Defect> vb = ItemValidationBuilder.of(request);
        vb.check(notNull());
        if (!vb.getResult().hasAnyErrors()) {
            vb.list(request.getConditions(),
                    "conditions")
                    .checkEachBy(this::validateCondition, When.isValid());
        }
        return vb.getResult();
    }

    public ValidationResult<ReachOutdoorRequest, Defect> validate(ReachOutdoorRequest request, ClientId clientId) {
        ItemValidationBuilder<ReachOutdoorRequest, Defect> vb = ItemValidationBuilder.of(request);
        vb.check(notNull());
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        if (CollectionUtils.isNotEmpty(request.getVideoCreativeIds())) {
            vb.item(request.getVideoCreativeIds(), ReachOutdoorRequest.Prop.VIDEO_CREATIVE_IDS)
                    .checkBy(videoCreativeIds -> validateVideoCreativeIds(videoCreativeIds, clientId,
                            CreativeType.CPM_OUTDOOR_CREATIVE));
        }

        if (CollectionUtils.isNotEmpty(request.getPageBlocks())) {
            vb.list(request.getPageBlocks(), ReachOutdoorRequest.Prop.PAGE_BLOCKS)
                    .checkEachBy(this::validatePageBlock, When.isValid());
        }

        return vb.getResult();
    }

    public ValidationResult<ReachIndoorRequest, Defect> validate(ReachIndoorRequest request, ClientId clientId) {
        ItemValidationBuilder<ReachIndoorRequest, Defect> vb = ItemValidationBuilder.of(request);
        vb.check(notNull());
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        if (CollectionUtils.isNotEmpty(request.getVideoCreativeIds())) {
            vb.item(request.getVideoCreativeIds(), ReachIndoorRequest.Prop.VIDEO_CREATIVE_IDS)
                    .checkBy(videoCreativeIds -> validateVideoCreativeIds(videoCreativeIds, clientId,
                            CreativeType.CPM_INDOOR_CREATIVE));
        }

        if (CollectionUtils.isNotEmpty(request.getPageBlocks())) {
            vb.list(request.getPageBlocks(), ReachIndoorRequest.Prop.PAGE_BLOCKS)
                    .checkEachBy(this::validatePageBlock, When.isValid());
        }

        return vb.getResult();
    }

    private ValidationResult<Condition, Defect> validateCondition(Condition condition) {
        ItemValidationBuilder<Condition, Defect> vb =
                ItemValidationBuilder.of(condition, Defect.class);
        vb.check(notNull());

        if (!vb.getResult().hasAnyErrors()) {
            vb.list(condition.getGoals(), "goals")
                    .checkEachBy(this::validateGoal);
        }
        return vb.getResult();
    }

    private ValidationResult<Goal, Defect> validateGoal(Goal goal) {
        ItemValidationBuilder<Goal, Defect> vb = ItemValidationBuilder.of(goal, Defect.class);
        vb.check(notNull());
        if (!vb.getResult().hasAnyErrors()) {
            vb.item(goal.getId(), "id").check(notNull());
        }
        return vb.getResult();
    }

    private ValidationResult<PageBlockWeb, Defect> validatePageBlock(PageBlockWeb pageBlock) {
        ItemValidationBuilder<PageBlockWeb, Defect> vb = ItemValidationBuilder.of(pageBlock, Defect.class);
        vb.check(notNull());
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        vb.item(pageBlock.getPageId(), PageBlockWeb.Prop.PAGE_ID)
                .check(notNull());

        vb.list(pageBlock.getBlockIds(), PageBlockWeb.Prop.BLOCK_IDS)
                .check(notNull())
                .check(notEmptyCollection(), When.isValid())
                .checkEach(notNull(), When.isValid());

        return vb.getResult();
    }

    private ValidationResult<List<Long>, Defect> validateVideoCreativeIds(List<Long> videoCreativeIds,
                                                                          ClientId clientId,
                                                                          CreativeType creativeType) {
        ListValidationBuilder<Long, Defect> vb = ListValidationBuilder.of(videoCreativeIds, Defect.class);
        vb.check(notNull())
                .checkEach(notNull(), When.isValid());
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        int shard = shardHelper.getShardByClientId(clientId);
        Set<Long> existingCreatives = creativeRepository.getExistingClientCreativeIds(shard, clientId,
                videoCreativeIds, singleton(creativeType));

        return vb.checkEach(isClientHasCreative(existingCreatives))
                .getResult();
    }
}
