package ru.yandex.direct.web.core.entity.inventori.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.entity.inventori.model.CampaignStrategy;
import ru.yandex.direct.web.core.entity.inventori.model.CpmCampaignType;
import ru.yandex.direct.web.core.entity.inventori.model.CpmForecastRequest;
import ru.yandex.direct.web.core.entity.inventori.model.ImpressionLimit;
import ru.yandex.direct.web.core.entity.inventori.model.TrafficTypeCorrectionsWeb;

import static ru.yandex.direct.validation.Predicates.inSet;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotBeforeThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.greaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;
import static ru.yandex.direct.web.core.entity.inventori.service.InventoriService.INVALID_ADGROUPS_TYPES;
import static ru.yandex.direct.web.core.entity.inventori.service.InventoriService.VALID_CAMPAIGN_TYPES;
import static ru.yandex.direct.web.core.entity.inventori.service.InventoriService.VALID_STRATEGIES;
import static ru.yandex.direct.web.core.entity.inventori.validation.InventoriDefects.campaignAlreadyDeleted;
import static ru.yandex.direct.web.core.entity.inventori.validation.InventoriDefects.campaignNotFound;
import static ru.yandex.direct.web.core.entity.inventori.validation.InventoriDefects.invalidAdGroupType;
import static ru.yandex.direct.web.core.entity.inventori.validation.InventoriDefects.invalidCampaignStrategy;
import static ru.yandex.direct.web.core.entity.inventori.validation.InventoriDefects.invalidCampaignType;

public class InventoriConstraints {

    private static final Set<String> STRATEGY_TYPES = ImmutableSet.of(
            "MAX_REACH", "MAX_REACH_CUSTOM_PERIOD",
            "MIN_CPM", "MIN_CPM_CUSTOM_PERIOD",
            "MAX_AVG_CPV", "MAX_AVG_CPV_CUSTOM_PERIOD");
    private static final Set<String> CPV_STRATEGY_TYPES = ImmutableSet.of(
            "MAX_AVG_CPV", "MAX_AVG_CPV_CUSTOM_PERIOD");

    public static Constraint<Long, Defect> campaignExists(Set<Long> existingCampaignIds) {
        return Constraint.fromPredicate(inSet(existingCampaignIds), campaignNotFound());
    }

    public static Constraint<Long, Defect> validStatusEmpty(Map<Long, Campaign> campaignById) {
        return Constraint.fromPredicate(cid -> !campaignById.get(cid).getStatusEmpty(), campaignAlreadyDeleted());
    }

    public static Constraint<Long, Defect> validStrategy(Map<Long, Campaign> campaignById) {
        return cid -> {
            StrategyName strategyName = campaignById.get(cid).getStrategy().getStrategyName();
            return VALID_STRATEGIES.contains(strategyName) ? null : invalidCampaignStrategy(strategyName);
        };
    }

    public static Constraint<Long, Defect> validGroups(Map<Long, List<AdGroupSimple>> adGroupsTypeByCampaignID) {
        return cid -> {
            if (adGroupsTypeByCampaignID.get(cid) == null) {
                return null;
            }
            var adGroups = adGroupsTypeByCampaignID.get(cid).stream().map(AdGroupSimple::getType).collect(Collectors.toList());
            return adGroups.stream().anyMatch(INVALID_ADGROUPS_TYPES::contains) ? invalidAdGroupType(adGroups.stream().filter(INVALID_ADGROUPS_TYPES::contains).collect(Collectors.toList())) : null;
        };
    }

    public static Constraint<Long, Defect> validCampaignType(Map<Long, Campaign> campaignById) {
        return cid -> {
            CampaignType campaignType = campaignById.get(cid).getType();
            return VALID_CAMPAIGN_TYPES.contains(campaignType) ? null : invalidCampaignType(campaignType);
        };
    }

    public static void validateCommon(CpmForecastRequest request, ItemValidationBuilder<CpmForecastRequest, Defect> vb) {
        vb.item(request.getNewCampaignExampleType(), "new_campaign_example_type")
                .check(notNull(), When.isTrue(request.getCampaignId() == null))
                .check(isNull(), When.isTrue(request.getCampaignId() != null))
                .check(CommonConstraints.inSet(ImmutableSet.of(0, 1)));

        vb.item(request.getCpmCampaignType(), "campaign_type")
                // TODO dimitrovsd: раскомментить после доработки фронта
                //.check(notNull(), When.isTrue(request.getCampaignId() == null))
                .check(isNull(), When.isTrue(request.getCampaignId() != null));

        vb.item(request.getTargetTags(), "target_tags")
                .check(notEmptyCollection(), When.isTrue(request.getCampaignId() == null
                        && request.getCpmCampaignType() == CpmCampaignType.CPM_YNDX_FRONTPAGE));
    }

    public static ValidationResult<CampaignStrategy, Defect> validateStrategy(CampaignStrategy strategy, boolean requiredCpm) {

        boolean requiredCpv = CPV_STRATEGY_TYPES.contains(strategy.getType());

        ItemValidationBuilder<CampaignStrategy, Defect> vb = ItemValidationBuilder.of(strategy);

        vb.item(strategy.getType(), "type")
                .check(notNull())
                .check(CommonConstraints.inSet(STRATEGY_TYPES), When.isValid());

        vb.item(strategy.getBudget(), "budget")
                .check(notNull())
                .check(greaterThan(0.0));

        vb.item(strategy.getStartDate(), "start_date")
                .check(notNull());

        vb.item(strategy.getEndDate(), "end_date")
                .check(notNull())
                .check(isNotBeforeThan(strategy.getStartDate()),
                        When.isValidAnd(When.isTrue(strategy.getStartDate() != null)));

        vb.item(strategy.getImpressionLimit(), "impression_limit")
                .check(notNull())
                .checkBy(InventoriConstraints::validateImpressionLimit, When.isValid());

        vb.item(strategy.getCpm(), "cpm")
                .check(notNull(), When.isTrue(requiredCpm && !requiredCpv))
                .check(greaterThan(0.0), When.isTrue(requiredCpm && !requiredCpv));

        vb.item(strategy.getCpv(), "cpv")
                .check(notNull(), When.isTrue(requiredCpv))
                .check(greaterThan(0.0), When.isTrue(requiredCpv));

        return vb.getResult();
    }

    public static ValidationResult<ImpressionLimit, Defect> validateImpressionLimit(ImpressionLimit limit) {
        ItemValidationBuilder<ImpressionLimit, Defect> vb = ItemValidationBuilder.of(limit);

        vb.item(limit.getDays(), "days")
                .check(notNull())
                .check(notLessThan(0L));

        vb.item(limit.getImpressions(), "impressions")
                .check(notNull())
                .check(notLessThan(0L));

        return vb.getResult();
    }

    public static ValidationResult<TrafficTypeCorrectionsWeb, Defect> validateCorrection(
            TrafficTypeCorrectionsWeb trafficTypeCorrections) {
        ItemValidationBuilder<TrafficTypeCorrectionsWeb, Defect> vb = ItemValidationBuilder.of(trafficTypeCorrections);
        vb.item(trafficTypeCorrections.getBanner(), "banner")
                .check(notLessThan(0), When.notNull())
                .check(notGreaterThan(1300), When.notNull());

        vb.item(trafficTypeCorrections.getVideoInpage(), "video_inpage")
                .check(notLessThan(0), When.notNull())
                .check(notGreaterThan(1300), When.notNull());

        vb.item(trafficTypeCorrections.getVideoInstream(), "video_instream")
                .check(notLessThan(0), When.notNull())
                .check(notGreaterThan(1300), When.notNull());

        vb.item(trafficTypeCorrections.getVideoInterstitial(), "video_interstitial")
                .check(notLessThan(0), When.notNull())
                .check(notGreaterThan(1300), When.notNull());
        return vb.getResult();
    }
}
