package ru.yandex.direct.web.core.entity.mobilecontent.converter;

import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.converter.MobileContentInfoConverter;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileContentWithExtraData;
import ru.yandex.direct.core.entity.mobilecontent.model.ApiMobileContentYT;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentAvatarSize;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.web.core.entity.mobilecontent.model.Action;
import ru.yandex.direct.web.core.entity.mobilecontent.model.AgeLabel;
import ru.yandex.direct.web.core.entity.mobilecontent.model.ApiMobileContent;
import ru.yandex.direct.web.core.entity.mobilecontent.model.DeviceType;
import ru.yandex.direct.web.core.entity.mobilecontent.model.DisplayedAttribute;
import ru.yandex.direct.web.core.entity.mobilecontent.model.Network;
import ru.yandex.direct.web.core.entity.mobilecontent.model.Store;
import ru.yandex.direct.web.core.entity.mobilecontent.model.TrackingSystem;
import ru.yandex.direct.web.core.entity.mobilecontent.model.WebMobileApp;
import ru.yandex.direct.web.core.entity.mobilecontent.model.WebMobileAppTracker;
import ru.yandex.direct.web.core.entity.mobilecontent.model.WebMobileContent;
import ru.yandex.direct.web.core.entity.mobilecontent.model.WebOsType;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.converter.Converters.mappingValueConverter;

@Component
@ParametersAreNonnullByDefault
public class MobileContentConverter {
    private static final Map<OsType, Store> OS_TYPE_TO_STORE =
            ImmutableMap.of(
                    OsType.IOS, Store.APP_STORE,
                    OsType.ANDROID, Store.GOOGLE_PLAY
            );
    public static final Map<OsType, WebOsType> OS_TYPE_MAPPING =
            ImmutableMap.of(
                    OsType.IOS, WebOsType.IOS,
                    OsType.ANDROID, WebOsType.ANDROID
            );

    private static final Map<ru.yandex.direct.core.entity.mobilecontent.model.AgeLabel, AgeLabel> AGE_LABEL_MAP =
            ImmutableMap.of(
                    ru.yandex.direct.core.entity.mobilecontent.model.AgeLabel._0_2B, AgeLabel.AGE_O,
                    ru.yandex.direct.core.entity.mobilecontent.model.AgeLabel._6_2B, AgeLabel.AGE_6,
                    ru.yandex.direct.core.entity.mobilecontent.model.AgeLabel._12_2B, AgeLabel.AGE_12,
                    ru.yandex.direct.core.entity.mobilecontent.model.AgeLabel._16_2B, AgeLabel.AGE_16,
                    ru.yandex.direct.core.entity.mobilecontent.model.AgeLabel._18_2B, AgeLabel.AGE_18
            );

    public MobileContentConverter() {
    }

    public WebMobileApp createWebMobileApp(MobileContentWithExtraData mobileContentWithExtraData,
                                           @Nullable String publisherDomain) {
        String storeHref = mobileContentWithExtraData.getStoreHref();
        MobileContent mobileContent = mobileContentWithExtraData.getMobileContent();
        Store store = OS_TYPE_TO_STORE.getOrDefault(mobileContent.getOsType(), Store.OTHER);
        @Nullable String trackingUrl = mobileContentWithExtraData.getTrackingUrl();

        return new WebMobileApp()
                .withId(mobileContent.getId())
                .withName(mobileContent.getName())
                .withStoreHref(storeHref)
                .withStore(store)
                .withMobileContentId(mobileContent.getId())
                .withMobileContent(createMobileContentInfo(mobileContent, publisherDomain))
                .withPrimaryAction(null)
                .withMinOsVersion(mobileContent.getMinOsVersion())
                .withDeviceTypeTargeting(EnumSet.allOf(DeviceType.class))
                .withNetworkTargeting(EnumSet.allOf(Network.class))
                .withDisplayedAttributes(EnumSet.allOf(DisplayedAttribute.class))
                .withPrimaryAction(Action.DOWNLOAD)
                .withGeoTargeting(getGeoTargeting(mobileContent))
                .withTrackers(createTrackers(trackingUrl))
                .withFiles(Collections.emptyList());
    }

    private List<WebMobileAppTracker> createTrackers(@Nullable String trackingUrl) {
        if (trackingUrl == null) {
            return Collections.emptyList();
        }
        return Collections.singletonList(
                new WebMobileAppTracker()
                        .withTrackingSystem(TrackingSystem.OTHER)
                        .withUrl(trackingUrl)
                        .withUserParams(emptyList()));
    }

    @Nullable
    Set<Long> getGeoTargeting(MobileContent mobileContent) {
        return mobileContent.getStoreCountry().equals("RU") ? Collections.singleton(Region.RUSSIA_REGION_ID) : null;
    }

    public WebMobileContent createMobileContentInfo(MobileContent mobileContent, @Nullable String publisherDomain) {
        return new WebMobileContent()
                .withId(mobileContent.getId())
                .withName(mobileContent.getName())
                .withContentType("app")
                .withOsType(mappingValueConverter(OS_TYPE_MAPPING).convert(mobileContent.getOsType()))
                .withStoreCountry(mobileContent.getStoreCountry())
                .withStoreContentId(mobileContent.getStoreContentId())
                .withBundleId(mobileContent.getBundleId())
                .withIconUrl(mobileContent.getIconHash() == null ? null : generateIconUrl(mobileContent))
                .withRating(mobileContent.getRating())
                .withRatingVotes(nvl(mobileContent.getRatingVotes(), 0L))
                .withAgeLabel(mappingValueConverter(AGE_LABEL_MAP).convert(mobileContent.getAgeLabel()))
                .withMinOsVersion(mobileContent.getMinOsVersion())
                .withAvailableActions(mobileContent.getAvailableActions())
                .withIsAvailable(nvl(mobileContent.getIsAvailable(), false))
                .withIsShowIcon(mobileContent.getIconHash() != null)
                .withPublisherDomain(publisherDomain)
                .withPrices(mobileContent.getPrices().getOrDefault(mobileContent.getStoreCountry(), emptyMap()));
    }

    private String generateIconUrl(MobileContent mobileContent) {
        // хост mds пока захардкожен продовый, иначе картинки не видны, см. DIRECT-80920
        return MobileContentService.generateUrlString(MobileContentInfoConverter.AVATARS_MDS_HOST,
                mobileContent.getOsType(), mobileContent.getIconHash(), MobileContentAvatarSize.ICON);
    }

    public ApiMobileContent createApiMobileContent(ApiMobileContentYT mobileContent) {
        return new ApiMobileContent()
                .withStore(mobileContent.getStore())
                .withAppId(mobileContent.getAppId())
                .withLang(mobileContent.getLang())
                .withBundle(mobileContent.getBundle())
                .withName(mobileContent.getName())
                .withIcon(mobileContent.getIcon())
                .withCurrency(mobileContent.getCurrency())
                .withPublisher(mobileContent.getPublisher())
                .withWebsite(mobileContent.getWebsite())
                .withRatingCount(mobileContent.getRatingCount());
    }
}
