package ru.yandex.direct.web.core.entity.mobilecontent.converter;

import java.util.Collections;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.mobileapp.Constants;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTracker;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTrackerTrackingSystem;
import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.web.core.entity.mobilecontent.model.TrackingSystem;
import ru.yandex.direct.web.core.entity.mobilecontent.model.WebMobileAppTracker;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.converter.Converters.mappingValueConverter;
import static ru.yandex.direct.utils.converter.Converters.nullSafeConverter;
import static ru.yandex.direct.web.core.entity.mobilecontent.model.TrackingSystem.APPMETRICA;

@Component
@ParametersAreNonnullByDefault
public class TrackerConverter {
    static final BiMap<MobileAppTrackerTrackingSystem, TrackingSystem> TRACKING_SYSTEM_MAP =
            ImmutableBiMap.<MobileAppTrackerTrackingSystem, TrackingSystem>builder()
                    .put(MobileAppTrackerTrackingSystem.OTHER, TrackingSystem.OTHER)
                    .put(MobileAppTrackerTrackingSystem.APPMETRICA, APPMETRICA)
                    .put(MobileAppTrackerTrackingSystem.ADJUST, TrackingSystem.ADJUST)
                    .put(MobileAppTrackerTrackingSystem.APPSFLYER, TrackingSystem.APPSFLYER)
                    .put(MobileAppTrackerTrackingSystem.TUNE, TrackingSystem.TUNE)
                    .put(MobileAppTrackerTrackingSystem.MAIL_RU, TrackingSystem.MAIL_RU)
                    .put(MobileAppTrackerTrackingSystem.FLURRY, TrackingSystem.FLURRY)
                    .put(MobileAppTrackerTrackingSystem.KOCHAVA, TrackingSystem.KOCHAVA)
                    .put(MobileAppTrackerTrackingSystem.SINGULAR, TrackingSystem.SINGULAR)
                    .put(MobileAppTrackerTrackingSystem.APPMETRICAUL, TrackingSystem.APPMETRICAUL)
                    .put(MobileAppTrackerTrackingSystem.ADJUSTUL, TrackingSystem.ADJUSTUL)
                    .put(MobileAppTrackerTrackingSystem.APPSFLYERUL, TrackingSystem.APPSFLYERUL)
                    .put(MobileAppTrackerTrackingSystem.BRANCH, TrackingSystem.BRANCH)
                    .build();

    private final TranslationService translationService;

    public TrackerConverter(TranslationService translationService) {
        this.translationService = translationService;
    }

    List<WebMobileAppTracker> convertTrackersToWeb(List<MobileAppTracker> trackers) {
        return mapList(trackers, this::convertTrackerToWeb);
    }

    Boolean convertTrackersToIsAppMetrika(List<MobileAppTracker> trackers) {
        return listToSet(trackers, MobileAppTracker::getTrackingSystem)
                .contains(MobileAppTrackerTrackingSystem.APPMETRICA);
    }

    private WebMobileAppTracker convertTrackerToWeb(MobileAppTracker tracker) {
        Translatable trackerSystemName = Constants.TRACKER_NAMES.get(tracker.getTrackingSystem());
        return new WebMobileAppTracker()
                .withTrackingSystem(convertTrackingSystemToWeb(tracker.getTrackingSystem()))
                .withTrackingSystemName(translationService.translate(trackerSystemName))
                .withTrackerId(tracker.getTrackerId())
                .withUrl(tracker.getUrl())
                .withImpressionUrl(tracker.getImpressionUrl())
                .withUserParams(tracker.getUserParams());
    }

    public List<MobileAppTracker> convertTrackersToCore(List<WebMobileAppTracker> trackers) {
        return mapList(trackers, this::convertTrackerToCore);
    }

    private MobileAppTracker convertTrackerToCore(WebMobileAppTracker tracker) {
        return new MobileAppTracker()
                .withTrackingSystem(convertTrackingSystemToCore(tracker.getTrackingSystem()))
                .withTrackerId(ifNotNull(tracker.getTrackerId(), String::trim))
                .withUrl(ifNotNull(tracker.getUrl(), String::trim))
                .withImpressionUrl(convertImpressionUrlToCore(tracker.getImpressionUrl()))
                .withUserParams(convertUserParams(tracker.getUserParams()));
    }

    private List<String> convertUserParams(@Nullable List<String> tracker) {
        if (tracker == null) {
            return Collections.emptyList();
        }
        return nullSafeConverter(String::trim).convertList(tracker);
    }

    private String convertImpressionUrlToCore(@Nullable String impressionUrl) {
        return impressionUrl != null ? impressionUrl.trim() : "";
    }

    public TrackingSystem convertTrackingSystemToWeb(@Nullable MobileAppTrackerTrackingSystem trackingSystem) {
        return mappingValueConverter(TRACKING_SYSTEM_MAP).convert(trackingSystem);
    }

    private MobileAppTrackerTrackingSystem convertTrackingSystemToCore(@Nullable TrackingSystem trackingSystem) {
        return mappingValueConverter(TRACKING_SYSTEM_MAP.inverse()).convert(trackingSystem);
    }
}
