package ru.yandex.direct.web.core.entity.mobilecontent.converter;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;
import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.mobileapp.model.AppmetrikaEventSubtype;
import ru.yandex.direct.core.entity.mobileapp.model.AppmetrikaEventType;
import ru.yandex.direct.core.entity.mobileapp.model.ExternalTrackerEventName;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppDeviceTypeTargeting;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppDisplayedAttribute;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppMetrikaEvent;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppNetworkTargeting;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppPrimaryAction;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppStoreType;
import ru.yandex.direct.core.entity.mobileapp.model.MobileExternalTrackerEvent;
import ru.yandex.direct.core.entity.mobileapp.service.SkAdNetworkSlotsConfigProvider;
import ru.yandex.direct.core.entity.mobilegoals.model.AppmetrikaInternalEvent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.utils.converter.Converter;
import ru.yandex.direct.utils.converter.Converters;
import ru.yandex.direct.web.core.entity.mobilecontent.model.Action;
import ru.yandex.direct.web.core.entity.mobilecontent.model.DeviceType;
import ru.yandex.direct.web.core.entity.mobilecontent.model.DisplayedAttribute;
import ru.yandex.direct.web.core.entity.mobilecontent.model.Network;
import ru.yandex.direct.web.core.entity.mobilecontent.model.Store;
import ru.yandex.direct.web.core.entity.mobilecontent.model.WebMobileApp;
import ru.yandex.direct.web.core.entity.mobilecontent.model.WebMobileEvent;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.converter.Converters.immutableSetConverter;
import static ru.yandex.direct.utils.converter.Converters.mappingValueConverter;

@Component
@ParametersAreNonnullByDefault
public class WebCoreMobileAppConverter {
    private static final Map<MobileAppStoreType, Store> STORE_MAP = ImmutableMap.of(
            MobileAppStoreType.APPLEAPPSTORE, Store.APP_STORE,
            MobileAppStoreType.GOOGLEPLAYSTORE, Store.GOOGLE_PLAY,
            MobileAppStoreType.OTHER, Store.OTHER
    );

    private static final BiMap<MobileAppPrimaryAction, Action> ACTION_MAP =
            ImmutableBiMap.<MobileAppPrimaryAction, Action>builder()
                    .put(MobileAppPrimaryAction.DOWNLOAD, Action.DOWNLOAD)
                    .put(MobileAppPrimaryAction.GET, Action.GET)
                    .put(MobileAppPrimaryAction.INSTALL, Action.INSTALL)
                    .put(MobileAppPrimaryAction.MORE, Action.MORE)
                    .put(MobileAppPrimaryAction.OPEN, Action.OPEN)
                    .put(MobileAppPrimaryAction.UPDATE, Action.UPDATE)
                    .put(MobileAppPrimaryAction.PLAY, Action.PLAY)
                    .put(MobileAppPrimaryAction.BUY, Action.BUY)
                    .build();


    private static final Converter<MobileAppPrimaryAction, Action> PRIMARY_ACTION_CONVERTER =
            mappingValueConverter(ACTION_MAP);

    private static final BiMap<MobileAppDisplayedAttribute, DisplayedAttribute> DISPLAYED_ATTRIBUTE_MAP =
            ImmutableBiMap.of(
                    MobileAppDisplayedAttribute.RATING, DisplayedAttribute.RATING,
                    MobileAppDisplayedAttribute.PRICE, DisplayedAttribute.PRICE,
                    MobileAppDisplayedAttribute.ICON, DisplayedAttribute.ICON,
                    MobileAppDisplayedAttribute.RATING_VOTES, DisplayedAttribute.RATING_VOTES);

    public static final Converter<Collection<MobileAppDisplayedAttribute>, Set<DisplayedAttribute>>
            DISPLAYED_ATTRIBUTE_SET_CONVERTER = immutableSetConverter(DISPLAYED_ATTRIBUTE_MAP);

    public static final Converter<Collection<DisplayedAttribute>, Set<MobileAppDisplayedAttribute>>
            DISPLAYED_ATTRIBUTE_SET_TO_CORE_CONVERTER = immutableSetConverter(DISPLAYED_ATTRIBUTE_MAP.inverse());

    private static final BiMap<MobileAppDeviceTypeTargeting, DeviceType> DEVICE_TYPE_TARGETING_MAP =
            ImmutableBiMap.of(
                    MobileAppDeviceTypeTargeting.PHONE, DeviceType.PHONE,
                    MobileAppDeviceTypeTargeting.TABLET, DeviceType.TABLET);

    private static final Converter<Collection<MobileAppDeviceTypeTargeting>, Set<DeviceType>>
            DEVICE_TYPE_TARGETING_SET_CONVERTER = immutableSetConverter(DEVICE_TYPE_TARGETING_MAP);

    public static final Converter<Collection<DeviceType>, Set<MobileAppDeviceTypeTargeting>>
            DEVICE_TYPE_TARGETING_SET_TO_CORE_CONVERTER = immutableSetConverter(DEVICE_TYPE_TARGETING_MAP.inverse());

    private static final BiMap<MobileAppNetworkTargeting, Network> NETWORK_TARGETING_MAP =
            ImmutableBiMap.of(
                    MobileAppNetworkTargeting.CELLULAR, Network.CELLULAR,
                    MobileAppNetworkTargeting.WI_FI, Network.WIFI);

    private static final Converter<Collection<MobileAppNetworkTargeting>, Set<Network>>
            NETWORK_TARGETING_SET_CONVERTER = immutableSetConverter(NETWORK_TARGETING_MAP);

    public static final Converter<Collection<Network>, Set<MobileAppNetworkTargeting>>
            NETWORK_TARGETING_SET_TO_CORE_CONVERTER = immutableSetConverter(NETWORK_TARGETING_MAP.inverse());

    public static final Converter<String, String> DOMAIN_CONVERTER = Converter.of(String::trim)
            .andThen(String::toLowerCase)
            .wrapTo(Converters::nullSafeConverter);

    private final TrackerConverter trackerConverter;
    private final MobileContentConverter mobileContentConverter;
    private final SkAdNetworkSlotsConfigProvider skAdNetworkSlotsConfigProvider;

    public WebCoreMobileAppConverter(TrackerConverter trackerConverter,
                                     MobileContentConverter mobileContentConverter,
                                     SkAdNetworkSlotsConfigProvider skAdNetworkSlotsConfigProvider) {
        this.trackerConverter = trackerConverter;
        this.mobileContentConverter = mobileContentConverter;
        this.skAdNetworkSlotsConfigProvider = skAdNetworkSlotsConfigProvider;
    }

    public List<WebMobileApp> convertMobileAppsToWeb(
            List<MobileApp> mobileApps,
            Map<Long, String> publisherDomainById,
            Map<Long, List<Long>> slotCampaignIdsToBundleIds,
            Set<Long> visibleCampaignIds) {
        return StreamEx.of(mobileApps)
                .map(mobileApp -> convertMobileAppToWeb(
                        mobileApp,
                        publisherDomainById.get(mobileApp.getMobileContent().getPublisherDomainId()),
                        slotCampaignIdsToBundleIds.getOrDefault(mobileApp.getId(), emptyList()),
                        visibleCampaignIds))
                .toList();
    }

    public WebMobileApp convertMobileAppToWeb(
            MobileApp mobileApp,
            @Nullable String publisherDomain,
            List<Long> slotCampaignIds,
            Set<Long> visibleCampaignIds) {
        checkNotNull(mobileApp.getMobileContent());

        String name = firstValidName(
                mobileApp.getName(),
                mobileApp.getMobileContent().getName(),
                mobileApp.getMobileContent().getStoreContentId());

        var campaignsWithSkadSlots = slotCampaignIds.stream().filter(visibleCampaignIds::contains).collect(toList());
        return new WebMobileApp()
                .withId(mobileApp.getId())
                .withName(name)
                .withStoreHref(mobileApp.getStoreHref())
                .withStore(checkNotNull(STORE_MAP.get(mobileApp.getStoreType()), "unknown store"))
                .withMobileContentId(mobileApp.getMobileContentId())
                .withMobileContent(
                        mobileContentConverter.createMobileContentInfo(mobileApp.getMobileContent(), publisherDomain))
                .withDomainId(mobileApp.getDomainId())
                .withDomain(mobileApp.getDomain())
                .withPrimaryAction(PRIMARY_ACTION_CONVERTER.convert(mobileApp.getPrimaryAction()))
                .withDisplayedAttributes(DISPLAYED_ATTRIBUTE_SET_CONVERTER.convert(mobileApp.getDisplayedAttributes()))
                .withMinOsVersion(mobileApp.getMinimalOperatingSystemVersion())
                .withDeviceTypeTargeting(
                        DEVICE_TYPE_TARGETING_SET_CONVERTER.convert(mobileApp.getDeviceTypeTargeting()))
                .withNetworkTargeting(NETWORK_TARGETING_SET_CONVERTER.convert(mobileApp.getNetworkTargeting()))
                .withGeoTargeting(mobileContentConverter.getGeoTargeting(mobileApp.getMobileContent()))
                .withTrackers(trackerConverter.convertTrackersToWeb(mobileApp.getTrackers()))
                .withIsAppMetrika(trackerConverter.convertTrackersToIsAppMetrika(mobileApp.getTrackers()))
                .withAppMetrikaApplicationId(mobileApp.getAppMetrikaApplicationId())
                .withMobileEvents(mobileApp.getAppMetrikaApplicationId() == null
                        ? mapList(mobileApp.getMobileExternalTrackerEvents(),
                        this::convertExternalTrackerEventToWeb)
                        : mapList(mobileApp.getMobileAppMetrikaEvents(),
                        this::convertAppmetrikaMobileEventToWeb)
                )
                .withHasVerification(mobileApp.getHasVerification())
                .withSkadNetworkSlotsCount(skAdNetworkSlotsConfigProvider.getConfig().getSlotsNumber())
                .withBusySkadNetworkSlotsCount(slotCampaignIds.size())
                .withCampaignsWithSkadSlots(campaignsWithSkadSlots)
                .withFiles(emptyList());
    }

    private WebMobileEvent convertExternalTrackerEventToWeb(MobileExternalTrackerEvent e) {
        return new WebMobileEvent()
                .withEventName(e.getEventName().name())
                .withIsInternal(true)
                .withCustomEventName(e.getCustomName());
    }

    private WebMobileEvent convertAppmetrikaMobileEventToWeb(MobileAppMetrikaEvent e) {
        return new WebMobileEvent()
                .withEventName(e.getEventName().equals("") ?
                        AppmetrikaInternalEvent.fromTypeAndSubtype(e.getEventType(), e.getEventSubtype(), true).name()
                        : e.getEventName())
                .withIsInternal(e.getEventName().equals(""))
                .withCustomEventName(e.getCustomName());
    }

    public List<MobileApp> convertMobileAppsToCore(ClientId clientId, List<WebMobileApp> webMobileApps) {
        return webMobileApps.stream()
                .map(obj -> convertMobileAppToCore(clientId, obj))
                .collect(toList());
    }

    private MobileApp convertMobileAppToCore(ClientId clientId, WebMobileApp webMobileApp) {
        return new MobileApp()
                .withId(webMobileApp.getId())
                .withClientId(clientId)
                .withName(ifNotNull(webMobileApp.getName(), String::trim))
                .withStoreHref(ifNotNull(webMobileApp.getStoreHref(), String::trim))
                .withDisplayedAttributes(
                        DISPLAYED_ATTRIBUTE_SET_TO_CORE_CONVERTER.convert(webMobileApp.getDisplayedAttributes()))
                .withMinimalOperatingSystemVersion(webMobileApp.getMinOsVersion())
                .withDeviceTypeTargeting(
                        DEVICE_TYPE_TARGETING_SET_TO_CORE_CONVERTER.convert(webMobileApp.getDeviceTypeTargeting()))
                .withNetworkTargeting(
                        NETWORK_TARGETING_SET_TO_CORE_CONVERTER.convert(webMobileApp.getNetworkTargeting()))
                .withTrackers(trackerConverter.convertTrackersToCore(webMobileApp.getTrackers()))
                .withDomain(DOMAIN_CONVERTER.convert(webMobileApp.getDomain()))
                .withAppMetrikaApplicationId(webMobileApp.getAppMetrikaApplicationId())
                .withMobileAppMetrikaEvents(webMobileApp.getAppMetrikaApplicationId() == null
                        ? emptyList()
                        : mapList(webMobileApp.getMobileEvents(),
                        e -> toCoreAppmetrikaEvent(e, webMobileApp.getAppMetrikaApplicationId())))
                .withMobileExternalTrackerEvents(webMobileApp.getAppMetrikaApplicationId() == null
                        ? mapList(webMobileApp.getMobileEvents(),
                        e -> toCoreExternalTrackerEvent(e, webMobileApp.getId()))
                        : emptyList())
                ;
    }

    public static MobileExternalTrackerEvent toCoreExternalTrackerEvent(WebMobileEvent e, Long mobileAppId) {
        return new MobileExternalTrackerEvent()
                .withEventName(ExternalTrackerEventName.valueOf(e.getEventName()))
                .withCustomName(e.getCustomEventName())
                .withMobileAppId(mobileAppId)
                .withIsDeleted(false);
    }

    public static MobileAppMetrikaEvent toCoreAppmetrikaEvent(WebMobileEvent e, Long appMetrikaApplicationId) {
        return new MobileAppMetrikaEvent()
                .withEventType(e.getIsInternal() ? AppmetrikaInternalEvent.valueOf(e.getEventName()).getType()
                        : AppmetrikaEventType.CLIENT)
                .withEventSubtype(e.getIsInternal() ? AppmetrikaInternalEvent.valueOf(e.getEventName()).getSubtype()
                        : AppmetrikaEventSubtype.OTHER)
                .withEventName(e.getIsInternal() ? "" : e.getEventName())
                .withCustomName(e.getCustomEventName())
                .withAppMetrikaAppId(appMetrikaApplicationId)
                .withIsDeleted(false);
    }

    private static String firstValidName(String... options) {
        String lastOption = null;

        for (String option : options) {
            lastOption = option;

            if (option == null) {
                continue;
            }

            // clean up non-BMP characters
            String sanitizedOption = option.replaceAll("[^\\u0000-\\uFFFF]", " ");
            if (!isBlank(sanitizedOption)) {
                return option;
            }

        }

        return lastOption;
    }
}
