package ru.yandex.direct.web.core.entity.mobilecontent.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.EnumUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.mobileapp.model.ExternalTrackerEventName;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.model.SkAdNetworkSlot;
import ru.yandex.direct.core.entity.mobileapp.service.IosSkAdNetworkSlotManager;
import ru.yandex.direct.core.entity.mobileapp.service.MobileAppAddOperation;
import ru.yandex.direct.core.entity.mobileapp.service.MobileAppService;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileContentWithExtraData;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService;
import ru.yandex.direct.core.entity.mobilegoals.model.AppmetrikaInternalEvent;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.entity.mobilecontent.converter.MobileContentConverter;
import ru.yandex.direct.web.core.entity.mobilecontent.converter.WebCoreMobileAppConverter;
import ru.yandex.direct.web.core.entity.mobilecontent.model.ApiMobileContent;
import ru.yandex.direct.web.core.entity.mobilecontent.model.WebMobileApp;
import ru.yandex.direct.web.core.entity.mobilecontent.model.WebMobileContent;
import ru.yandex.direct.web.core.entity.mobilecontent.model.WebMobileEvent;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.core.entity.mobileapp.model.MobileAppStoreType.APPLEAPPSTORE;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Lazy
@Service
public class WebCoreMobileAppService {
    private static final int MAX_MOBILE_CONTENT_ITEMS = 1000;

    private final IosSkAdNetworkSlotManager iosSkAdNetworkSlotManager;
    private final CampaignRepository campaignRepository;
    private final MobileAppService mobileAppService;
    private final MobileContentService mobileContentService;
    private final MobileContentConverter mobileContentConverter;
    private final WebCoreMobileAppConverter webCoreMobileAppConverter;
    private final DomainService domainService;
    private final ShardHelper shardHelper;

    public WebCoreMobileAppService(
            IosSkAdNetworkSlotManager iosSkAdNetworkSlotManager,
            CampaignRepository campaignRepository,
            MobileAppService mobileAppService,
            MobileContentService mobileContentService,
            MobileContentConverter mobileContentConverter,
            WebCoreMobileAppConverter webCoreMobileAppConverter,
            DomainService domainService, ShardHelper shardHelper
    ) {
        this.iosSkAdNetworkSlotManager = iosSkAdNetworkSlotManager;
        this.campaignRepository = campaignRepository;
        this.mobileAppService = mobileAppService;
        this.mobileContentService = mobileContentService;
        this.mobileContentConverter = mobileContentConverter;
        this.webCoreMobileAppConverter = webCoreMobileAppConverter;
        this.domainService = domainService;
        this.shardHelper = shardHelper;
    }

    public List<Long> generateIds(int size) {
        return shardHelper.generateMobileAppIds(size);
    }

    public MassResult<Long> addApps(User operator, ClientId clientId,
                                    List<WebMobileApp> webMobileApps) {
        ValidationResult<List<WebMobileApp>, Defect> requestValidationResult = validateAddRequest(webMobileApps);
        if (requestValidationResult.hasAnyErrors()) {
            return MassResult.brokenMassAction(emptyList(), requestValidationResult);
        }

        List<MobileApp> mobileApps = webCoreMobileAppConverter.convertMobileAppsToCore(clientId, webMobileApps);
        MobileAppAddOperation addOperation = mobileAppService.createAddPartialOperation(operator, clientId, mobileApps);
        return addOperation.prepareAndApply();
    }

    private ValidationResult<List<WebMobileApp>, Defect> validateAddRequest(List<WebMobileApp> webMobileApps) {
        ItemValidationBuilder<List<WebMobileApp>, Defect> vb = ItemValidationBuilder.of(webMobileApps, Defect.class);
        vb.list(webMobileApps, "mobileApps")
                .check(notNull())
                .checkEachBy(this::validateMobileApp);
        return vb.getResult();
    }

    private ValidationResult<WebMobileApp, Defect> validateMobileApp(WebMobileApp mobileApp) {
        ItemValidationBuilder<WebMobileApp, Defect> vb = ItemValidationBuilder.of(mobileApp, Defect.class);
        vb.list(mobileApp.getMobileEvents(), "mobileEvents")
                .checkEachBy(event -> validateEvent(event, mobileApp.getAppMetrikaApplicationId() != null),
                        When.notNull());
        return vb.getResult();
    }

    public static ValidationResult<WebMobileEvent, Defect> validateEvent(WebMobileEvent event, boolean isAppmetrika) {
        ItemValidationBuilder<WebMobileEvent, Defect> vb = ItemValidationBuilder.of(event, Defect.class);
        vb.item(event.getEventName(), "eventName")
                .check(Constraint.fromPredicate(name -> EnumUtils.isValidEnum(AppmetrikaInternalEvent.class, name),
                        invalidValue()), When.isTrue(isAppmetrika && event.getIsInternal()))
                .check(Constraint.fromPredicate(name -> EnumUtils.isValidEnum(ExternalTrackerEventName.class, name),
                        invalidValue()), When.isFalse(isAppmetrika));
        return vb.getResult();
    }

    public Optional<WebMobileApp> getApp(ClientId clientId, Long mobileAppId) {
        Optional<MobileApp> mobileApp = mobileAppService.getMobileApp(clientId, mobileAppId);
        String publisherDomain = mobileApp
                .map(MobileApp::getMobileContent)
                .map(MobileContent::getPublisherDomainId)
                .map(this::getDomainById)
                .orElse(null);

        if (mobileApp.isPresent()
                && mobileApp.get().getMobileContent() != null
                && mobileApp.get().getMobileContent().getBundleId() != null) {
            var slotsInfoToAppIds = getSkadNetworkSlotCampaignIdsForAppIds(List.of(mobileApp.get()));
            var visibleCampaignIds = getVisibleCampaignIds(clientId, slotsInfoToAppIds);
            return mobileApp.map(ma -> webCoreMobileAppConverter.convertMobileAppToWeb(
                    ma,
                    publisherDomain,
                    slotsInfoToAppIds.getOrDefault(mobileApp.get().getId(), emptyList()),
                    visibleCampaignIds));
        }

        return mobileApp.map(ma -> webCoreMobileAppConverter.convertMobileAppToWeb(
                ma, publisherDomain, emptyList(), emptySet()));
    }

    public List<WebMobileApp> getAppList(ClientId clientId, @Nullable String storeContentId,
                                         @Nullable String storeCountry) {
        List<MobileApp> mobileApps = mobileAppService.getMobileApps(clientId);
        if (storeContentId != null) {
            mobileApps = filterList(mobileApps,
                    app -> storeContentId.equals(app.getMobileContent().getStoreContentId()));
        }
        if (storeCountry != null) {
            mobileApps = filterList(mobileApps,
                    app -> storeCountry.equals(app.getMobileContent().getStoreCountry()));
        }
        Map<Long, String> publisherDomainById = getDomainByIdMappingForMobileApps(
                mobileApps, MobileApp::getMobileContent);

        var slotCampaignIdsToAppIds = getSkadNetworkSlotCampaignIdsForAppIds(mobileApps);
        var visibleCampaignIds = getVisibleCampaignIds(clientId, slotCampaignIdsToAppIds);

        return webCoreMobileAppConverter.convertMobileAppsToWeb(
                mobileApps, publisherDomainById, slotCampaignIdsToAppIds, visibleCampaignIds);
    }

    public List<WebMobileApp> getOldAppList(ClientId clientId) {
        List<MobileContentWithExtraData> mobileContentApps = mobileContentService.getStoreHrefWithMobileContent(
                clientId, LimitOffset.limited(MAX_MOBILE_CONTENT_ITEMS));
        Map<Long, String> publisherDomainById = getDomainByIdMappingForMobileApps(
                mobileContentApps, MobileContentWithExtraData::getMobileContent);
        return StreamEx.of(mobileContentApps)
                .mapToEntry(mca -> mca.getMobileContent().getPublisherDomainId())
                .mapValues(publisherDomainById::get)
                .mapKeyValue(mobileContentConverter::createWebMobileApp)
                .collect(toList());
    }

    public Optional<WebMobileContent> getMobileContent(ClientId clientId, String storeUrl,
                                                       MobileAppStoreUrl parsedStoreUrl, boolean dryRun) {
        Optional<MobileContent> result = mobileContentService.getMobileContent(
                clientId, storeUrl, parsedStoreUrl, dryRun, true);
        String publisherDomain = result.map(MobileContent::getPublisherDomainId)
                .map(this::getDomainById)
                .orElse(null);
        return result.map(mc -> mobileContentConverter.createMobileContentInfo(mc, publisherDomain));
    }

    public List<ApiMobileContent> listApiMobileContent(Map<String, List<String>> storeToIds) {
        var apiContent = new ArrayList<ApiMobileContent>();
        storeToIds.forEach((store, appIds) -> {
            for (var subList : Iterables.partition(appIds, MAX_MOBILE_CONTENT_ITEMS)) {
                var apiMobileContentList = mobileContentService.getApiMobileContentFromYt(store, subList);
                for (var apiMobileContent : apiMobileContentList) {
                    apiContent.add(mobileContentConverter.createApiMobileContent(apiMobileContent));
                }
            }
        });

        return apiContent;
    }

    private <T> Map<Long, String> getDomainByIdMappingForMobileApps(List<T> objects,
                                                                    Function<T, MobileContent> extractMobileContent) {
        Set<Long> publisherDomainIds = StreamEx.of(objects)
                .map(extractMobileContent)
                .map(MobileContent::getPublisherDomainId)
                .nonNull()
                .toSet();
        return getDomainByIdMapping(publisherDomainIds);
    }

    @Nullable
    private String getDomainById(Long publisherDomainId) {
        return Iterables.getFirst(getDomainByIdMapping(Collections.singleton(publisherDomainId)).values(), null);
    }

    /**
     * Получить мапу идентификатор_домена -> домен для коллекции идентификаторов.
     *
     * @param publisherDomainIds коллекция идентификторов доменов
     */
    private Map<Long, String> getDomainByIdMapping(Collection<Long> publisherDomainIds) {
        List<Domain> publisherDomains = domainService.getDomainsByIdsFromDict(publisherDomainIds);
        return StreamEx.of(publisherDomains)
                .mapToEntry(Domain::getId, Domain::getDomain)
                .toMap();
    }

    private Map<Long, List<Long>> getSkadNetworkSlotCampaignIdsForAppIds(List<MobileApp> mobileApps) {
        if (mobileApps.isEmpty()) {
            return emptyMap();
        }

        var bundleIds = mobileApps.stream()
                .filter(e -> e.getMobileContent() != null
                        && e.getMobileContent().getBundleId() != null
                        && APPLEAPPSTORE.equals(e.getStoreType()))
                .map(e -> e.getMobileContent().getBundleId())
                .collect(toList());

        var campaignsWithSlots = iosSkAdNetworkSlotManager.getAllocatedSlotsByBundleIds(bundleIds);
        var campaignsByBundleId = campaignsWithSlots.stream()
                .collect(groupingBy(
                        SkAdNetworkSlot::getAppBundleId,
                        mapping(SkAdNetworkSlot::getCampaignId, toList())));

        return mobileApps.stream()
                .collect(toMap(MobileApp::getId,
                        app -> {
                            if (app.getMobileContent() != null
                                    && app.getMobileContent().getBundleId() != null) {
                                return campaignsByBundleId.getOrDefault(
                                        app.getMobileContent().getBundleId(), emptyList());
                            }
                            return emptyList();
                        }));
    }

    private Set<Long> getVisibleCampaignIds(ClientId clientId, Map<Long, List<Long>> slotsInfoToAppIds) {
        var campaignIds = slotsInfoToAppIds.values().stream().flatMap(Collection::stream).collect(toList());
        var shard = shardHelper.getShardByClientIdStrictly(clientId);
        return listToSet(campaignRepository.getCampaignsForClient(shard, clientId, campaignIds), CampaignSimple::getId);
    }
}
