package ru.yandex.direct.web.core.entity.placement.service;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.placements.model1.Codable;
import ru.yandex.direct.core.entity.placements.model1.GeoBlock;
import ru.yandex.direct.core.entity.placements.model1.IndoorBlock;
import ru.yandex.direct.core.entity.placements.model1.IndoorFacilityType;
import ru.yandex.direct.core.entity.placements.model1.OutdoorBlock;
import ru.yandex.direct.core.entity.placements.model1.OutdoorFacilityType;
import ru.yandex.direct.core.entity.placements.model1.Placement;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementType;
import ru.yandex.direct.core.entity.placements.model1.PlacementsFilter;
import ru.yandex.direct.core.entity.placements.model1.ZoneCategory;
import ru.yandex.direct.core.entity.placements.repository.PlacementBlockRepository;
import ru.yandex.direct.core.entity.placements.repository.PlacementRepository;
import ru.yandex.direct.core.entity.placements.service.GeoBlockAddressTranslatable;
import ru.yandex.direct.core.entity.placements.service.PlacementsTranslateService;
import ru.yandex.direct.core.entity.placements.service.TranslateFacilityTypeIndoorService;
import ru.yandex.direct.core.entity.placements.service.TranslateFacilityTypeOutdoorService;
import ru.yandex.direct.core.entity.placements.service.TranslateZoneCategoryService;
import ru.yandex.direct.core.entity.region.RegionDesc;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.web.core.entity.placement.model.PlacementsResponse;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * ПИ может независимо от нас завести нового оператора, тип торговой точки, тип рекламного щита,
 * тип зоны размещения и не сказать нам. Для этого есть мониторинг, а здесь мы фильтруем невалидные
 * данные и не отдаем на фронт.
 */
@Service
public class PlacementsService {
    private static final int ENRICH_PLACEMENT_MAX_RESULT = 20;//Ограничение на результат в поиске по домену
    private static final String WWW_PREFIX = "www.";

    private final PlacementRepository placementRepository;
    private final PlacementBlockRepository placementBlockRepository;
    private final TranslateFacilityTypeIndoorService translateFacilityTypeIndoorService;
    private final TranslateFacilityTypeOutdoorService translateFacilityTypeOutdoorService;
    private final TranslateZoneCategoryService translateZoneCategoryService;
    private final TranslationService translationService;
    private final FeatureService featureService;
    private final GeoTreeFactory geoTreeFactory;

    @Autowired
    public PlacementsService(PlacementRepository placementRepository,
                             PlacementBlockRepository placementBlockRepository,
                             TranslateFacilityTypeIndoorService translateFacilityTypeIndoorService,
                             TranslateFacilityTypeOutdoorService translateFacilityTypeOutdoorService,
                             TranslateZoneCategoryService translateZoneCategoryService,
                             TranslationService translationService,
                             FeatureService featureService,
                             GeoTreeFactory geoTreeFactory) {
        this.placementRepository = placementRepository;
        this.placementBlockRepository = placementBlockRepository;
        this.translateFacilityTypeIndoorService = translateFacilityTypeIndoorService;
        this.translateFacilityTypeOutdoorService = translateFacilityTypeOutdoorService;
        this.translateZoneCategoryService = translateZoneCategoryService;
        this.translationService = translationService;
        this.featureService = featureService;
        this.geoTreeFactory = geoTreeFactory;
    }

    public PlacementsResponse getPlacements(PlacementsFilter placementsFilter, ClientId clientId) {
        PlacementType placementType = placementsFilter.getPlacementType();
        checkArgument(placementsFilter.getPlacementType() != null, "placement type is required parameter");

        List<PlacementBlock> blocks;
        Map<Integer, String> facilityDictionary;
        Map<Integer, String> zoneDictionary = null;
        boolean testingPagesEnabled =
                featureService.isEnabledForClientId(clientId, FeatureName.OUTDOOR_INDOOR_TESTING_PAGES);

        switch (placementType) {
            case INDOOR:
                @SuppressWarnings("unchecked")
                List<IndoorBlock> indoorBlocks =
                        (List<IndoorBlock>) placementBlockRepository.getBlocksByFilter(placementsFilter);
                facilityDictionary = createDictionary(translateFacilityTypeIndoorService, IndoorFacilityType.class);
                zoneDictionary = createDictionary(translateZoneCategoryService, ZoneCategory.class);
                indoorBlocks = filterInconsistentIndoorBlocks(indoorBlocks, facilityDictionary.keySet(),
                        zoneDictionary.keySet());
                blocks = new ArrayList<>(indoorBlocks);
                break;
            case OUTDOOR:
                @SuppressWarnings("unchecked")
                List<OutdoorBlock> outdoorBlocks =
                        (List<OutdoorBlock>) placementBlockRepository.getBlocksByFilter(placementsFilter);
                facilityDictionary = createDictionary(translateFacilityTypeOutdoorService, OutdoorFacilityType.class);
                outdoorBlocks = filterInconsistentOutdoorBlocks(outdoorBlocks, facilityDictionary.keySet());
                blocks = new ArrayList<>(outdoorBlocks);
                break;
            default:
                throw new UnsupportedOperationException("PlacementType not supported");
        }

        Map<Long, Placement> placements;
        if (!blocks.isEmpty()) {
            Set<Long> pageIds = listToSet(blocks, PlacementBlock::getPageId);
            Map<Long, Placement> allPlacements = placementRepository.getPlacements(pageIds);
            placements = testingPagesEnabled
                    ? allPlacements
                    : EntryStream.of(allPlacements).removeValues(Placement::isTesting).toMap();
        } else {
            placements = new HashMap<>();
        }

        blocks = filterOutdoorBlocksWithUnknownOperator(blocks, placements);
        if (!testingPagesEnabled) {
            Set<Long> visiblePages = placements.keySet();
            blocks = filterList(blocks, block -> visiblePages.contains(block.getPageId()));
        }

        blocks = setGeoBlocksAddressTranslations(blocks);
        placements = setGeoBlocksAddressTranslationsInPlacements(placements);

        return new PlacementsResponse()
                .withPlacementBlocks(blocks)
                .withPlacements(new ArrayList<>(placements.values()))
                .withFacilityDictionary(facilityDictionary)
                .withZoneDictionary(zoneDictionary)
                .withRegionDictionary(getUniqueRegionsFromBlocks(blocks));
    }

    private List<OutdoorBlock> filterInconsistentOutdoorBlocks(List<OutdoorBlock> outdoorBlocks,
                                                               Set<Integer> knownFacilities) {
        Predicate<OutdoorBlock> validFacilityTypePredicate = block -> {
            Integer facilityType = block.getFacilityType();
            return facilityType != null && knownFacilities.contains(facilityType);
        };
        return StreamEx.of(outdoorBlocks)
                .select(OutdoorBlock.class)
                .filter(validFacilityTypePredicate)
                .toList();
    }

    private List<IndoorBlock> filterInconsistentIndoorBlocks(List<IndoorBlock> indoorBlocks,
                                                             Set<Integer> knownFacilities, Set<Integer> knownZones) {
        Predicate<IndoorBlock> validFacilityTypePredicate = block -> {
            Integer facilityType = block.getFacilityType();
            return facilityType != null && knownFacilities.contains(facilityType);
        };
        Predicate<IndoorBlock> validZoneCategoryPredicate = block -> {
            Integer zoneCategory = block.getZoneCategory();
            return zoneCategory != null && knownZones.contains(zoneCategory);
        };
        return StreamEx.of(indoorBlocks)
                .select(IndoorBlock.class)
                .filter(validFacilityTypePredicate)
                .filter(validZoneCategoryPredicate)
                .toList();
    }

    private List<PlacementBlock> filterOutdoorBlocksWithUnknownOperator(List<PlacementBlock> blocks,
                                                                        Map<Long, Placement> placementMap) {
        Set<Long> placementIdsWithoutOperatorName = StreamEx.of(placementMap.values())
                .remove(placement -> placement.getOperatorName() != null)
                .map(Placement::getId)
                .toSet();
        return StreamEx.of(blocks)
                .remove(block -> block instanceof OutdoorBlock &&
                        placementIdsWithoutOperatorName.contains(block.getPageId()))
                .toList();
    }

    private Map<Long, Placement> setGeoBlocksAddressTranslationsInPlacements(Map<Long, Placement> placements) {
        return EntryStream.of(placements)
                .mapValues(placement -> placement.replaceBlocks(setGeoBlocksAddressTranslations(placement.getBlocks())))
                .toMap();
    }

    private List<PlacementBlock> setGeoBlocksAddressTranslations(List<PlacementBlock> blocks) {
        return StreamEx.of(blocks)
                .map(block -> {
                    if (!(block instanceof GeoBlock)) {
                        return block;
                    }
                    GeoBlock<?> geoBlock = (GeoBlock<?>) block;

                    String addressTranslation = translationService.translate(new GeoBlockAddressTranslatable(geoBlock));
                    return (PlacementBlock) geoBlock.withAddress(addressTranslation);
                })
                .toList();
    }

    private static <E extends Enum<E> & Codable> Map<Integer, String> createDictionary(
            PlacementsTranslateService<E> translateService, Class<E> elementType) {
        Map<Integer, String> dictionary = new HashMap<>();
        Set<E> allTypes = EnumSet.allOf(elementType);
        for (E type : allTypes) {
            dictionary.put(type.getCode(), translateService.getTranslation(type));
        }
        return dictionary;
    }

    private Map<Long, RegionDesc> getUniqueRegionsFromBlocks(List<PlacementBlock> blocks) {
        return StreamEx.of(blocks)
                .select(GeoBlock.class)
                .map(GeoBlock::getGeoId)
                .distinct()
                .mapToEntry(getGeoTree()::getRegion)
                .filterValues(Objects::nonNull)
                .mapValues(RegionDesc::fromRegion)
                .toMap();
    }

    public List<Placement> findPlacements(String filterString, ClientId clientId) {
        //clientId пробросил сразу, так как предполагаю фильтры по фичам
        if (filterString == null) {
            filterString = "";
        }
        if (filterString.startsWith(WWW_PREFIX)) {
            filterString = filterString.substring(WWW_PREFIX.length());
        }
        return placementRepository.findPlacementsByDomain(filterString, ENRICH_PLACEMENT_MAX_RESULT);
    }

    public List<Placement> getPlacementsByIds(List<Long> ids) {
        return placementRepository.getPlacementsByIds(ids);
    }

    private GeoTree getGeoTree() {
        return geoTreeFactory.getGlobalGeoTree();
    }
}
