package ru.yandex.direct.web.core.exception;

import java.io.IOException;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.http.entity.ContentType;
import org.eclipse.jetty.io.EofException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.filter.OncePerRequestFilter;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.TranslatableException;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.web.auth.blackbox.PassportUrls;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.security.WebCoreTranslations;
import ru.yandex.direct.web.core.security.authentication.exception.BlackboxCredentialsExpiredException;
import ru.yandex.direct.web.core.security.authentication.exception.RecoverableAuthenticationException;
import ru.yandex.direct.web.core.security.authentication.exception.TranslatableAuthenticationException;
import ru.yandex.direct.web.core.security.configuration.BlackboxWebAuthenticationConfiguration;
import ru.yandex.misc.io.http.HttpHeaderNames;

import static ru.yandex.direct.web.core.exception.ExceptionUtils.isAjaxRequest;
import static ru.yandex.direct.web.core.exception.ExceptionUtils.toErrorResponse;
import static ru.yandex.misc.io.http.HttpStatus.SC_499_CLIENT_CLOSED_REQUEST;

/**
 * В этом фильтре отлавливаются исключения из других фильтров, таких как
 * {@link BlackboxWebAuthenticationConfiguration#blackBoxWebAuthenticationFilter}
 * и в ответ записывается сообщение вида {@link WebErrorResponse} и соответствующий http статус
 */
@Component
public class DirectWebExceptionFilter extends OncePerRequestFilter {

    private static final Logger logger = LoggerFactory.getLogger(DirectWebExceptionFilter.class);

    private static final String CONTENT_TYPE_JSON = ContentType.APPLICATION_JSON.getMimeType();
    private static final String CONTENT_TYPE_HTML = ContentType.TEXT_HTML.getMimeType();
    private final String defaultEncoding;

    private final TranslationService translationService;

    @Autowired
    public DirectWebExceptionFilter(TranslationService translationService,
                                    @Value("${multipart_config.default_encoding}") String defaultEncoding) {
        this.translationService = translationService;
        this.defaultEncoding = defaultEncoding;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain)
            throws ServletException, IOException {
        try {
            try {
                filterChain.doFilter(request, response);
            } catch (BlackboxCredentialsExpiredException e) {
                logger.info("Необходимо обновить сессию: " + e.getMessage());
                response.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
                String authUrl = PassportUrls.AUTH_UPDATE.build(request);
                RecoverableBlackboxErrorResponse error = new RecoverableBlackboxErrorResponse("need_reset",
                        translationService.translate(e.getShortMessage()), authUrl);
                writeResponse(response, error, request);
            } catch (RecoverableAuthenticationException e) {
                logger.info("Неверные аутентификационные данные: " + e.getMessage());
                response.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
                String authUrl = PassportUrls.AUTH.build(request);
                RecoverableBlackboxErrorResponse error = new RecoverableBlackboxErrorResponse("need_auth",
                        translationService.translate(e.getShortMessage()), authUrl);
                writeResponse(response, error, request);
            } catch (TranslatableAuthenticationException e) {
                logger.warn("Ошибка аутентификации", e);
                response.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
                writeResponse(response, toErrorResponse(e, translationService), request);
            } catch (TranslatableException e) {
                logger.warn("Ошибка выполнения операции", e);
                response.setStatus(HttpServletResponse.SC_BAD_REQUEST);
                writeResponse(response, toErrorResponse(e, translationService), request);
            } catch (EofException e) {
                logger.info("Клиент разорвал соединение", e);
                response.setStatus(SC_499_CLIENT_CLOSED_REQUEST);
            }
        } catch (Exception e) {
            logger.error("Внутренняя ошибка сервера", e);
            response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
            writeResponse(response, new WebErrorResponse(500, translationService.translate(
                    WebCoreTranslations.INSTANCE.serviceUnavailable())), request);
        }
    }

    private void writeResponseJson(HttpServletResponse response, WebErrorResponse responseBody) {
        try {
            response.setContentType(CONTENT_TYPE_JSON);
            response.setCharacterEncoding(defaultEncoding);
            response.getWriter().write(JsonUtils.toJson(responseBody));
        } catch (Exception e) {
            response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        }
    }

    private void writeResponseHtml(HttpServletResponse response, WebErrorResponse responseBody) {
        try {
            response.setContentType(CONTENT_TYPE_HTML);
            response.setCharacterEncoding(defaultEncoding);

            if (responseBody instanceof RecoverableBlackboxErrorResponse) {
                RecoverableBlackboxErrorResponse blackboxErrorResponse = (RecoverableBlackboxErrorResponse) responseBody;
                String recoveryUrl = blackboxErrorResponse.getRecoveryUrl();
                response.addHeader(HttpHeaderNames.LOCATION, recoveryUrl);
                response.setStatus(HttpServletResponse.SC_MOVED_TEMPORARILY);
            }

            response.getWriter().write(responseBody.getText());
        } catch (Exception e) {
            response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        }
    }

    protected void writeResponse(HttpServletResponse response, WebErrorResponse responseBody, HttpServletRequest request) {
        if (isAjaxRequest(request)) {
            writeResponseJson(response, responseBody);
        } else {
            writeResponseHtml(response, responseBody);
        }
    }
}
