package ru.yandex.direct.web.core.exception;

import java.io.IOException;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.fasterxml.jackson.databind.JsonMappingException;
import org.apache.commons.fileupload.FileUploadBase;
import org.apache.http.entity.ContentType;
import org.eclipse.jetty.io.EofException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.multipart.MaxUploadSizeExceededException;
import org.springframework.web.servlet.HandlerInterceptor;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.TranslatableException;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.security.captcha.CaptchaFirewallInterceptor;
import ru.yandex.direct.web.core.security.csrf.CsrfInterceptor;
import ru.yandex.direct.web.core.security.csrf.CsrfValidationFailureException;

import static ru.yandex.direct.utils.JsonUtils.toJson;
import static ru.yandex.direct.web.core.exception.ExceptionUtils.isAjaxRequest;
import static ru.yandex.direct.web.core.exception.ExceptionUtils.toErrorResponse;

/**
 * Тут обрататываются исключения, возникающие в контроллерах, а так же в {@link HandlerInterceptor}, например,
 * {@link CaptchaFirewallInterceptor} и {@link CsrfInterceptor}
 */
@ControllerAdvice
public class DirectWebExceptionHandlerController {
    private static final Logger logger = LoggerFactory.getLogger(DirectWebExceptionHandlerController.class);
    private static final String CONTENT_TYPE_JSON = ContentType.APPLICATION_JSON.getMimeType();
    private static final String CONTENT_TYPE_HTML = ContentType.TEXT_HTML.getMimeType();

    private final TranslationService translationService;
    private final String defaultEncoding;

    @Autowired
    public DirectWebExceptionHandlerController(TranslationService translationService,
                                               @Value("${multipart_config.default_encoding}") String defaultEncoding) {
        this.defaultEncoding = defaultEncoding;
        this.translationService = translationService;
    }

    //MethodArgumentTypeMismatchException - если не смогли сериализовать параметр во внутреннее представление -
    // к примеру, ошибка синтаксиса в Json параметре
    @ExceptionHandler({
            HttpMediaTypeNotSupportedException.class,
            MethodArgumentTypeMismatchException.class,
            WebBadRequestException.class
    })
    @ResponseBody
    public ResponseEntity<String> badRequestExceptionHandler(Exception exception) {
        logger.error("BadRequestExceptionHandler", exception);
        return new ResponseEntity<>(HttpStatus.BAD_REQUEST);
    }

    @ExceptionHandler({
            CsrfValidationFailureException.class
    })
    @ResponseBody
    public ResponseEntity<WebErrorResponse> csrfValidationFailureHandler(CsrfValidationFailureException exception) {
        logger.info("CsrfValidationFailureHandler", exception);
        return new ResponseEntity<>(toErrorResponse(exception, translationService), HttpStatus.FORBIDDEN);
    }

    @ExceptionHandler(RateLimitExceededException.class)
    public void rateLimitExceededExceptionHandler(HttpServletRequest request, HttpServletResponse response,
                                                  RateLimitExceededException exception) throws IOException {
        logger.info("RateLimitExceededExceptionHandler", exception);
        handleTranslatableException(request, response, exception);
    }

    @ExceptionHandler(TranslatableException.class)
    public void translatableExceptionHandler(HttpServletRequest request, HttpServletResponse response,
                                             TranslatableException exception) throws IOException {
        logger.error("TranslatableExceptionHandler", exception);
        handleTranslatableException(request, response, exception);
    }

    private void handleTranslatableException(HttpServletRequest request, HttpServletResponse response,
                                             TranslatableException exception) throws IOException {
        response.setStatus(HttpStatus.OK.value());
        response.setCharacterEncoding(defaultEncoding);
        if (isAjaxRequest(request)) {
            response.setContentType(CONTENT_TYPE_JSON);
            response.getWriter().write(toJson(toErrorResponse(exception, translationService)));
        } else {
            response.setContentType(CONTENT_TYPE_HTML);
            response.getWriter().write(toErrorResponse(exception, translationService).getText());
        }
    }

    @ExceptionHandler(HttpMessageNotReadableException.class)
    @ResponseBody
    public ResponseEntity<String> jsonMappingErrorHandler(HttpMessageNotReadableException exception) {
        logger.error("jsonMappingErrorHandler", exception);
        Throwable cause = exception.getCause();
        if (cause != null && cause instanceof JsonMappingException) {
            JsonMappingException jsonMappingException = (JsonMappingException) cause;
            WebErrorResponse response = new WebErrorResponse("",
                    JsonUtils.extractJsonMappingErrorLocation(jsonMappingException));
            return new ResponseEntity<>(toJson(response), HttpStatus.BAD_REQUEST);
        }
        return new ResponseEntity<>(HttpStatus.BAD_REQUEST);
    }

    @ExceptionHandler(MaxUploadSizeExceededException.class)
    @ResponseBody
    public ResponseEntity<String> maxUploadSizeExceededErrorHandler(MaxUploadSizeExceededException exception) {
        logger.error("maxUploadSizeExceededErrorHandler", exception);
        Throwable cause = exception.getCause();
        if (cause != null && cause instanceof FileUploadBase.SizeLimitExceededException) {
            WebErrorResponse response = new WebErrorResponse("", cause.getMessage());
            return new ResponseEntity<>(toJson(response), HttpStatus.BAD_REQUEST);
        }
        return new ResponseEntity<>(HttpStatus.BAD_REQUEST);
    }

    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    @ResponseBody
    public ResponseEntity<String> httpRequestMethodNotSupportedException(Exception e) {
        logger.error("httpRequestMethodNotSupportedException", e);
        return new ResponseEntity<>(HttpStatus.METHOD_NOT_ALLOWED);
    }

    @ExceptionHandler(AccessDeniedException.class)
    @ResponseBody
    public ResponseEntity<String> accessDeniedErrorHandler(Exception e) {
        logger.error("accessDeniedErrorHandler", e);
        return new ResponseEntity<>(HttpStatus.FORBIDDEN);
    }

    @ExceptionHandler(EofException.class)
    @ResponseBody
    public ResponseEntity<String> eofErrorHandler(Exception e) {
        // Jetty EOF -- это QuietException, который обозначает разрыв соединения с клиентской стороны. Логируем как INFO
        logger.info("eofException", e);
        return ResponseEntity.status(ru.yandex.misc.io.http.HttpStatus.SC_499_CLIENT_CLOSED_REQUEST).build();
    }

    @ExceptionHandler
    @ResponseBody
    public ResponseEntity<String> defaultWebErrorHandler(Exception e) {
        logger.error("DefaultWebErrorHandler", e);
        return new ResponseEntity<>(HttpStatus.INTERNAL_SERVER_ERROR);
    }
}
