package ru.yandex.direct.web.core.security;

import java.io.IOException;
import java.util.Locale;
import java.util.Set;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import org.springframework.http.HttpHeaders;
import org.springframework.web.filter.OncePerRequestFilter;

import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.utils.FunctionalUtils;

import static ru.yandex.direct.web.core.WebLocaleResolver.DEFAULT_LOCALE;
import static ru.yandex.direct.web.core.WebLocaleResolver.DETECTED_LOCALE_HEADER_NAME;

/**
 * Locale-resolver для backend'а, работающего с web.
 * <p>
 * При работе с браузером из 'внешнего' мира (т. е. с клиентом Яндекса) локаль определяется на основе нескольких характеристик:
 * <ul>
 * <li>
 * Top level domain(TLD): com/ru/ua/com.tr
 * </li>
 * <li>
 * Параметр 'Accept-Language' Http-запроса
 * </li>
 * <li>
 * Географическое положение пользователя
 * </li>
 * <li>
 * Пользовательские настройки: выставляются запросом вида "https://www.yandex.ru/portal/set/lang/?intl=ru&retpath=ret_url",
 * либо на странице "https://yandex.ru/tune/lang"; сохраняются в cookie {@code my}
 * </li>
 * </ul>
 * Часть этой логики инкапсулирована в С++-библиотеке lang-detect; Java-binding на момент написания кода остутствует.
 * Поэтому логика определения языка вынесена в nginx+perl; он выставляет HTTP-заголовок {@link #DETECTED_LOCALE_HEADER_NAME}
 *
 * @see <a href="http://doc.yandex-team.ru/lib/lang-detect/">lang-detect</a>
 */
public class WebLocaleResolverFilter extends OncePerRequestFilter {


    private final Locale defaultLocale;
    private final ImmutableSet<Locale> supportedLocales;

    public WebLocaleResolverFilter(Locale defaultLocale, Set<Locale> supportedLocales) {
        Preconditions.checkArgument(supportedLocales.contains(defaultLocale));
        this.defaultLocale = defaultLocale;
        this.supportedLocales = ImmutableSet.copyOf(supportedLocales);
    }

    public WebLocaleResolverFilter() {
        this(DEFAULT_LOCALE, supportedLocales());
    }

    private static Set<Locale> supportedLocales() {
        return ImmutableSet.copyOf(FunctionalUtils.mapList(Language.getValueSet(),
                language -> new Locale(language.getLangString())));
    }

    public void resolveLocaleAndSaveInRequest(HttpServletRequest request) {
        Locale localeLanguage = defaultLocale;
        String detectedLocale = request.getHeader(DETECTED_LOCALE_HEADER_NAME);
        if (detectedLocale != null) {
            Locale locale = new Locale(detectedLocale);
            localeLanguage = new Locale(locale.getLanguage());
        } else {
            // проверяем заголовок Accept-Language
            String acceptLanguageHeader = request.getHeader(HttpHeaders.ACCEPT_LANGUAGE);
            if (acceptLanguageHeader != null) {
                try {
                    //Иногда присылают язык "en_US" вместо "en-us". Мы хотим уметь обрабатывать такие случаи
                    var list = Locale.LanguageRange.parse(acceptLanguageHeader.replace('_', '-'));
                    localeLanguage = Locale.lookup(list, supportedLocales);
                } catch (IllegalArgumentException e) {
                    localeLanguage = defaultLocale;
                }
            }
        }
        if (!supportedLocales.contains(localeLanguage)) {
            localeLanguage = defaultLocale;
        }
        request.setAttribute(DETECTED_LOCALE_HEADER_NAME, localeLanguage);
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain)
            throws ServletException, IOException {
        resolveLocaleAndSaveInRequest(request);
        filterChain.doFilter(request, response);
    }
}
