package ru.yandex.direct.web.core.security.authentication;

import java.io.IOException;
import java.net.InetAddress;
import java.util.Optional;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.CredentialsExpiredException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.tracing.util.TraceCommentVarsHolder;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.web.auth.blackbox.BlackboxCookieAuthRequest;
import ru.yandex.direct.web.auth.blackbox.BlackboxCookieCredentials;
import ru.yandex.direct.web.auth.blackbox.BlackboxCookieFakeAuth;
import ru.yandex.direct.web.auth.blackbox.exception.BadBlackboxCredentialsException;
import ru.yandex.direct.web.core.security.authentication.exception.BlackboxBadCredentialsException;
import ru.yandex.direct.web.core.security.authentication.exception.BlackboxCredentialsExpiredException;
import ru.yandex.direct.web.core.security.authentication.exception.DirectAuthenticationServiceException;
import ru.yandex.direct.web.core.security.authentication.exception.TranslatableAuthenticationException;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.web.core.security.authentication.AuthenticationErrorCodes.AUTHENTICATION_SERVICE_ERROR;
import static ru.yandex.direct.web.core.security.authentication.AuthenticationErrorCodes.BAD_CREDENTIALS;
import static ru.yandex.direct.web.core.security.authentication.AuthenticationErrorCodes.CREDENTIALS_EXPIRED;
import static ru.yandex.direct.web.core.security.authentication.WebAuthTranslations.WEB_AUTH_TRANSLATIONS;

/**
 * This filter obtains CookieCredentials from request and calls <code>AuthenticationManager</code>.
 * Authentication <code>AuthenticationManager</code> must support <code>BlackboxCookieAuthRequest</code>.
 * TODO(zhur)
 */
public class BlackboxCookieAuthFilter implements Filter {

    private static final Logger logger = LoggerFactory.getLogger(BlackboxCookieAuthFilter.class);
    private static final String FAKE_LOGIN_HEADER_NAME = "X-Fake-Operator-Login";

    private final AuthenticationManager authenticationManager;
    private final EnvironmentType environmentType;
    private final NetAcl netAcl;
    private final UserService userService;
    private final TvmIntegration tvmIntegration;
    private final TvmService tvmService;

    public BlackboxCookieAuthFilter(
            EnvironmentType environmentType,
            NetAcl netAcl,
            UserService userService,
            TvmIntegration tvmIntegration,
            TvmService tvmService,
            AuthenticationManager authenticationManager) {
        checkNotNull(authenticationManager, "authenticationManager is required");
        this.environmentType = environmentType;
        this.netAcl = netAcl;
        this.userService = userService;
        this.authenticationManager = authenticationManager;
        this.tvmIntegration = tvmIntegration;
        this.tvmService = tvmService;
    }

    @Override
    public void init(FilterConfig filterConfig) {
    }

    @Override
    public void destroy() {
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response,
                         FilterChain chain) throws IOException, ServletException {

        Authentication authentication;
        try {
            String tvmTicket = tvmIntegration.getTicket(tvmService);
            authentication = authenticate(request, tvmTicket);

        } catch (BadCredentialsException | BadBlackboxCredentialsException e) {
            throw new BlackboxBadCredentialsException("Bad credentials", e, BAD_CREDENTIALS,
                    WEB_AUTH_TRANSLATIONS.badCredentials(), null);
        } catch (CredentialsExpiredException e) {
            throw new BlackboxCredentialsExpiredException("Credentials has expired", e, CREDENTIALS_EXPIRED,
                    WEB_AUTH_TRANSLATIONS.credentialsHasExpired(), null);
        } catch (AuthenticationServiceException e) {
            throw new DirectAuthenticationServiceException(e, AUTHENTICATION_SERVICE_ERROR,
                    WEB_AUTH_TRANSLATIONS.authenticationServiceUnavailable(), null);
        } catch (TranslatableAuthenticationException e) {
            throw e;
        } catch (Exception e) {
            throw new DirectAuthenticationServiceException(e, AUTHENTICATION_SERVICE_ERROR,
                    WEB_AUTH_TRANSLATIONS.authenticationServiceUnavailable(), null);
        }
        if (authentication == null) {
            throw new DirectAuthenticationServiceException("authentication manager returned null authentication",
                    AUTHENTICATION_SERVICE_ERROR, WEB_AUTH_TRANSLATIONS.authenticationServiceUnavailable(), null);
        }

        SecurityContextHolder.getContext().setAuthentication(authentication);

        try {
            if (authentication instanceof DirectAuthentication) {
                DirectAuthentication directAuth = (DirectAuthentication) authentication;
                if (directAuth.getOperator() != null && directAuth.getOperator().getUid() != null) {
                    TraceCommentVarsHolder.get().setOperator(String.valueOf(directAuth.getOperator().getUid()));
                }
            }
            chain.doFilter(request, response);
        } finally {
            TraceCommentVarsHolder.get().removeOperator();
        }

    }

    private Authentication authenticate(ServletRequest request, String tvmTicket) {
        HttpServletRequest httpRequest = (HttpServletRequest) request;
        logger.debug("authenticating incoming request: {}", httpRequest.getRequestURL());

        Optional<BlackboxCookieFakeAuth> fakeAuthOptional = fakeAuth(httpRequest);
        if (fakeAuthOptional.isPresent()) {
            BlackboxCookieFakeAuth fakeAuth = fakeAuthOptional.get();
            User newOperator = userService.getUser(fakeAuth.getUid());
            checkArgument(newOperator != null, "fake operator with uid %s not found", fakeAuth);

            return new DirectAuthentication(newOperator, newOperator);
        } else {
            BlackboxCookieCredentials credentials = BlackboxCookieCredentials.extract(httpRequest);
            BlackboxCookieAuthRequest authRequest = new BlackboxCookieAuthRequest(credentials, tvmTicket);
            return authenticationManager.authenticate(authRequest);
        }
    }

    private Optional<BlackboxCookieFakeAuth> fakeAuth(HttpServletRequest httpRequest) {
        if (environmentType != EnvironmentType.DEVELOPMENT) {
            return Optional.empty();
        }

        Optional<String> loginOpt = HttpUtil.getHeaderValue(httpRequest, FAKE_LOGIN_HEADER_NAME);
        if (!loginOpt.isPresent()) {
            return Optional.empty();
        }
        String login = loginOpt.get();

        InetAddress remoteAddress = HttpUtil.getRemoteAddress(httpRequest);
        if (!netAcl.isInternalIp(remoteAddress)) {
            throw new AuthenticationServiceException("Fake auth: not internal ip: " + remoteAddress);
        }

        Long uid = userService.getUidByLogin(login);
        if (uid == null) {
            throw new AuthenticationServiceException("Fake auth: no uid for " + login);
        }

        logger.error("Fake authentication used for {}", login);
        return Optional.of(new BlackboxCookieFakeAuth(uid, login));
    }
}
