package ru.yandex.direct.web.core.security.authentication;

import java.net.InetAddress;

import javax.servlet.http.HttpServletRequest;

import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.common.net.NetRangeUtil;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.core.security.SecurityTranslations;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.web.auth.blackbox.BlackboxBasedAuth;
import ru.yandex.direct.web.core.security.authentication.exception.BlackboxCookieAuthenticationException;
import ru.yandex.direct.web.core.security.authentication.exception.DirectAuthenticationServiceException;
import ru.yandex.direct.web.core.security.authentication.exception.NeedsPostregistrationException;

import static ru.yandex.direct.rbac.RbacRole.AGENCY;
import static ru.yandex.direct.rbac.RbacRole.EMPTY;
import static ru.yandex.direct.web.core.security.authentication.AuthenticationErrorCodes.ACCESS_DENIED;
import static ru.yandex.direct.web.core.security.authentication.AuthenticationErrorCodes.AUTHENTICATION_SERVICE_ERROR;
import static ru.yandex.direct.web.core.security.authentication.AuthenticationErrorCodes.INVALID_REQUEST;
import static ru.yandex.direct.web.core.security.authentication.AuthenticationErrorCodes.ONLY_FOR_INTERNAL_USERS;
import static ru.yandex.direct.web.core.security.authentication.AuthenticationErrorCodes.PDD_USER;
import static ru.yandex.direct.web.core.security.authentication.AuthenticationErrorCodes.POST_REGISTRATION_REQUIRED;
import static ru.yandex.direct.web.core.security.authentication.AuthenticationErrorCodes.UNKNOWN_UID;
import static ru.yandex.direct.web.core.security.authentication.AuthenticationErrorCodes.UNKNOWN_USER;
import static ru.yandex.direct.web.core.security.authentication.WebAuthTranslations.WEB_AUTH_TRANSLATIONS;

public class DirectCookieAuthProvider implements AuthenticationProvider {

    public static final String PARAMETER_ULOGIN = "ulogin";
    public static final String PARAMETER_CLIENT_UID = "client_uid";

    private static final Logger logger = LoggerFactory.getLogger(DirectCookieAuthProvider.class);

    private final UserService userService;
    private final RbacService rbacService;
    private final ClientService clientService;
    private final NetAcl netAcl;

    public DirectCookieAuthProvider(UserService userService, RbacService rbacService, NetAcl netAcl,
                                    ClientService clientService) {
        this.userService = userService;
        this.rbacService = rbacService;
        this.netAcl = netAcl;
        this.clientService = clientService;
    }

    @Override
    public Authentication authenticate(Authentication auth) {
        logger.debug("internal direct authentication requested ({})", auth);
        BlackboxBasedAuth blackboxAuth = (BlackboxBasedAuth) auth;
        if (Strings.isNullOrEmpty(blackboxAuth.getPrincipal())) {
            throw new NeedsPostregistrationException(POST_REGISTRATION_REQUIRED,
                    WEB_AUTH_TRANSLATIONS.registrationDataAbsent(), null);
        }
        if (blackboxAuth.isHostedPdd()) {
            throw new BlackboxCookieAuthenticationException(PDD_USER, WEB_AUTH_TRANSLATIONS.accessDeniedForPDDUsers(),
                    null);
        }
        try {
            logger.debug("fetching authentication info");
            User operator = userService.getUser(blackboxAuth.getUid());
            if (operator == null) {
                throw new BlackboxCookieAuthenticationException(UNKNOWN_USER, WEB_AUTH_TRANSLATIONS.userNotFound(),
                        null);
            }

            if (operator.getRole() == null) {
                //директ не знает про роль, считаем, что в этом случае права минимальны
                operator.setRole(EMPTY);
            }
            DirectWebAuthRequest authRequest = getWebAuthRequest(operator.getRole().isInternal());
            validateIpIsAllowed(authRequest.getClientAddress(), operator);
            User subjectUser = resolveSubjectUser(authRequest, operator);
            //Пользователь яндекс.агенств не имеет прав на WEB-API
            if (clientService.isYaAgencyClient(subjectUser.getClientId()) && !operator.getRole().isInternal()) {
                throw new BlackboxCookieAuthenticationException(ACCESS_DENIED,
                        SecurityTranslations.INSTANCE.accessDenied(), null);
            }
            return new DirectAuthentication(operator, subjectUser, blackboxAuth.getTvmUserTicket(), null);
        } catch (BlackboxCookieAuthenticationException e) {
            throw e;
        } catch (AuthenticationException ex) {
            throw new BlackboxCookieAuthenticationException(ex, ACCESS_DENIED,
                    SecurityTranslations.INSTANCE.accessDenied(), null);
        } catch (Exception ex) {
            throw new DirectAuthenticationServiceException(ex, AUTHENTICATION_SERVICE_ERROR,
                    WEB_AUTH_TRANSLATIONS.authenticationServiceUnavailable(), null);
        }
    }

    private void validateIpIsAllowed(InetAddress clientAddress, User operator) {
        if (operator.getRole().isInternal() && !netAcl.isInternalIp(clientAddress)) {
            logger.warn("internal network violation for {}: {}", operator.getRole(), clientAddress);
            throw new BlackboxCookieAuthenticationException(ONLY_FOR_INTERNAL_USERS,
                    WEB_AUTH_TRANSLATIONS.onlyForInternalNetwork(), null);
        }
        String allowedIps = operator.getAllowedIps();
        if (!(allowedIps == null || allowedIps.isEmpty() || NetRangeUtil.isIpInNetworks(clientAddress, allowedIps))) {
            throw new AccessDeniedException("Попытка доступа из неавторизованной сети");
        }
    }

    /**
     * Возвращает subclient'а для пользовательского запроса.
     * <p>
     * Если сабклиент отличается от оператора - проверяет, есть ли у оператора права на этого клиента.
     * <p>
     * Оператор может указать логин желаемого клиента с пом. параметра {@value PARAMETER_ULOGIN}. Кроме того, внутренние
     * операторы ({@link RbacRole#isInternal}) могут указать непосредственно uid клиента с пом. параметра
     * {@value PARAMETER_CLIENT_UID}
     * <p>
     * N. B.: любое агентство имеет права на любого клиента с ролью {@link RbacRole#EMPTY}. Кроме того, проверка прав на
     * клиента не осуществляется для внутренних операторов.
     *
     * @param authRequest
     * @param operator
     * @return
     */
    private User resolveSubjectUser(DirectWebAuthRequest authRequest, User operator) {
        String ulogin = authRequest.getUlogin();
        boolean operatorIsInternal = operator.getRole().isInternal();
        if (ulogin == null && operatorIsInternal && authRequest.getClientUid() != null) {
            User subClient = userService.getUser(authRequest.getClientUid());
            if (subClient == null) {
                throw new BlackboxCookieAuthenticationException(UNKNOWN_UID,
                        WEB_AUTH_TRANSLATIONS.uidNotFound(authRequest.getClientUid()), null);
            }
            return subClient;
        }
        if (ulogin != null) {
            Long requestedUid = userService.getUidByLogin(ulogin);
            if (requestedUid == null) {
                logger.debug("can't find uid for login {}", ulogin);
                throw new BlackboxCookieAuthenticationException(ACCESS_DENIED,
                        SecurityTranslations.INSTANCE.accessDenied(), null);
            }
            if (requestedUid.longValue() != operator.getUid().longValue()) {
                boolean ownsRequestedUid = rbacService.isOwner(operator.getUid(), requestedUid);
                User requestedClient = userService.getUser(requestedUid);
                if (requestedClient == null) {
                    throw new BlackboxCookieAuthenticationException(UNKNOWN_UID,
                            WEB_AUTH_TRANSLATIONS.uidNotFound(requestedUid), null);
                }
                RbacRole opRole = operator.getRole();
                if (operatorIsInternal || ownsRequestedUid || (opRole == AGENCY
                        && requestedClient.getRole() == EMPTY)) {
                    return requestedClient;
                } else {
                    throw new BlackboxCookieAuthenticationException(ACCESS_DENIED,
                            SecurityTranslations.INSTANCE.accessDenied(), null);
                }
            }
        }
        return operator;
    }

    protected DirectWebAuthRequest getWebAuthRequest(boolean isInternalRole) {
        HttpServletRequest request = HttpUtil.getRequest();
        InetAddress clientAddress = HttpUtil.getSecretRemoteAddress(request);
        Long clientUid = null;
        if (isInternalRole) {
            String clientUidString = request.getParameter(PARAMETER_CLIENT_UID);
            if (clientUidString != null) {
                try {
                    clientUid = Long.parseLong(clientUidString);
                } catch (NumberFormatException nfe) {
                    throw new BlackboxCookieAuthenticationException(nfe, INVALID_REQUEST,
                            WEB_AUTH_TRANSLATIONS.uidFormatError(clientUidString), null);
                }
            }
        }
        return new DirectWebAuthRequest(clientAddress, request.getParameter(PARAMETER_ULOGIN), clientUid);
    }

    @Override
    public boolean supports(Class<?> authClass) {
        return BlackboxBasedAuth.class.isAssignableFrom(authClass);
    }
}
