package ru.yandex.direct.web.core.security.authentication;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;

import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.tracing.util.TraceCommentVarsHolder;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.web.auth.blackbox.BlackboxCookieAuthRequest;
import ru.yandex.direct.web.auth.blackbox.BlackboxCookieCredentials;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Фильтр для предоставления доступа (к публичным ручкам) без аутентификации.
 */
public class DirectPublicAuthFilter implements Filter {

    private static final Logger logger = LoggerFactory.getLogger(DirectPublicAuthFilter.class);

    private final AuthenticationManager authenticationManager;
    private final TvmIntegration tvmIntegration;
    private final TvmService tvmService;

    public DirectPublicAuthFilter(AuthenticationManager authenticationManager,
                                  TvmIntegration tvmIntegration,
                                  TvmService tvmService) {
        checkNotNull(authenticationManager, "authenticationManager is required");
        this.authenticationManager = authenticationManager;
        this.tvmIntegration = tvmIntegration;
        this.tvmService = tvmService;
    }

    @Override
    public void init(FilterConfig filterConfig) {
    }

    @Override
    public void destroy() {
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
            throws IOException, ServletException {
        Authentication auth;
        HttpServletRequest httpRequest = (HttpServletRequest) request;
        logger.debug("authenticating incoming request: {}", httpRequest.getRequestURL().toString());
        BlackboxCookieCredentials credentials = BlackboxCookieCredentials.extract(httpRequest);

        String tvmTicket = tvmIntegration.getTicket(tvmService);
        BlackboxCookieAuthRequest authRequest = new BlackboxCookieAuthRequest(credentials, tvmTicket);

        auth = authenticationManager.authenticate(authRequest);

        if (auth != null) {
            SecurityContextHolder.getContext().setAuthentication(auth);
        }

        // Код дублирует участок из BlackboxCookieAuthFilter.
        // Пока нет необходимости выносить отдельно, но при появлении новых дубликатов стоит это сделать.
        try {
            if (auth instanceof DirectAuthentication) {
                DirectAuthentication directAuth = (DirectAuthentication) auth;
                if (directAuth.getOperator() != null && directAuth.getOperator().getUid() != null) {
                    TraceCommentVarsHolder.get().setOperator(String.valueOf(directAuth.getOperator().getUid()));
                }
            }
            chain.doFilter(request, response);
        } finally {
            TraceCommentVarsHolder.get().removeOperator();
        }
    }
}
