package ru.yandex.direct.web.core.security.authentication;


import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;
import reactor.util.annotation.NonNull;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.core.security.SecurityTranslations;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.web.annotations.AllowedBlockedOperatorOrUser;
import ru.yandex.direct.web.annotations.AllowedOperatorRoles;
import ru.yandex.direct.web.annotations.AllowedSubjectRoles;
import ru.yandex.direct.web.annotations.CheckSubjectAndUserFeaturesSeparately;
import ru.yandex.direct.web.annotations.OperatorHasFeatures;
import ru.yandex.direct.web.annotations.SubjectHasFeatures;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;

import static ru.yandex.direct.core.entity.user.utils.BlockedUserUtil.checkOperatorIsBlocked;
import static ru.yandex.direct.core.entity.user.utils.BlockedUserUtil.checkSubjectUserIsBlocked;

/**
 * Проверяет, разрешен ли для данного оператора c заданным набором фичей доступ на уровне контроллера к клиенту с
 * набором фичей клиента.
 * Парсит {@link AllowedBlockedOperatorOrUser}, {@link AllowedOperatorRoles}, {@link AllowedSubjectRoles},
 * {@link OperatorHasFeatures} и {@link SubjectHasFeatures}
 * над контроллерами. Аннотации могут быть установлены над методом или классом контроллеров.
 * Аннотация над методом имеет приоритет.
 * Если нет аннотаций, по умолчанию роль оператора разрешена любая, роль субъекта - клиент.
 */
public class DirectWebAuthenticationInterceptor extends HandlerInterceptorAdapter {
    private final FeatureService featureService;
    private final DirectWebAuthenticationSource authenticationSource;

    public DirectWebAuthenticationInterceptor(
            FeatureService featureService,
            DirectWebAuthenticationSource authenticationSource) {
        this.featureService = featureService;
        this.authenticationSource = authenticationSource;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {
        if (!(handler instanceof HandlerMethod) || !authenticationSource.isAuthenticated()) {
            return true;
        }
        HandlerMethod handlerMethod = (HandlerMethod) handler;
        DirectAuthentication authentication = authenticationSource.getAuthentication();
        checkOperatorAndSubjectUser(handlerMethod, authentication);
        return true;
    }

    void checkOperatorAndSubjectUser(HandlerMethod handlerMethod, DirectAuthentication authentication) {
        User operator = authentication.getOperator();
        User subjectUser = authentication.getSubjectUser();

        boolean methodAllowedForBlockedOperator =
                getMethodOrClassAnnotation(handlerMethod, AllowedBlockedOperatorOrUser.class) != null;
        if (checkOperatorIsBlocked(operator, methodAllowedForBlockedOperator)) {
            throw new AccessDeniedException("Оператор заблокирован", SecurityTranslations.INSTANCE.loginIsDisabled());
        }

        AllowedOperatorRoles allowedOperatorRoles =
                getMethodOrClassAnnotation(handlerMethod, AllowedOperatorRoles.class);
        if (allowedOperatorRoles != null && !Arrays.asList(allowedOperatorRoles.value()).contains(operator.getRole())) {
            throwOperatorNoRightsException(operator);
        }

        boolean checkClientAndOperatorFeaturesSeparately = getMethodOrClassAnnotation(handlerMethod,
                CheckSubjectAndUserFeaturesSeparately.class) != null;

        OperatorHasFeatures[] operatorHasOneFeaturesSet =
                getMethodOrClassAnnotations(handlerMethod, OperatorHasFeatures.class);
        SubjectHasFeatures subjectHasFeatures = getMethodOrClassAnnotation(handlerMethod, SubjectHasFeatures.class);

        boolean operatorHasNoRightsByFeature = operatorHasOneFeaturesSet.length != 0 &&
                Stream.of(operatorHasOneFeaturesSet).noneMatch(operatorHasFeatures -> featureService
                        .isEnabledForUid(operator.getUid(), Arrays.asList(operatorHasFeatures.value())));
        if (operatorHasNoRightsByFeature &&
                (!checkClientAndOperatorFeaturesSeparately || subjectHasFeatures == null)) {
            throwOperatorNoRightsException(operator);
        }

        boolean methodAllowedForBlockedUser =
                getMethodOrClassAnnotation(handlerMethod, AllowedBlockedOperatorOrUser.class) != null;
        if (checkSubjectUserIsBlocked(subjectUser, methodAllowedForBlockedUser, authentication.getOperator())) {
            throw new AccessDeniedException("Клиент заблокирован", SecurityTranslations.INSTANCE.loginIsDisabled());
        }

        AllowedSubjectRoles allowedSubjectRoles = getMethodOrClassAnnotation(handlerMethod, AllowedSubjectRoles.class);
        List<RbacRole> allowedRoles = allowedSubjectRoles == null ?
                Collections.singletonList(RbacRole.CLIENT) :
                Arrays.asList(allowedSubjectRoles.value());
        if (!allowedRoles.contains(subjectUser.getRole())) {
            throwClientNoRightsException(subjectUser);
        }

        boolean clientHasNoRightsByFeature = subjectHasFeatures != null && !featureService
                .isEnabledForUid(subjectUser.getUid(), Arrays.asList(subjectHasFeatures.value()));
        if (clientHasNoRightsByFeature &&
                (!checkClientAndOperatorFeaturesSeparately || operatorHasOneFeaturesSet.length == 0)) {
            throwClientNoRightsException(subjectUser);
        }
        if (operatorHasNoRightsByFeature && clientHasNoRightsByFeature) {
            throwOperatorNoRightsException(operator);
        }
    }

    private void throwOperatorNoRightsException(User operator) {
        throw new AccessDeniedException(
                String.format("У оператора [uid=%s, role=%s] нет прав на выполнение операции", operator.getUid(),
                        operator.getRole()),
                SecurityTranslations.INSTANCE.accessDenied());
    }

    private void throwClientNoRightsException(User subjectUser) {
        throw new AccessDeniedException(
                String.format("Операцию нельзя выполнить для данного клиента [uid=%s, role=%s] ",
                        subjectUser.getUid(), subjectUser.getRole()),
                SecurityTranslations.INSTANCE.accessDenied());
    }

    @Nullable
    private <T extends Annotation> T getMethodOrClassAnnotation(HandlerMethod handlerMethod, Class<T> annotationClass) {
        T annotation = null;

        Method method = handlerMethod.getMethod();
        if (method.getDeclaringClass().isAnnotationPresent(annotationClass)) {
            annotation = method.getDeclaringClass().getAnnotation(annotationClass);
        }
        if (handlerMethod.hasMethodAnnotation(annotationClass)) {
            annotation = handlerMethod.getMethodAnnotation(annotationClass);
        }
        return annotation;
    }

    @NonNull
    private <T extends Annotation> T[] getMethodOrClassAnnotations(HandlerMethod handlerMethod,
                                                                   Class<T> annotationClass) {
        T[] annotations;

        Method method = handlerMethod.getMethod();
        annotations = method.getAnnotationsByType(annotationClass);
        if (annotations.length == 0) {
            annotations = method.getDeclaringClass().getAnnotationsByType(annotationClass);
        }
        return annotations;
    }
}
