package ru.yandex.direct.web.core.security.authentication;

import java.io.IOException;
import java.net.InetAddress;
import java.util.EnumSet;
import java.util.Optional;
import java.util.Set;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.context.SecurityContextHolder;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.rbac.RbacRole;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Фильтр подменяет исходного оператора, определенного по паспортным кукам,
 * на оператора, переданного в куке fake_name в виде логина или uid'а, при ее наличии.
 * <p>
 * Механизм работает только для ролей {@link FakeCookieAuthFilter#ALLOWED_ROLES}, а так же только для внутренней сети.
 * Если эти два условия не выполнены, то он не делает ничего, полностью прозрачен.
 * <p>
 * Если в процессе применения фэйковой авторизации случится исключение, то оно будет
 * записано в лог и запрос будет завершен.
 * <p>
 * Фильтр должен срабатывать, после того как реальная аутентификация полностью пройдена,
 * и определен настоящий оператор.
 * <p>
 * Фильтр не должен присутствовать в продакшен-конфигурациях!
 */
public class FakeCookieAuthFilter implements Filter {

    public static final String FAKE_OPERATOR_COOKIE_NAME = "fake_name";

    /**
     * Роли, которым разрешено использовать фейковую авторизацию.
     */
    private static final Set<RbacRole> ALLOWED_ROLES = EnumSet.of(
            RbacRole.SUPER,
            RbacRole.SUPERREADER,
            RbacRole.SUPPORT
    );

    private static final Logger logger = LoggerFactory.getLogger(FakeCookieAuthFilter.class);

    private final UserService userService;
    private final NetAcl netAcl;

    public FakeCookieAuthFilter(UserService userService, NetAcl netAcl) {
        this.userService = userService;
        this.netAcl = netAcl;
    }

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
    }

    @Override
    public void destroy() {
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
            throws IOException, ServletException {
        HttpServletRequest httpRequest = (HttpServletRequest) request;
        try {
            Optional<Cookie> fakeOperatorCookie = findFakeOperatorCookie(httpRequest);
            if (fakeOperatorCookie.isPresent()) {
                replaceOperatorIfPossible(httpRequest, fakeOperatorCookie.get().getValue());
            }
        } catch (Exception e) {
            logger.error("fake-login failed", e);
            ((HttpServletResponse) response).sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
                    "Fake-login failed with " + e + ". Check \"fake_name\" cookie or see logs.");
            return;
        }

        chain.doFilter(request, response);
    }

    private void replaceOperatorIfPossible(HttpServletRequest httpRequest, String newOperatorValue) {
        DirectAuthentication directAuthentication =
                (DirectAuthentication) SecurityContextHolder.getContext().getAuthentication();
        User currentOperator = directAuthentication.getOperator();
        if (!ALLOWED_ROLES.contains(currentOperator.getRole())) {
            return;
        }

        InetAddress operatorAddress = HttpUtil.getRemoteAddress(httpRequest);
        if (!netAcl.isInternalIp(operatorAddress)) {
            return;
        }

        Long newUid = extractUid(newOperatorValue);
        checkArgument(newUid != null, "fake operator %s not found", newOperatorValue);

        User newOperator = userService.getUser(newUid);
        checkArgument(newOperator != null, "fake operator with uid %s not found", newUid);

        User subjectUser = directAuthentication.getSubjectUser();
        User newSubjectUser = currentOperator.getUid().equals(subjectUser.getUid()) ? newOperator : subjectUser;

        DirectAuthentication newAuthentication = new DirectAuthentication(newOperator, newSubjectUser);
        SecurityContextHolder.getContext().setAuthentication(newAuthentication);

        logger.info("fake authentication is completed: operator {} is replaced by {}",
                directAuthentication.getOperator().getLogin(),
                newAuthentication.getOperator().getLogin());
    }

    private Optional<Cookie> findFakeOperatorCookie(HttpServletRequest httpRequest) {
        Cookie[] cookies = httpRequest.getCookies();
        if (cookies == null) {
            return Optional.empty();
        }
        return StreamEx.of(cookies)
                .findFirst(cookie -> cookie.getName().equals(FAKE_OPERATOR_COOKIE_NAME));
    }

    private Long extractUid(String newOperator) {
        try {
            return Long.valueOf(newOperator);
        } catch (NumberFormatException e) {
            return userService.getUidByLogin(newOperator);
        }
    }
}
