package ru.yandex.direct.web.core.security.captcha;

import java.io.IOException;
import java.io.PrintWriter;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.net.InetAddresses;
import com.google.common.primitives.Ints;
import one.util.streamex.StreamEx;
import org.apache.http.entity.ContentType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import ru.yandex.direct.captcha.passport.YandexCaptchaClient;
import ru.yandex.direct.captcha.passport.YandexCaptchaException;
import ru.yandex.direct.captcha.passport.entity.CaptchaType;
import ru.yandex.direct.captcha.passport.entity.check.response.CheckStatus;
import ru.yandex.direct.captcha.passport.entity.generate.response.CaptchaGenerateResponse;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.env.Environment;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.web.core.model.WebCaptchaResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.web.core.semaphore.RedisSemaphoreInterceptor.SHOULD_SEE_CAPTCHA_ENTRY_NAME;

/**
 * Запрашивает у клиента распознавание капчи в случае превышения числа запросов. Способы 'появления' капчи:
 * <ul>
 * <li>
 * Пользователь забанен автоматически({@link User#getAutobanned}), у него ненулевая Карма ({@link User#getPassportKarma}),
 * либо капчу включили вручную({@link User#getCaptchaFreq}). Эту 'автокапчу' можно отключить, добавив {@link DisableAutoCaptcha}.
 * См. {@link #buildAutoCaptchaParams}
 * </li>
 * <li>
 * На методе контроллера стоит одна или несколько аннотаций {@link CaptchaSecured}
 * </li>
 * </ul>
 */
public class CaptchaFirewallInterceptor extends HandlerInterceptorAdapter {
    private static final String CAPTCHA_ID_PARAM = "captcha_id";
    private static final String CAPTCHA_CODE_PARAM = "captcha_code";
    private static final String DO_NOT_SHOW_CAPTCHA_PARAM = "do_not_show_captcha";
    private static final int HTTP_CODE_TOO_MANY_REQUESTS = 429;
    private static final CaptchaType DEFAULT_CAPTCHA_TYPE = CaptchaType.STD;
    private static final String CONTENT_TYPE = ContentType.APPLICATION_JSON.getMimeType();
    private static final Logger logger = LoggerFactory.getLogger(CaptchaFirewallInterceptor.class);
    private final DirectWebAuthenticationSource authenticationSource;
    private final YandexCaptchaClient yandexCaptchaClient;
    private final LettuceConnectionProvider lettuce;
    private final CaptchaFirewallInterceptorConfig config;
    private final String defaultEncoding;
    private final PpcProperty<Boolean> noKarmaCapthaProperty;

    public CaptchaFirewallInterceptor(CaptchaFirewallInterceptorConfig config,
                                      LettuceConnectionProvider lettuceConnectionProvider,
                                      YandexCaptchaClient yandexCaptchaClient,
                                      DirectWebAuthenticationSource authenticationSource,
                                      String defaultEncoding,
                                      PpcPropertiesSupport ppcPropertiesSupport) {
        this.authenticationSource = authenticationSource;
        this.yandexCaptchaClient = yandexCaptchaClient;
        this.lettuce = lettuceConnectionProvider;
        this.config = config;
        this.defaultEncoding = defaultEncoding;
        this.noKarmaCapthaProperty = ppcPropertiesSupport.get(PpcPropertyNames.DISABLE_KARMA_CAPTCHA,
                Duration.ofSeconds(60L));
    }

    private static String createRequestType(HandlerMethod handlerMethod) {
        return handlerMethod.getMethod().getAnnotation(RequestMapping.class).path()[0];
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {
        if (isCaptchaDisabled(request)) {
            return true;
        }
        if (!(handler instanceof HandlerMethod)) {
            return true;
        }
        CaptchaChecker captchaChecker = getCaptchaChecker(request, (HandlerMethod) handler);

        boolean needsCaptchaSolving = captchaChecker.onUserRequest();
        if (needsCaptchaSolving) {
            String captchaId = request.getParameter(CAPTCHA_ID_PARAM);
            String captchaCode = request.getParameter(CAPTCHA_CODE_PARAM);
            if (captchaId != null && captchaCode != null) {
                boolean isCaptchaOk;
                try {
                    isCaptchaOk = yandexCaptchaClient.
                            check(findOutCaptchaType(), captchaId, captchaCode).getCheckStatus() == CheckStatus.OK;
                } catch (YandexCaptchaException ex) {
                    logger.error("can't check captcha", ex);
                    return true;
                }
                if (isCaptchaOk) {
                    captchaChecker.onRecognition();
                    var operator = getOperator();
                    logger.info("Captcha success for uid={}, login={}, autobanned={}, passportKarma={}",
                            operator.getUid(), operator.getLogin(), operator.getAutobanned(),
                            operator.getPassportKarma());
                    return true;
                } else {
                    //клиент прислал результаты распознавания капчи, но Паспорт их не принял
                    askClientToSolveCaptcha(request, response);
                    return false;
                }
            } else {
                //распознать надо, но результаты распознавания не пришли(т.е. первое срабатывание)
                askClientToSolveCaptcha(request, response);
                return false;
            }
        } else {
            return true;
        }
    }

    /**
     * Возвращает объект-капчу для запроса {@code request} к web-методу{@code handler}.
     */
    private CaptchaChecker getCaptchaChecker(HttpServletRequest request, HandlerMethod handlerMethod) {
        CaptchaSecured[] firewalls = handlerMethod.getMethod().getAnnotationsByType(CaptchaSecured.class);
        List<CaptchaParams> firewallParams = new ArrayList<>(firewalls.length + 1);
        for (CaptchaSecured firewall : firewalls) {
            CaptchaLimitsParams captchaLimits = CaptchaLimitsParams.fromCaptchaLimits(firewall.limits());
            CaptchaParams fwRequest = new CaptchaParams(
                    createRequestType(handlerMethod), createRequestKey(firewall.keys(), request), captchaLimits);
            firewallParams.add(fwRequest);
        }
        CaptchaParams autoCaptchaParams = buildAutoCaptchaParams(handlerMethod);
        if (autoCaptchaParams != null) {
            firewallParams.add(autoCaptchaParams);
        }

        var requestsLimitReached = new CaptchaChecker() {
            @Override
            public boolean onUserRequest() {
                Object attribute = request.getAttribute(SHOULD_SEE_CAPTCHA_ENTRY_NAME);
                return attribute instanceof Boolean && (Boolean) attribute;
            }

            @Override
            public void onRecognition() {
            }
        };

        List<CaptchaChecker> checkers = StreamEx.of(firewallParams)
                .map(this::toCaptchaChecker)
                .append(requestsLimitReached)
                .toList();

        return new CombinedCaptchaChecker(checkers);
    }

    private CaptchaChecker toCaptchaChecker(CaptchaParams params) {
        return new RedisCaptchaChecker(config.getRedisKeyPrefix(), lettuce, params);
    }

    private CaptchaType findOutCaptchaType() {
        //todo добавить проверку аналогичную перловому коду $login_rights->{role} =~ /^(empty|client)$/ && !Captcha::has_paid($ClientID)
//        User operator = getOperator();
//        operator.getClientId();
//        operator.getRole()
        return DEFAULT_CAPTCHA_TYPE;
    }

    private CaptchaParams buildAutoCaptchaParams(HandlerMethod handlerMethod) {
        if (handlerMethod.getMethodAnnotation(DisableAutoCaptcha.class) != null) {
            return null;
        }
        User operator = getOperator();
        long captchaFreq = 0;
        if (operator.getCaptchaFreq() > 0) {
            captchaFreq = operator.getCaptchaFreq();
        } else {
            if (!nvl(noKarmaCapthaProperty.get(), false) && operator.getPassportKarma() > 0) {
                Long passportKarma = operator.getPassportKarma();
                captchaFreq = Interpolator.interpolateConst(config.getKarmaLimits(), passportKarma);
            }
        }
        if (operator.getAutobanned() && (captchaFreq > config.getAutobanCaptchaFreq() || captchaFreq == 0)) {
            logger.info("operator with uid={} is autobanned", operator.getUid());
            captchaFreq = config.getAutobanCaptchaFreq();
        }
        if (captchaFreq > 0) {
            int freq = Ints.saturatedCast(captchaFreq);
            CaptchaLimitsParams captchaLimits = new CaptchaLimitsParams(freq, 86400, freq);
            return new CaptchaParams("uid_manual",
                    operator.getUid().toString(), captchaLimits);
        }
        return null;
    }

    private User getOperator() {
        return authenticationSource.getAuthentication().getOperator();
    }

    private void askClientToSolveCaptcha(HttpServletRequest request, HttpServletResponse response) {
        CaptchaGenerateResponse capthaResponse = yandexCaptchaClient.generate(findOutCaptchaType(), 1);
        try {
            if (HttpUtil.isAjax(request)) {
                logger.info("asking client to solve captcha for ajax request");
                response.setStatus(HTTP_CODE_TOO_MANY_REQUESTS);
                response.setContentType(CONTENT_TYPE);
                response.setCharacterEncoding(defaultEncoding);
                try (PrintWriter writer = response.getWriter()) {
                    WebResponse responseObject = new WebCaptchaResponse()
                            .withCaptchaId(capthaResponse.getRequestId())
                            .withCaptchaUrl(capthaResponse.getUrl());
                    writer.write(JsonUtils.toJson(responseObject));
                }
            } else {
                //todo kuhtich: добавить в редирект id и url каптчи
                logger.info("asking client to solve captcha for non-ajax request");
                response.sendRedirect(config.getCaptchaSolvingPageUrl());
            }
        } catch (IOException ex) {
            logger.error("error writing http response", ex);
        }
    }

    private boolean isCaptchaDisabled(HttpServletRequest request) {
        if (Environment.getCached().isBeta() || Environment.getCached() == EnvironmentType.TESTING) {
            if (request.getCookies() == null) {
                return false;
            }

            return StreamEx.of(request.getCookies())
                    .map(Cookie::getName)
                    .anyMatch(name -> name.equals(DO_NOT_SHOW_CAPTCHA_PARAM));
        }

        return false;
    }

    private String createRequestKey(CaptchaConditionKey[] conditionKeys, HttpServletRequest request) {
        return Arrays.stream(conditionKeys)
                .map(el -> valueForKey(el, request))
                .collect(Collectors.joining("/"));
    }

    private String valueForKey(CaptchaConditionKey key, HttpServletRequest request) {
        switch (key) {
            case IP_VALUE:
                return InetAddresses.toAddrString(HttpUtil.getRemoteAddress(request));
            case UID_VALUE:
                return String.valueOf(authenticationSource.getAuthentication().getOperator().getUid());
            default:
                throw new IllegalArgumentException("unknown CaptchaConditionKey value " + key);
        }
    }
}
