package ru.yandex.direct.web.core.security.captcha;

import java.nio.charset.StandardCharsets;
import java.util.zip.CRC32;

import com.google.common.net.InetAddresses;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.common.lettuce.LettuceExecuteException;
import ru.yandex.direct.common.util.HttpUtil;

class RedisCaptchaChecker implements CaptchaChecker {
    private static final Logger logger = LoggerFactory.getLogger(RedisCaptchaChecker.class);

    private final LettuceConnectionProvider lettuce;

    /**
     * Глобальный префикс. Используется в качестве префикса при генерации redis-ключа.
     */
    private final String keyPrefix;
    private final CaptchaParams firewallRequest;
    private String checkKey;


    public RedisCaptchaChecker(String keyPrefix, LettuceConnectionProvider lettuce, CaptchaParams firewallRequest) {
        this.lettuce = lettuce;
        this.keyPrefix = keyPrefix;
        this.firewallRequest = firewallRequest;
        this.checkKey = null;
    }


    /**
     * Генерирует ключ для использования в redis.
     * <p>
     * Ключ формируется в формате "{@link #keyPrefix}/{@link CaptchaParams#getRequestType()}/{@code shiftedTime}/{@link CaptchaParams#getRequestKey()}",
     * где {@code shiftedTime} - текущее время со сдвигом (от 0 до ({@link CaptchaLimitsParams#getInterval} - 1)). Значение сдвига зависит от
     * {@link CaptchaParams#requestKey}, либо от IP клиента, если {requestKey == null}.
     * Такой трюк нужен, чтобы рандомизировать выбор новый ключей и как следствие - сгладить нагрузку.
     */
    private String genRequestKey() {
        CRC32 crc32 = new CRC32();
        String requestIdentity = firewallRequest.getRequestKey() == null
                ? InetAddresses.toAddrString(HttpUtil.getRemoteAddress(HttpUtil.getRequest()))
                : firewallRequest.getRequestKey();
        crc32.update(requestIdentity.getBytes(StandardCharsets.UTF_8));
        CaptchaLimitsParams limits = firewallRequest.getCaptchaLimits();
        long timeShift = crc32.getValue() % limits.getInterval();
        long shiftedTime = (System.currentTimeMillis() / 1000 - timeShift) / limits.getInterval();
        return keyPrefix + "/" + firewallRequest.getRequestType() + "/" + shiftedTime + "/" + requestIdentity;
    }

    /**
     * @return true, если для данного метода превышено число запросов - и требуется распознание капчи
     */
    @Override
    public boolean onUserRequest() {
        checkKey = genRequestKey();
        try {
            CaptchaLimitsParams limits = firewallRequest.getCaptchaLimits();
            Long incremented = lettuce.call("redis:incrBy", cmd -> cmd.incrby(checkKey, limits.getIncrement()));
            if (incremented == null) {
                throw new LettuceExecuteException("incrBy returned null");
            } else if (incremented == limits.getIncrement()) {
                //ключа не было; incrBy сработал как add => надо проставить expire
                Boolean expired = lettuce.call("expire", cmd -> cmd.expire(checkKey, limits.getInterval()));
                if (expired == null) {
                    throw new LettuceExecuteException("expire returned null");
                } else if (!expired) {
                    logger.error("captcha discrepancy: redis was not able to set expiration time");
                }
                return false;
            } else {
                return incremented > limits.getMaxFreq();
            }
        } catch (LettuceExecuteException ex) {
            logger.error("captcha redis exception", ex);
            return false;
        }
    }

    @Override
    public void onRecognition() {
        if (checkKey != null) {
            CaptchaLimitsParams limits = firewallRequest.getCaptchaLimits();

            String value = String.valueOf(limits.getMaxFreq() - limits.getFreq() + 1);
            lettuce.call("redis:setex",
                    cmd -> cmd.setex(checkKey, limits.getInterval(), value)
            );
        }
    }
}
