package ru.yandex.direct.web.core.security.configuration;

import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.servlet.Filter;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.security.web.DefaultSecurityFilterChain;
import org.springframework.security.web.util.matcher.AndRequestMatcher;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.NegatedRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;

import ru.yandex.direct.blackbox.client.BlackboxClient;
import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.configuration.CoreConfiguration;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.integrations.configuration.IntegrationsConfiguration;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.web.auth.BestEffortAuthenticationManager;
import ru.yandex.direct.web.auth.SequenceAuthenticationManager;
import ru.yandex.direct.web.auth.blackbox.BlackboxCookieAuthProvider;
import ru.yandex.direct.web.core.security.authentication.BlackboxCookieAuthFilter;
import ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider;
import ru.yandex.direct.web.core.security.authentication.DirectPublicAuthFilter;
import ru.yandex.direct.web.core.security.authentication.FakeCookieAuthFilter;
import ru.yandex.direct.web.core.security.authentication.WebAuthenticationFilter;

import static java.util.Arrays.asList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

/**
 * Инкапсулирует аутентификацию в BlackBox и Rbac (в указанной последовательности)
 */

@Configuration
@Import({CoreConfiguration.class, IntegrationsConfiguration.class})
public class BlackboxWebAuthenticationConfiguration {

    public static final String PUBLIC_BASE_URL = "/public/";
    public static final String PARTNER_BASE_URL = "/partner/";

    public static final String BLACK_BOX_WEB_AUTHENTICATION_FILTER = "blackBoxWebAuthenticationFilter";

    private static final String MATCH_ALL_URLS = "/**";
    private static final String MATCH_PUBLIC_URLS = PUBLIC_BASE_URL + "**";
    private static final String MATCH_PARTNER_URLS = PARTNER_BASE_URL + "**";
    private static final String MATCH_ALIVE_URL = "/alive";
    private static final String MATCH_ADMIN_URL = "/admin";
    private static final String MATCH_MONITORING_URL = "/monitoring";
    private static final String MATCH_EXTERNAL_MONITORING_URL = "/external-metrics";
    private static final String MATCH_FRONTEND_TIMINGS_METRICS_URL = "/frontend-timings-metrics";
    private static final String MATCH_LOG_URL = "/log/**";
    private static final String MATCH_DOCS_API_URL = "/docs/api";

    /**
     * Bean name must be the same as filter-name element of DelegatingFilterProxy filter at web.xml.
     * {@link org.springframework.web.filter.DelegatingFilterProxy} is adapter between servlet container
     * and spring-managed security filter beans.
     * <p>
     * We dont use SecurityContextPersistenceFilter for clearing security context,
     * this operation is provided by FilterChainProxy itself.
     */
    @Bean(name = BLACK_BOX_WEB_AUTHENTICATION_FILTER)
    public WebAuthenticationFilter blackBoxWebAuthenticationFilter(
            BlackboxCookieAuthFilter blackboxCookieAuthFilter,
            DirectPublicAuthFilter directPublicAuthFilter,
            @Nullable FakeCookieAuthFilter fakeCookieAuthFilter) {
        RequestMatcher requestMatcher = new AndRequestMatcher(
                new AntPathRequestMatcher(MATCH_ALL_URLS),
                new NegatedRequestMatcher(new AntPathRequestMatcher(MATCH_ALIVE_URL)),
                new NegatedRequestMatcher(new AntPathRequestMatcher(MATCH_ADMIN_URL)),
                new NegatedRequestMatcher(new AntPathRequestMatcher(MATCH_MONITORING_URL)),
                new NegatedRequestMatcher(new AntPathRequestMatcher(MATCH_EXTERNAL_MONITORING_URL)),
                new NegatedRequestMatcher(new AntPathRequestMatcher(MATCH_FRONTEND_TIMINGS_METRICS_URL)),
                new NegatedRequestMatcher(new AntPathRequestMatcher(MATCH_LOG_URL)),
                new NegatedRequestMatcher(new AntPathRequestMatcher(MATCH_PUBLIC_URLS)),
                new NegatedRequestMatcher(new AntPathRequestMatcher(MATCH_PARTNER_URLS)),
                new NegatedRequestMatcher(new AntPathRequestMatcher(MATCH_DOCS_API_URL))
        );

        List<Filter> filters = filterList(asList(blackboxCookieAuthFilter, fakeCookieAuthFilter), Objects::nonNull);
        DefaultSecurityFilterChain trueAuthSecurityChain = new DefaultSecurityFilterChain(requestMatcher, filters);

        // Маппинг и фильтр для ручек с необязательной аутентификацией
        RequestMatcher publicRequestMatcher = new AndRequestMatcher(
                new AntPathRequestMatcher(MATCH_PUBLIC_URLS)
        );

        List<Filter> publicFilters = filterList(asList(directPublicAuthFilter, fakeCookieAuthFilter), Objects::nonNull);
        DefaultSecurityFilterChain publicBestEffortAuthSecurityChain =
                new DefaultSecurityFilterChain(publicRequestMatcher, publicFilters);

        return new WebAuthenticationFilter(trueAuthSecurityChain, publicBestEffortAuthSecurityChain);
    }

    @Bean
    public BlackboxCookieAuthFilter cookieAuthFilter(BlackboxClient blackboxClient,
                                                     EnvironmentType environmentType,
                                                     TvmIntegration tvmIntegration,
                                                     UserService userService,
                                                     RbacService rbacService,
                                                     NetAcl netAcl,
                                                     ClientService clientService) {
        TvmService tvmService = environmentType.isProductionOrPrestable()
                ? TvmService.BLACKBOX_PROD
                : TvmService.BLACKBOX_MIMINO;
        return new BlackboxCookieAuthFilter(
                environmentType, netAcl, userService, tvmIntegration,
                tvmService, cookieAuthManager(blackboxClient, userService, rbacService, netAcl, clientService)
        );
    }

    @Bean
    public DirectPublicAuthFilter directPublicAuthFilter(BlackboxClient blackboxClient,
                                                         UserService userService, RbacService rbacService,
                                                         NetAcl netAcl, ClientService clientService,
                                                         TvmIntegration tvmIntegration,
                                                         EnvironmentType environmentType) {
        TvmService tvmService = environmentType.isProductionOrPrestable()
                ? TvmService.BLACKBOX_PROD
                : TvmService.BLACKBOX_MIMINO;
        return new DirectPublicAuthFilter(
                bestEffortAuthManager(blackboxClient, userService, rbacService, netAcl, clientService),
                tvmIntegration, tvmService);
    }

    @Bean
    public SequenceAuthenticationManager cookieAuthManager(BlackboxClient blackboxClient,
                                                           UserService userService, RbacService rbacService,
                                                           NetAcl netAcl, ClientService clientService) {
        return new SequenceAuthenticationManager(asList(
                blackboxCookieAuthProvider(blackboxClient),
                directCookieAuthProvider(userService, rbacService, netAcl, clientService)));
    }

    @Bean
    public BestEffortAuthenticationManager bestEffortAuthManager(BlackboxClient blackboxClient,
                                                                 UserService userService, RbacService rbacService,
                                                                 NetAcl netAcl, ClientService clientService) {
        return new BestEffortAuthenticationManager(asList(
                blackboxCookieAuthProvider(blackboxClient),
                directCookieAuthProvider(userService, rbacService, netAcl, clientService)));
    }


    @Bean
    public BlackboxCookieAuthProvider blackboxCookieAuthProvider(BlackboxClient blackboxClient) {
        return new BlackboxCookieAuthProvider(blackboxClient);
    }


    @Bean
    public DirectCookieAuthProvider directCookieAuthProvider(UserService userService,
                                                             RbacService rbacService,
                                                             NetAcl netAcl, ClientService clientService) {
        return new DirectCookieAuthProvider(userService, rbacService, netAcl, clientService);
    }


    @Bean
    @Nullable
    public FakeCookieAuthFilter fakeCookieAuthFilter(EnvironmentType environmentType, UserService userService,
                                                     NetAcl netAcl) {
        if (environmentType == EnvironmentType.PRODUCTION
                || environmentType == EnvironmentType.PRESTABLE
                || environmentType == EnvironmentType.SANDBOX) {
            return null;
        }
        return new FakeCookieAuthFilter(userService, netAcl);
    }
}
