package ru.yandex.direct.web.core.security.csrf;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.HttpMethod;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;

/**
 * Проверяет корректность csrf токена.
 * Для POST запросов проверка производится всегда, кроме случая, если в методе для аннотации {@link CsrfCheck} enabled = false.
 * Для GET запросов проверка не производится, кроме случая, если в методе для аннотации {@link CsrfCheck} enabled = true.
 * Необходимость отдельного интерцептора вызвана тем, что часть read-only методов работает с HTTP POST.
 * В такой ситуации {@link org.springframework.security.web.csrf.CsrfFilter} не подойдет, т. к. он выполняется до того,
 * как будет найден обработчик запроса (т. о., до аннотации не добраться)
 */
@Component
public class CsrfInterceptor extends HandlerInterceptorAdapter {
    private static final Logger logger = LoggerFactory.getLogger(CsrfInterceptor.class);

    /*  Токен может передаваться как в заголовке, так и в параметрах,
        если в заголовке токен не указан, проверяется его наличие в параметрах */
    public static final String X_CSRF_TOKEN_HEADER = "X-CSRF-TOKEN";
    public static final String CSRF_TOKEN_PARAMETER_NAME = "csrf_token";
    public static final String CSRF_TOKEN_COOKIE_NAME = "_direct_csrf_token";

    private final CsrfValidator csrfValidator;
    private final DirectWebAuthenticationSource authenticationSource;
    private final EnvironmentType env;

    @Autowired
    public CsrfInterceptor(DirectWebAuthenticationSource authenticationSource, CsrfValidator csrfValidator,
                           EnvironmentType env) {
        this.authenticationSource = authenticationSource;
        this.csrfValidator = csrfValidator;
        this.env = env;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler)
            throws Exception {
        if (!(handler instanceof HandlerMethod)) {
            return true;
        }
        HandlerMethod handlerMethod = (HandlerMethod) handler;

        // если полькозватель неаутентифицирован (системная ручка) - пропускаем
        if (!authenticationSource.isAuthenticated()) {
            return true;
        }

        long uid = authenticationSource.getAuthentication().getOperator().getUid();

        // ставить куку в postHandle может быть слишком поздно, ставим её тут
        // сейчас ставим её безусловно, возможна оптимизация - ставим только если кука скоро протухнет
        setCookie(response, uid);

        CsrfCheck csrfCheck = handlerMethod.getMethod().getAnnotation(CsrfCheck.class);

        if (csrfCheck != null) {
            if (!csrfCheck.enabled()) {
                return true;
            }
        } else {
            if (HttpMethod.GET.equals(request.getMethod())) {
                return true;
            }
        }

        String csrfTokenHeaderValue = request.getHeader(X_CSRF_TOKEN_HEADER);
        String csrfTokenParameterValue = request.getParameter(CSRF_TOKEN_PARAMETER_NAME);
        String csrfTokenValue = csrfTokenHeaderValue != null ? csrfTokenHeaderValue : csrfTokenParameterValue;

        boolean csrfIsValid = csrfTokenValue != null && csrfValidator.checkCsrfToken(csrfTokenValue, uid);
        if (!csrfIsValid) {
            if (env == EnvironmentType.DEVELOPMENT && logger.isDebugEnabled()) {
                logger.debug("wrong csrf for request; correct one is {}", csrfValidator.createCsrfToken(uid));
            }
            throw new CsrfValidationFailureException("csrf is invalid");
        }
        return true;
    }

    private void setCookie(HttpServletResponse response, long uid) {
        Cookie cookie = new Cookie(CSRF_TOKEN_COOKIE_NAME, csrfValidator.createCsrfToken(uid));
        cookie.setSecure(true);
        cookie.setPath("/");
        response.addCookie(cookie);
    }
}
