package ru.yandex.direct.web.core.security.netacl;

import java.net.InetAddress;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.common.net.NetworkName;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.core.security.SecurityTranslations;
import ru.yandex.direct.env.EnvironmentType;

/**
 * Разрешает, либо отклоняет доступ к API-endpoint на основании присутствующих {@link AllowNetworks}.
 * Наибольший приоритет у аннотации на методе. Если аннотации на методе нет - используется аннотация класса.
 * Доступ не ограничивается если ни у метода, ни у класса нет аннотации {@link AllowNetworks}.
 * Родительские классы на наличие аннотаций не проверяются.
 */
public class NetAclInterceptor extends HandlerInterceptorAdapter {
    private static final Logger logger = LoggerFactory.getLogger(NetAclInterceptor.class);

    private final NetAcl netAcl;
    private final EnvironmentType env;
    private final List<String> extendedNetworks;

    public NetAclInterceptor(NetAcl netAcl, EnvironmentType env, List<String> extendedNetworks) {
        this.netAcl = netAcl;
        this.env = env;
        this.extendedNetworks = extendedNetworks;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler)
            throws Exception {
        if (!(handler instanceof HandlerMethod)) {
            return true;
        }
        HandlerMethod handlerMethod = ((HandlerMethod) handler);
        AllowNetworks networksRestriction = handlerMethod.getMethod().getAnnotation(AllowNetworks.class);
        if (networksRestriction == null) {
            networksRestriction = handlerMethod.getBeanType().getAnnotation(AllowNetworks.class);
        }

        if (networksRestriction != null) {
            List<String> networkNames = Arrays.stream(networksRestriction.value())
                    .map(NetworkName::networkName)
                    .collect(Collectors.toList());

            if (env.isBeta() || env.isTesting()) {
                networkNames.addAll(extendedNetworks);
            }

            InetAddress clientAddress = HttpUtil.getRemoteAddress(request);
            if (!netAcl.isIpInNetworks(clientAddress, networkNames)) {
                logger.error("Acl check failed for ip: {}, networks: {}", clientAddress, networkNames);
                throw new AccessDeniedException(
                        String.format("Операцию можно выполнить только из следующих сетей: %s", networkNames),
                        SecurityTranslations.INSTANCE.accessDenied());
            }
        }
        return true;
    }
}
