package ru.yandex.direct.web.core.semaphore;

import java.time.Duration;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.RandomUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.redislock.DistributedLock;
import ru.yandex.direct.redislock.DistributedLockException;
import ru.yandex.direct.redislock.lettuce.LettuceLockBuilder;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;

import static ru.yandex.direct.common.configuration.RedisConfiguration.LETTUCE;
import static ru.yandex.direct.common.db.PpcPropertyNames.DEFAULT_INLFY_REQUESTS_LIMIT;
import static ru.yandex.direct.common.db.PpcPropertyNames.INFLY_REQUESTS_SHOW_CAPTCHA;

/**
 * Для защиты от большого числа одновременно выполняющихся запросов (которые, например, грузят базу)
 * берем семафор в Redis на оператора. Если семафор взять не получилось,
 * то для снижения нагрузки от клиента — безусловно показываем ему капчу.
 * <p>
 * Регулируется работа двумя пропертями:
 * <ul>
 *     <li>{@link DEFAULT_INLFY_REQUESTS_LIMIT} — значение семафора для каждого оператора.
 *     если не задано или меньше 1 — семафор в редисе не берется</li>
 *     <li>{@link INFLY_REQUESTS_SHOW_CAPTCHA} — булево значение, показывать ли капчу при превышении семафора
 *     (можно экстренно выключить капчу, но оставить семафоры — чтобы по логам понимать их утилизацию)</li>
 * </ul>
 */
@Component
public class RedisSemaphoreInterceptor extends HandlerInterceptorAdapter implements AutoCloseable {
    public static final String SHOULD_SEE_CAPTCHA_ENTRY_NAME = "SHOW_CAPTCHA_ON_REQUESTS_LIMIT_REACHED";
    private static final String REQUEST_LOCK_ENTRY_NAME = RedisSemaphoreInterceptor.class.getName()
            + ".SIMULTANEOUS_CONN_LIMIT_LOCK";
    private static final Logger logger = LoggerFactory.getLogger(RedisSemaphoreInterceptor.class);
    private static final int DISABLED = -1;
    private final DirectWebAuthenticationSource authenticationSource;
    private final LettuceConnectionProvider lettuce;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final Timer timer;
    private final AtomicInteger limit;
    private final AtomicBoolean captchaEnabled;
    private final String lockKeyPrefix;

    @Autowired
    public RedisSemaphoreInterceptor(
            DirectWebAuthenticationSource directWebAuthenticationSource,
            @Qualifier(LETTUCE) LettuceConnectionProvider lettuce,
            @Value("${web-connections-semaphore.lock-key-prefix}") String lockKeyPrefix,
            PpcPropertiesSupport ppcPropertiesSupport
    ) {
        this.authenticationSource = directWebAuthenticationSource;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.lettuce = lettuce;
        this.lockKeyPrefix = lockKeyPrefix;


        limit = new AtomicInteger(DISABLED);
        captchaEnabled = new AtomicBoolean();

        long delay = RandomUtils.nextLong(100, 60 * 1000);
        logger.debug("Start settings refresher timer with delay {}ms", delay);
        timer = new Timer(getClass().getSimpleName() + "-Timer", true);
        timer.schedule(new LimitLoader(), delay, Duration.ofMinutes(2).toMillis());
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        int semaphoreLimit = limit.get();
        if (semaphoreLimit <= 0) {
            logger.trace("semaphore limit is not set");
            return true;
        }

        String lockName;
        String logId;
        if (authenticationSource.isAuthenticated()) {
            DirectAuthentication authentication = authenticationSource.getAuthentication();
            User operator = authentication.getOperator();
            lockName = operator.getUid().toString();
            logId = "uid=" + operator.getUid() + " login=" + operator.getLogin();
        } else {
            lockName = HttpUtil.getRemoteIp(request);
            logId = "ip=" + lockName;
        }
        // TODO: подтягивать переопределение лимита из настроек

        DistributedLock lock = createLock(lockName, semaphoreLimit);


        boolean locked;
        try {
            locked = lock.tryLock();
        } catch (DistributedLockException ex) {
            logger.error("cant acquire semaphore: lock service exception", ex);
            return true;
        }
        if (locked) {
            request.setAttribute(REQUEST_LOCK_ENTRY_NAME, lock);
        } else {
            logger.info("Forcing captcha for {} due to requests limit {} reached", logId, semaphoreLimit);
            request.setAttribute(SHOULD_SEE_CAPTCHA_ENTRY_NAME, captchaEnabled.get());
        }

        return true;
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex)
            throws Exception {
        DistributedLock lock = (DistributedLock) request.getAttribute(REQUEST_LOCK_ENTRY_NAME);
        request.removeAttribute(REQUEST_LOCK_ENTRY_NAME);
        if (lock != null) {
            try {
                lock.unlock();
                logger.debug("Lock object {} unlocked", lock);
            } catch (DistributedLockException lockEx) {
                logger.error("cant release semaphore lock", lockEx);
            }
        }
    }

    @Override
    public void close() throws Exception {
        timer.cancel();
    }

    private DistributedLock createLock(String lockKey, int maxLocks) {
        LettuceLockBuilder lockBuilder = LettuceLockBuilder.newBuilder(lettuce::getConnection)
                .withKeyPrefix(lockKeyPrefix)
                .withLockAttemptTimeout(0)
                .withTTL(Duration.ofMinutes(10).toMillis());    // берем по максимально допустимому таймауту запроса
        return lockBuilder.createLock(lockKey, maxLocks);
    }

    private class LimitLoader extends TimerTask {
        private final PpcProperty<Integer> limitProperty;
        private final PpcProperty<Boolean> showCaptchaProperty;

        private LimitLoader() {
            this.limitProperty = ppcPropertiesSupport.get(DEFAULT_INLFY_REQUESTS_LIMIT);
            this.showCaptchaProperty = ppcPropertiesSupport.get(INFLY_REQUESTS_SHOW_CAPTCHA);
        }

        @Override
        public void run() {
            try {
                Integer newValue = limitProperty.getOrDefault(DISABLED);
                Boolean enabled = showCaptchaProperty.getOrDefault(Boolean.FALSE);

                int oldValue = limit.getAndSet(newValue);
                boolean oldEnabled = captchaEnabled.getAndSet(enabled);
                if (oldValue != newValue || oldEnabled != enabled) {
                    logger.info("Set new infly requests limit value to {}, captcha enabled {}", newValue, enabled);
                } else {
                    logger.debug("Settings not changed: requests limit {}, captcha enabled {}", oldValue, oldEnabled);
                }
            } catch (RuntimeException e) {
                logger.error("Failed to refresh settings", e);
            }
        }
    }

}
