package ru.yandex.direct.web.core.semaphore.jvm;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicLong;

import one.util.streamex.EntryStream;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

/**
 * Позволяет разграничить доступ к ресурсу нескольким сервисам
 * Каждому сервису в конфиге сопостовляется семафор с определенным лимитом
 */

public class JvmSemaphore {

    // service -> semaphore_name
    private final Map<String, String> mapping;
    // semaphore_name -> semaphore
    private final Map<String, Semaphore> semaphores;
    private final Map<String, Stats> stats;

    public JvmSemaphore(DirectConfig config) {
        DirectConfig semCfg = config.getBranch("semaphores");
        semaphores = new HashMap<>();
        stats = new HashMap<>();
        semCfg.entrySet().stream()
                .map(Map.Entry::getKey)
                .forEach(semName -> {
                    int limit = semCfg.getInt(semName);
                    semaphores.put(semName, new Semaphore(limit));
                    stats.put(semName, new Stats(limit));
                });

        DirectConfig mapCfg = config.getBranch("mapping");
        mapping = EntryStream.of(mapCfg.entrySet().iterator())
                .keys()
                .toMap(mapCfg::getString);

        List<String> unexistentSemaphores =
                mapping.values().stream().filter(n -> !semaphores.containsKey(n)).collect(toList());
        if (!unexistentSemaphores.isEmpty()) {
            throw new IllegalStateException("No such semaphores: " + String.join(", ", unexistentSemaphores));
        }

        for (Map.Entry<String, Semaphore> entry : semaphores.entrySet()) {
            String semName = entry.getKey();
            Semaphore semaphore = entry.getValue();
            Stats semStats = stats.get(semName);

            MetricRegistry semaphoreRegistry = SOLOMON_REGISTRY.subRegistry("jvm_semaphore", semName);
            semaphoreRegistry.lazyGaugeInt64("limit", () -> semStats.limit);
            semaphoreRegistry.lazyGaugeInt64("used", () -> semStats.limit - semaphore.availablePermits());
            semaphoreRegistry.lazyGaugeDouble("errors", semStats.errors::doubleValue);
        }
    }

    /**
     * @param serviceName имя сервиса, который хочет получиьт доступ к ресурсу
     *                    метод проверяет, не исчерпан ли лимит доступа для сервиса, и увеличивает статистику обращения к семафору
     *                    если для сервиса не установлено ограничений, возвращает null
     *                    если лимит не исчерпан, возвращает Semaphore, который освобождается извне
     *                    если лимит исчерпан, кидает JvmSemaphoreException
     */

    public Semaphore tryAcquireAndIncrementStat(String serviceName) {
        String semaphoreName = mapping.get(serviceName);
        if (semaphoreName == null) {
            return null;
        }

        Semaphore sem = semaphores.get(semaphoreName);
        if (sem.tryAcquire()) {
            stats.get(semaphoreName).acquired.incrementAndGet();
            return sem;
        } else {
            stats.get(semaphoreName).errors.incrementAndGet();
            throw new JvmSemaphoreException(
                    "semaphore " + semaphoreName + " has a limit of " + stats.get(semaphoreName).limit);
        }
    }

    /**
     * Класс для сбора статистики обращения к семафору
     */

    static public class Stats {
        final int limit;
        AtomicLong acquired = new AtomicLong();
        AtomicLong errors = new AtomicLong();

        Stats(int limit) {
            this.limit = limit;
        }
    }
}
