package ru.yandex.direct.web.core.swagger;

import javax.servlet.http.HttpServletRequest;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.servlet.ModelAndView;
import springfox.documentation.annotations.ApiIgnore;

import ru.yandex.direct.web.core.configuration.SwaggerConfiguration;


/**
 * Springfox Swagger сервис привязан к URL {@code /swagger-resource}
 * ({@link springfox.documentation.swagger.web.ApiResourceController}).
 * <p>
 * Чтобы вся документация была доступна через URL, указанный в конфигурации как "{@code springfox.documentation.swagger.ui.baseurl}"
 * используем механизм forwarding'а Spring'а (<a href="http://www.baeldung.com/spring-redirect-and-forward">подробнее</a>).
 *
 * @see SwaggerConfiguration
 */
@Controller
@ApiIgnore
@RequestMapping("${springfox.documentation.swagger.ui.baseurl}")
public class SwaggerApiRedirectController {

    /**
     * Возможно, эту константу стоит сделать final-переменной и тянуть из аннотации {@link RequestMapping} класса
     * {@link springfox.documentation.swagger.web.ApiResourceController}
     */
    private final String swaggerApiDocUrl;

    @Autowired
    public SwaggerApiRedirectController(@Value("${springfox.documentation.swagger.v2.path}") String swaggerApiDocUrl) {
        this.swaggerApiDocUrl = swaggerApiDocUrl;
    }

    /**
     * JS код swagger-ui пытается искать API docs по относительному URL (/docs/docs/api в нашем случае)
     * Этот redirect осуществляет проброс этого запроса на правильный endpoint
     */
    @RequestMapping("${springfox.documentation.swagger.v2.path}")
    public ModelAndView redirectToDocsApi(ModelMap model, HttpServletRequest req) {
        return new ModelAndView("forward:" + req.getServletPath() + swaggerApiDocUrl, model);
    }
}
