package ru.yandex.direct.web.core.swagger;

import java.io.IOException;
import java.nio.charset.Charset;

import javax.servlet.http.HttpServletRequest;

import com.google.common.io.Resources;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.MediaType;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.ModelAndView;
import springfox.documentation.annotations.ApiIgnore;

import ru.yandex.direct.common.spring.TestingComponent;
import ru.yandex.direct.web.core.configuration.SwaggerConfiguration;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.core.security.csrf.CsrfValidator;

import static com.google.common.io.Resources.getResource;


/**
 * Springfox Swagger сервис привязан к URL {@code /swagger-resource}
 * ({@link springfox.documentation.swagger.web.ApiResourceController}).
 * <p>
 * Чтобы вся документация была доступна через URL, указанный в конфигурации как "{@code springfox.documentation.swagger.ui.baseurl}"
 * используем механизм forwarding'а Spring'а (<a href="http://www.baeldung.com/spring-redirect-and-forward">подробнее</a>).
 *
 * @see SwaggerConfiguration
 */
@ApiIgnore
@RequestMapping("${springfox.documentation.swagger.ui.baseurl}")
@TestingComponent
public class SwaggerRedirectController {

    /**
     * Возможно, эту константу стоит сделать final-переменной и тянуть из аннотации {@link RequestMapping} класса
     * {@link springfox.documentation.swagger.web.ApiResourceController}
     */
    public static final String SWAGGER_RESOURCES_MAPPING_PREFIX = "swagger-resources";

    private final CsrfValidator csrfValidator;
    private final DirectWebAuthenticationSource authenticationSource;
    private final String swaggerUiBaseUrl;

    @Autowired
    public SwaggerRedirectController(CsrfValidator csrfValidator,
                                     DirectWebAuthenticationSource authenticationSource,
                                     @Value("${springfox.documentation.swagger.ui.baseurl}") String swaggerUiBaseUrl,
                                     @Value("${springfox.documentation.swagger.v2.path}") String swaggerApiDocUrl) {
        this.csrfValidator = csrfValidator;
        this.authenticationSource = authenticationSource;
        this.swaggerUiBaseUrl = swaggerUiBaseUrl;
    }


    @RequestMapping("/" + SWAGGER_RESOURCES_MAPPING_PREFIX + "/**")
    public ModelAndView redirectToSwagger(ModelMap model, HttpServletRequest req) {
        String servletPath = req.getServletPath();
        //Редирект на основе @PathVariable сделать не удается, см. "https://jira.spring.io/browse/SPR-12546".
        String relativeSwaggerUrl =
                StringUtils.removeStart(req.getRequestURI(), req.getContextPath() + servletPath + swaggerUiBaseUrl);
        return new ModelAndView("forward:" + servletPath + relativeSwaggerUrl, model);
    }

    //todo подумать над нормальной шаблонизацией

    /**
     * отвечает сгенеренной html, вместо стандартной swagger-ui.html из ресурсов springfox-swagger-ui
     * так же на страницу добавлен js код, который устанавливает хедер с csrf токеном
     */
    @RequestMapping(value = "/swagger-ui.html", produces = MediaType.TEXT_HTML_VALUE)
    @ResponseBody
    public String swaggerPage() throws IOException {
        long uid = authenticationSource.getAuthentication().getOperator().getUid();
        String html = Resources.toString(getResource("META-INF/resources/swagger-ui.tmpl"), Charset.defaultCharset());
        return html.replace("${csrfTokenValue}", csrfValidator.createCsrfToken(uid));
    }
}
