package springfox.documentation.swagger2.mappers;

import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import com.fasterxml.classmate.ResolvedType;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Multimap;
import io.swagger.annotations.ApiModel;
import io.swagger.models.Model;
import io.swagger.models.ModelImpl;
import io.swagger.models.RefModel;
import io.swagger.models.properties.AbstractNumericProperty;
import io.swagger.models.properties.ArrayProperty;
import io.swagger.models.properties.MapProperty;
import io.swagger.models.properties.ObjectProperty;
import io.swagger.models.properties.Property;
import io.swagger.models.properties.StringProperty;
import springfox.documentation.schema.ModelProperty;
import springfox.documentation.schema.ModelReference;
import springfox.documentation.service.AllowableRangeValues;
import springfox.documentation.service.AllowableValues;
import springfox.documentation.service.ApiListing;

import static com.google.common.base.Predicates.not;
import static com.google.common.collect.Maps.filterEntries;
import static com.google.common.collect.Maps.newHashMap;
import static springfox.documentation.schema.Maps.isMapType;
import static springfox.documentation.swagger2.mappers.EnumMapper.maybeAddAllowableValues;
import static springfox.documentation.swagger2.mappers.EnumMapper.safeDouble;
import static springfox.documentation.swagger2.mappers.Properties.defaultOrdering;
import static springfox.documentation.swagger2.mappers.Properties.itemTypeProperty;
import static springfox.documentation.swagger2.mappers.Properties.property;
import static springfox.documentation.swagger2.mappers.Properties.voidProperties;

/**
 * <pre>
 * Костылек для поддержки полиморфизма в spring-fox.
 * Лечит генерацию клиента web api в https://github.yandex-team.ru/direct-qa/direct-web-api-client
 * </pre>
 * Было:
 * <pre>
 * <code>{
 * "CryptaGoalWeb": {
 * "type": "object",
 * "required": [
 * "classType",
 * "id",
 * "parent_id"
 * ],
 * "properties": {
 * "classType": {
 * "type": "string",
 * "description": "имя класса (CryptaGoalWeb/MetrikaGoalWeb)"
 * },
 * "description": {
 * "type": "string",
 * "description": "описание сегмента (с учетом языка пользователя)",
 * "readOnly": true
 * },
 * "id": {
 * "type": "integer",
 * "format": "int64"
 * },
 * "name": {
 * "type": "string",
 * "description": "название сегмента (с учетом языка пользователя)",
 * "readOnly": true
 * },
 * "parent_id": {
 * "type": "integer",
 * "format": "int64",
 * "description": "ID родительского сегмента крипты (для тех, у кого нет родителя = 0)"
 * },
 * "type": {
 * "type": "string",
 * "readOnly": true,
 * "enum": [
 * "goal",
 * "segment",
 * "ecommerce",
 * "audience",
 * "social_demo",
 * "family",
 * "interests"
 * ]
 * }
 * }
 * }
 * }
 * </code>
 * </pre>
 * <p>
 * Стало:
 * <pre>
 * <code>{
 * "CryptaGoalWeb": {
 * "allOf": [
 * {
 * "$ref": "#/definitions/AbstractGoalWeb"
 * },
 * {
 * "type": "object",
 * "required": [
 * "parent_id"
 * ],
 * "properties": {
 * "description": {
 * "type": "string",
 * "description": "описание сегмента (с учетом языка пользователя)",
 * "readOnly": true
 * },
 * "parent_id": {
 * "type": "integer",
 * "format": "int64",
 * "description": "ID родительского сегмента крипты (для тех, у кого нет родителя = 0)"
 * }
 * }
 * }
 * ]
 * }
 * }
 * </code>
 * </pre>
 * <p>
 * Удалить, когда допилят https://github.com/springfox/springfox/pull/1152
 */
public class DirectModelMapperImpl extends ModelMapper {
    public Map<String, Model> mapModels(Map<String, springfox.documentation.schema.Model> from) {
        if (from == null) {
            return null;
        }

        Map<String, Model> map = new HashMap<>();

        for (Map.Entry<String, springfox.documentation.schema.Model> entry : from.entrySet()) {
            String key = entry.getKey();
            Model value = mapProperties(entry.getValue());
            // start of workaround
            final ApiModel apiModel = entry.getValue().getType().getErasedType().getDeclaredAnnotation(ApiModel.class);
            if (apiModel != null && apiModel.parent() != Void.class) {
                final springfox.documentation.schema.Model parent = from.get(apiModel.parent().getSimpleName());

                final Map<String, Property> properties = value.getProperties();
                parent.getProperties().forEach((k, v) -> properties.remove(k));

                ModelImpl hacked = new ModelImpl();
                hacked.getVendorExtensions()
                        .put("allOf", Arrays.asList(new RefModel("#/definitions/" + parent.getId()), value));
                value = hacked;
            }
            // end of workaround
            map.put(key, value);
        }

        return map;
    }

    private Model mapProperties(springfox.documentation.schema.Model source) {
        ModelImpl model = new ModelImpl()
                .description(source.getDescription())
                .discriminator(source.getDiscriminator())
                .example(source.getExample())
                .name(source.getName());

        // start of workaround
        final ApiModel apiModel = source.getType().getErasedType().getDeclaredAnnotation(ApiModel.class);
        if (apiModel != null && !apiModel.discriminator().isEmpty()) {
            model.discriminator(apiModel.discriminator());
        }
        // end of workaround

        SortedMap<String, ModelProperty> sortedProperties = sort(source.getProperties());
        Map<String, Property> modelProperties = mapProperties(sortedProperties);
        model.setProperties(modelProperties);

        FluentIterable<String> requiredFields = FluentIterable.from(source.getProperties().values())
                .filter(requiredProperty())
                .transform(propertyName());
        model.setRequired(requiredFields.toList());
        model.setSimple(false);
        model.setType(ModelImpl.OBJECT);
        if (isMapType(source.getType())) {
            Optional<Class> clazz = typeOfValue(source);
            if (clazz.isPresent()) {
                model.additionalProperties(property(clazz.get().getSimpleName()));
            } else {
                model.additionalProperties(new ObjectProperty());
            }
        }
        return model;
    }

    private Map<String, Property> mapProperties(SortedMap<String, ModelProperty> properties) {
        Map<String, Property> mappedProperties = new LinkedHashMap<>();
        SortedMap<String, ModelProperty> nonVoidProperties = filterEntries(properties, not(voidProperties()));
        for (Map.Entry<String, ModelProperty> propertyEntry : nonVoidProperties.entrySet()) {
            mappedProperties.put(propertyEntry.getKey(), mapProperty(propertyEntry.getValue()));
        }
        return mappedProperties;
    }

    /**
     * Returns a {@link TreeMap} where the keys are sorted by their respective property position values in ascending
     * order.
     *
     * @param modelProperties
     * @return
     */
    private SortedMap<String, ModelProperty> sort(Map<String, ModelProperty> modelProperties) {

        SortedMap<String, ModelProperty> sortedMap =
                new TreeMap<>(defaultOrdering(modelProperties));
        sortedMap.putAll(modelProperties);
        return sortedMap;
    }

    @VisibleForTesting
    Optional<Class> typeOfValue(springfox.documentation.schema.Model source) {
        Optional<ResolvedType> mapInterface = findMapInterface(source.getType());
        if (mapInterface.isPresent()) {
            if (mapInterface.get().getTypeParameters().size() > 0) {
                return Optional.of(mapInterface.get().getTypeParameters().get(1).getErasedType());
            }
            return Optional.of(Object.class);
        }
        return Optional.absent();
    }

    private Optional<ResolvedType> findMapInterface(ResolvedType type) {
        return Optional.fromNullable(type.findSupertype(Map.class));
    }

    private Property mapProperty(ModelProperty source) {
        Property property = modelRefToProperty(source.getModelRef());

        maybeAddAllowableValues(property, source.getAllowableValues());

        if (property instanceof ArrayProperty) {
            ArrayProperty arrayProperty = (ArrayProperty) property;
            maybeAddAllowableValues(arrayProperty.getItems(), source.getAllowableValues());
        }

        if (property instanceof AbstractNumericProperty) {
            AbstractNumericProperty numericProperty = (AbstractNumericProperty) property;
            AllowableValues allowableValues = source.getAllowableValues();
            if (allowableValues instanceof AllowableRangeValues) {
                AllowableRangeValues range = (AllowableRangeValues) allowableValues;
                numericProperty.maximum(safeDouble(range.getMax()));
                numericProperty.exclusiveMaximum(range.getExclusiveMax());
                numericProperty.minimum(safeDouble(range.getMin()));
                numericProperty.exclusiveMinimum(range.getExclusiveMin());
            }
        }

        if (property instanceof StringProperty) {
            StringProperty stringProperty = (StringProperty) property;
            AllowableValues allowableValues = source.getAllowableValues();
            if (allowableValues instanceof AllowableRangeValues) {
                AllowableRangeValues range = (AllowableRangeValues) allowableValues;
                stringProperty.maxLength(safeInteger(range.getMax()));
                stringProperty.minLength(safeInteger(range.getMin()));
            }
            if (source.getPattern() != null) {
                stringProperty.setPattern(source.getPattern());
            }
        }

        if (property != null) {
            property.setDescription(source.getDescription());
            property.setName(source.getName());
            property.setRequired(source.isRequired());
            property.setReadOnly(source.isReadOnly());
            property.setExample(source.getExample());
        }
        return property;
    }

    static Integer safeInteger(String doubleString) {
        try {
            return Integer.valueOf(doubleString);
        } catch (NumberFormatException e) {
            return null;
        }
    }

    static Property modelRefToProperty(ModelReference modelRef) {
        if (modelRef == null || "void".equalsIgnoreCase(modelRef.getType())) {
            return null;
        }
        Property responseProperty;
        if (modelRef.isCollection()) {
            responseProperty = new ArrayProperty(
                    maybeAddAllowableValues(itemTypeProperty(modelRef.itemModel().get()),
                            modelRef.getAllowableValues()));
        } else if (modelRef.isMap()) {
            responseProperty = new MapProperty(property(modelRef.itemModel().get()));
        } else {
            responseProperty = property(modelRef.getType());
        }

        maybeAddAllowableValues(responseProperty, modelRef.getAllowableValues());

        return responseProperty;
    }

    Map<String, Model> modelsFromApiListings(Multimap<String, ApiListing> apiListings) {
        Map<String, springfox.documentation.schema.Model> definitions = newHashMap();
        for (ApiListing each : apiListings.values()) {
            definitions.putAll(each.getModels());
        }
        return mapModels(definitions);
    }

    private Function<ModelProperty, String> propertyName() {
        return input -> input.getName();
    }

    private Predicate<ModelProperty> requiredProperty() {
        return input -> input.isRequired();
    }
}
