package ru.yandex.direct.ytcomponents.config;

import java.time.Duration;
import java.util.Collection;
import java.util.stream.Collectors;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.utils.FunctionalUtils;
import ru.yandex.direct.ytwrapper.model.YtCluster;

/**
 * Директ-специфичная конфигурация для работы с yt.
 * Перечень используемых кластеров и таблиц.
 */
public class DirectYtDynamicConfig {
    private final DirectConfig config;
    private final Tables tables;

    public DirectYtDynamicConfig(DirectConfig config) {
        this.config = config;
        this.tables = new Tables(config);
    }

    public Duration getClusterRefreshPeriod() {
        return config.getDuration("cluster-refresh-period");
    }

    public Collection<YtCluster> getClusters() {
        return getClustersByKey("supported-clusters");
    }

    public Collection<YtCluster> getOrderStatClusters() {
        return getClustersByKey("order-stat-clusters");
    }

    public Collection<YtCluster> getStatClusters() {
        return getClustersByKey("stat-clusters");
    }

    public Collection<YtCluster> getTestLogsClusters() {
        return getClustersByKey("test-logs-clusters");
    }

    public Collection<YtCluster> getCounterByDomainClusters() {
        return getClustersByKey("counter-by-domain-clusters");
    }

    public Collection<YtCluster> getAllCountersByDomainClusters() {
        return getClustersByKey("all-counters-by-domain-clusters");
    }

    public Collection<YtCluster> getCountersByDomainHitLogClusters() {
        return getClustersByKey("counters-by-domain-hit-log-clusters");
    }

    public Collection<YtCluster> getSitelinksImportClusters() {
        return getClustersByKey("sitelinks-import-clusters");
    }

    public Collection<YtCluster> getConversionsStatClusters() {
        return getClustersByKey("conversions-stat-clusters");
    }

    public Collection<YtCluster> getBrandSurveysClusters() {
        return getClustersByKey("brand-surveys-clusters");
    }

    public Collection<YtCluster> getMobileAppStatClusters() {
        return getClustersByKey("mobile-app-stat-clusters");
    }

    public Collection<YtCluster> getMobileEventsStatClusters() {
        return getClustersByKey("mobile-events-stat-clusters");
    }

    public Collection<YtCluster> getCategoriesClusters() {
        return getClustersByKey("categories-clusters");
    }

    public Collection<YtCluster> getPhraseMd5Clusters() {
        return getClustersByKey("phrase-md5-clusters");
    }

    public Collection<YtCluster> getBsExportClusters() {
        return getClustersByKey("bs-export-clusters");
    }

    public Collection<YtCluster> getCalltrackingNumberClicksClusters() {
        return getClustersByKey("calltracking-number-clicks-clusters");
    }

    public Collection<YtCluster> getCalltrackingPhonsWithoutReplacementsClusters() {
        return getClustersByKey("calltracking-phones-without-replacements-clusters");
    }

    public Collection<YtCluster> getCalltrackingExternalByDomainsClusters() {
        return getClustersByKey("calltracking-external-by-domain-clusters");
    }

    public Collection<YtCluster> getCommunicationPlatformClusters() {
        return getClustersByKey("communication-platform-clusters");
    }

    public Collection<YtCluster> getSuggestConversionPriceClusters() {
        return getClustersByKey("suggest-conversion-price-clusters");
    }

    public Collection<YtCluster> getMetrikaConvAdGoalsClusters() {
        return getClustersByKey("metrika-conv-ad-goals-clusters");
    }

    public Collection<YtCluster> getAssetsStatClusters() {
        return getClustersByKey("assets-stat-clusters");
    }

    public Collection<YtCluster> getBlrtClusters() {
        return getClustersByKey("blrt-clusters");
    }

    public Collection<YtCluster> getConvMainCountersClusters() {
        return getClustersByKey("conv-main-counters-clusters");
    }

    public Collection<YtCluster> getPostViewOfflineReportYtClusters() {
        return getClustersByKey("postview-offline-report-clusters");
    }

    public Collection<YtCluster> getActStatYtClusters() {
        return getClustersByKey("act-stat-clusters");
    }

    public Collection<YtCluster> getYtTestStatClusters() { return getClustersByKey("yt-test-stat-clusters"); }

    protected Collection<YtCluster> getClustersByKey(String confKey) {
        return config.getStringList(confKey).stream()
                .map(c -> c.replace("-", "_"))
                .map(String::toUpperCase)
                .map(YtCluster::valueOf)
                .collect(Collectors.toSet());
    }

    public Tables tables() {
        return tables;
    }

    public CheckSyncStateConfig checkSyncStateConfig() {
        return new CheckSyncStateConfig(config);
    }

    public static class Tables {
        private final DirectConfig tablesConfig;

        private Tables(DirectConfig config) {
            this.tablesConfig = config.getBranch("tables");
        }

        public DirectTables direct() {
            return new DirectTables(tablesConfig);
        }

        public YabsStatTables yabsStat() {
            return new YabsStatTables(tablesConfig);
        }

        public YabsStatTestTables yabsStatTestTables() { return new YabsStatTestTables(tablesConfig); }

        public YabsDictTables yabsDict() {
            return new YabsDictTables(tablesConfig);
        }

        public MetrikaConvAdGoalsTables metrikaConvAdGoals() {
            return new MetrikaConvAdGoalsTables(tablesConfig);
        }

        public RecommendationTables recommendations() {
            return new RecommendationTables(tablesConfig);
        }

        public SitelinksTables sitelinks() {
            return new SitelinksTables(tablesConfig);
        }

        public CommunicationPlatformTables communicationPlatform() {
            return new CommunicationPlatformTables(tablesConfig);
        }

        public BlrtTables blrt() {
            return new BlrtTables(tablesConfig);
        }

        public ConvMainCountersTables convMainCountersTables() {
            return new ConvMainCountersTables(tablesConfig);
        }

        public ActMoneyStatsTables actMoneyStatsTables() {
            return new ActMoneyStatsTables(tablesConfig);
        }
    }

    public static class DirectTables {
        private final DirectConfig directTablesConfig;

        private DirectTables(DirectConfig tablesConfig) {
            this.directTablesConfig = tablesConfig.getBranch("direct");
        }

        public String baseDirectoryPath() {
            return directTablesConfig.getString("prefix");
        }

        public String baseContentDirectoryPath() {
            return directTablesConfig.getString("combined-prefix");
        }

        public String syncStatesTablePath() {
            return directTablesConfig.getString("sync-states");
        }

        public String campaignsTablePath() {
            return directTablesConfig.getString("campaigns");
        }

        public String phrasesTablePath() {
            return directTablesConfig.getString("phrases");
        }

        public String bidsTablePath() {
            return directTablesConfig.getString("bids");
        }

        public String bidsRetargetingTablePath() {
            return directTablesConfig.getString("bids-retargeting");
        }

        public String bannersTablePath() {
            return directTablesConfig.getString("banners");
        }

        public String recommendationsStatusTablePath() {
            return directTablesConfig.getString("recommendations-status");
        }

        public String retargetingConditionsTablePath() {
            return directTablesConfig.getString("retargeting-conditions");
        }

        public String bsExportlogsTablePath() {
            return directTablesConfig.getString("bsexport-logs");
        }

        public String brandSurveysTablePath() {
            return directTablesConfig.getString("brand-surveys");
        }

        public String categoriesWhitelistTablePath() {
            return directTablesConfig.getString("categories-whitelist");
        }

        public String categoriesBlockedDomainsTablePath() {
            return directTablesConfig.getString("categories-blocked-domains");
        }

        public String postViewOfflineReportTasksTablePath() {
            return directTablesConfig.getString("postview-offline-report-tasks");
        }

        public String postViewOfflineReportReportsDirectoryPath() {
            return directTablesConfig.getString("postview-offline-report-reports-directory");
        }

        public String counterByDomainTablePath() {
            return directTablesConfig.getString("counter-by-domain");
        }

        public String allCountersByDomainTablePath() {
            return directTablesConfig.getString("all-counters-by-domain");
        }

        public String countersByDomainHitLogTablePath() {
            return directTablesConfig.getString("counters-by-domain-hit-log");
        }

        public String calltrackingNumberClicksTablePath() {
            return directTablesConfig.getString("calltracking-number-clicks");
        }

        public String calltrackingPhonesWithoutReplacementsTablePath() {
            return directTablesConfig.getString("calltracking-phones-without-replacements");
        }

        public String calltrackingExternalByDomainTablePath() {
            return directTablesConfig.getString("calltracking-external-by-domain");
        }

        public String suggestConversionPriceTablePath() {
            return directTablesConfig.getString("conversion-price-recommendation");
        }
    }

    public static class YabsStatTables {
        private final DirectConfig yabsStatTablesConfig;

        private YabsStatTables(DirectConfig tablesConfig) {
            this.yabsStatTablesConfig = tablesConfig.getBranch("yabs-stat");
        }

        public String taxHistoryTablePath() {
            return yabsStatTablesConfig.getString("tax-history");
        }

        public String phrasesTablePath() {
            return yabsStatTablesConfig.getString("phrases");
        }

        public String ordersTablePath() {
            return yabsStatTablesConfig.getString("orders");
        }

        public String orderStatDayTablePath() {
            return yabsStatTablesConfig.getString("order-stat-day");
        }

        public String orderStatFraudTablePath() {
            return yabsStatTablesConfig.getString("order-stat-fraud");
        }

        public String conversionsTablePath() {
            return yabsStatTablesConfig.getString("conversions");
        }

        public String phraseGoalsTablePath() {
            return yabsStatTablesConfig.getString("phrase-goals");
        }

        public String orderGoalsTablePath() {
            return yabsStatTablesConfig.getString("order-goals");
        }

        public String orderStatTablePath() {
            return yabsStatTablesConfig.getString("order-stat");
        }

        public String brandSafetyFiltrationsTablePath() {
            return yabsStatTablesConfig.getString("brand-safety-filtrations");
        }

        public String brandSafetyStatTablePath() {
            return yabsStatTablesConfig.getString("brand-safety-stat");
        }

        public String brandSafetyCategoryStatTablePath() {
            return yabsStatTablesConfig.getString("brand-safety-category-stat");
        }

        public String mobileAppStatTablePath() {
            return yabsStatTablesConfig.getString("mobile-app-stat");
        }

        public String mobileEventsStatTablePath() {
            return yabsStatTablesConfig.getString("mobile-events-stat");
        }

        public String assetsStatTablePath() {
            return yabsStatTablesConfig.getString("assets-stat");
        }

        public String offerStatTablePath() {
            return yabsStatTablesConfig.getString("offer-stat");
        }

        public String offerAttributesTablePath() {
            return yabsStatTablesConfig.getString("offer-attributes");
        }
    }

    public static class YabsDictTables {
        private final DirectConfig yabsDictTablesConfig;

        private YabsDictTables(DirectConfig tablesConfig) {
            this.yabsDictTablesConfig = tablesConfig.getBranch("yabs-dict");
        }

        public String orderInfoDictTablePath() {
            return yabsDictTablesConfig.getString("order-info");
        }

        public String currencyDictTablePath() {
            return yabsDictTablesConfig.getString("currency");
        }

        public String oldPhraseIdsTablePath() {
            return yabsDictTablesConfig.getString("old-phrase-ids");
        }
    }

    public static class YabsStatTestTables {
        private final DirectConfig yabsStatTablesConfig;

        private YabsStatTestTables(DirectConfig tablesConfig) {
            this.yabsStatTablesConfig = tablesConfig.getBranch("yabs-stat-test");
        }

        public String ordersTestTablePath() {
            return yabsStatTablesConfig.getString("orders-test");
        }

        public String orderGoalsTestTablePath() {
            return yabsStatTablesConfig.getString("order-goals-test");
        }
    }

    public static class MetrikaConvAdGoalsTables {
        private final DirectConfig metrikaConvAdGoalsTableConfig;

        private MetrikaConvAdGoalsTables(DirectConfig tableConfig) {
            this.metrikaConvAdGoalsTableConfig = tableConfig.getBranch("metrika-conv-ad-goals");
        }

        public String metrikaConvAdGoalsTablePath() {
            return metrikaConvAdGoalsTableConfig.getString("conversion-goals");
        }

    }

    public static class CheckSyncStateConfig {
        private final DirectConfig checkSyncStateConfig;

        private CheckSyncStateConfig(DirectConfig tablesConfig) {
            this.checkSyncStateConfig = tablesConfig.getBranch("check-sync-state");
        }

        public Integer maxAttemptsToCheckGtidSet() {
            return checkSyncStateConfig.getInt("max-attempts-to-check-gtid-set");
        }

        public Duration sleepDurationBetweenTries() {
            return checkSyncStateConfig.getDuration("sleep-duration-between-tries");
        }
    }

    public static class RecommendationTables {
        private final DirectConfig config;

        private RecommendationTables(DirectConfig tablesConfig) {
            this.config = tablesConfig.getBranch("recommendations");
        }

        public String baseDir() {
            return config.getString("base-dir");
        }

        public String recommendationsTablePath() {
            return config.getString("recommendations");
        }
    }

    public static class SitelinksTables {
        private final DirectConfig config;

        private SitelinksTables(DirectConfig tablesConfig) {
            config = tablesConfig.getBranch("sitelinks");
        }

        public String sitelinksImportTablePath() {
            return config.getString("sitelinks-import");
        }
    }

    public static class CommunicationPlatformTables {
        private static final String EVENT_CONFIG_TABLE_NAME = "Events";
        private static final String EVENT_VERSIONS_CONFIG_TABLE_NAME = "EventVersions";
        private static final String SLOT_CONFIG_TABLE_NAME = "Slots";
        private final DirectConfig config;

        private CommunicationPlatformTables(DirectConfig tablesConfig) {
            config = tablesConfig.getBranch("communication-platform");
        }

        public String messagesTablePath() {
            return config.getString("messages");
        }

        public String eventConfigTablePath() {
            return config.getString("config-dir") + "/" + EVENT_CONFIG_TABLE_NAME;
        }

        public String eventVersionConfigTablePath() {
            return config.getString("config-dir") + "/" + EVENT_VERSIONS_CONFIG_TABLE_NAME;
        }

        public String slotConfigTablePath() {
            return config.getString("config-dir") + "/" + SLOT_CONFIG_TABLE_NAME;
        }

        public YtCluster writeConfigCluster() {
            return YtCluster.parse(config.getString("write-config-cluster"));
        }

        public Collection<YtCluster> readConfigClusters() {
            return FunctionalUtils.mapList(
                    config.getStringList("read-config-clusters"),
                    YtCluster::parse
            );
        }
    }

    public static class BlrtTables {
        private final DirectConfig blrtTablesConfig;

        private BlrtTables(DirectConfig tablesConfig) {
            blrtTablesConfig = tablesConfig.getBranch("blrt");
        }

        public String dynFeedToTasksTablePath() {
            return blrtTablesConfig.getString("dyn-feed-to-tasks");
        }

        public String perfFeedToTasksTablePath() {
            return blrtTablesConfig.getString("perf-feed-to-tasks");
        }
    }

    public static class ConvMainCountersTables {
        private final DirectConfig convMainCountersTableConfig;

        private ConvMainCountersTables(DirectConfig tablesConfig) {
            this.convMainCountersTableConfig = tablesConfig.getBranch("conv-main-counters");
        }

        public String  convMainCountersTablePath() {
            return convMainCountersTableConfig.getString("metrika-counters");
        }
    }

    public static class ActMoneyStatsTables {
        private final DirectConfig actStatTableConfig;

        private ActMoneyStatsTables(DirectConfig tablesConfig) {
            this.actStatTableConfig = tablesConfig.getBranch("act-stat");
        }

        public String actMoneyStatsTablePath() {
            return actStatTableConfig.getString("money-stats");
        }
    }

    public Duration getFreshnessThresholdForOrderStat() {
        return config.getDuration("freshness-threshold-for-order-stat");
    }
}
