package ru.yandex.direct.ytcomponents.repository;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.direct.ytcomponents.model.MysqlSyncStates;
import ru.yandex.direct.ytwrapper.exceptions.OperationRunningException;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.inside.yt.kosher.impl.common.YtException;

import static ru.yandex.direct.grid.schema.yt.Tables.DIRECT_GRID_MYSQL_SYNC_STATES;

/**
 * Умеет получать таймстемпы свежести данных mysql->yt синхронизатора по шардам.
 */
@ParametersAreNonnullByDefault
public class YtClusterFreshnessRepository {
    public static final Field<String> DBNAME_FIELD = DIRECT_GRID_MYSQL_SYNC_STATES.DBNAME;
    public static final Field<Long> LAST_TIMESTAMP_FIELD = DIRECT_GRID_MYSQL_SYNC_STATES.LAST_TIMESTAMP;
    public static final Field<String> GTID_SET_FIELD = DIRECT_GRID_MYSQL_SYNC_STATES.GTID_SET;

    private static final Logger logger = LoggerFactory.getLogger(YtClusterFreshnessRepository.class);

    private static final Duration SYNC_TABLE_SELECT_TIMEOUT = Duration.ofSeconds(4);

    private final String syncStatesTablePath;

    public YtClusterFreshnessRepository(DirectYtDynamicConfig dynamicConfig) {
        syncStatesTablePath = dynamicConfig.tables().direct().syncStatesTablePath();
    }

    /**
     * Загружает информацию о таймстемпах свежести синхронизируемых таблиц для каждого шарда.
     * Возвращает null, если по какой-либо причине информацию получить не удалось.
     *
     * @param ytOperator Коннектор к кластеру динтаблиц, у которого мы запрашиваем эту информацию
     * @return map shard -> lastSyncTimstamp или null, если кластер недоступен
     */
    @Nullable
    public Map<Integer, Long> loadShardToTimestamp(YtDynamicOperator ytOperator) {
        return loadShardToTimestamp(ytOperator, syncStatesTablePath);
    }

    @Nullable
    public Map<Integer, Long> loadShardToTimestamp(YtDynamicOperator ytOperator, Set<Integer> skipShards) {
        return loadShardToTimestamp(ytOperator, syncStatesTablePath, skipShards);
    }

    @Nullable
    public Map<Integer, Long> loadShardToTimestamp(YtDynamicOperator ytOperator, String syncStatesTablePath) {
        return loadShardToTimestamp(ytOperator, syncStatesTablePath, Set.of());
    }

    @Nullable
    public Map<Integer, Long> loadShardToTimestamp(YtDynamicOperator ytOperator, String syncStatesTablePath,
                                                   Set<Integer> skipShards) {
        Set<String> skipDbNames = skipShards.stream().map(shard -> String.format("ppc:%d", shard)).collect(Collectors.toSet());
        var timestampMap = loadShardToTimestamp(ytOperator, syncStatesTablePath, "ppc:\\d+", skipDbNames);
        return timestampMap
                .map(map -> EntryStream.of(map).mapKeys(YtClusterFreshnessRepository::dbNameToShard).toMap())
                .orElse(null);
    }

    @Nullable
    public Long loadPpcdictTimestamp(YtDynamicOperator ytOperator) {
        return loadPpcdictTimestamp(ytOperator, syncStatesTablePath);
    }

    @Nullable
    public Long loadPpcdictTimestamp(YtDynamicOperator ytOperator, String syncStatesTablePath) {
        var timestampMap = loadShardToTimestamp(ytOperator, syncStatesTablePath, "ppcdict", Set.of());
        return timestampMap.map(map -> map.get("ppcdict")).orElse(null);
    }

    private Optional<Map<String, Long>> loadShardToTimestamp(YtDynamicOperator ytOperator, String syncStatesTablePath,
                                                             String dbNameRegExp, Set<String> skipDbNames) {
        Optional<List<MysqlSyncStates>> syncStates = loadSyncStates(ytOperator, syncStatesTablePath, dbNameRegExp,
                skipDbNames);
        return syncStates.map(s -> StreamEx.of(s).toMap(MysqlSyncStates::getDbName, MysqlSyncStates::getLastTimestamp));
    }

    public Optional<List<MysqlSyncStates>> loadSyncStatesForAllShards(YtDynamicOperator ytOperator) {
        return loadSyncStates(ytOperator, syncStatesTablePath, "ppc:\\d+", Set.of());
    }

    private Optional<List<MysqlSyncStates>> loadSyncStates(YtDynamicOperator ytOperator, String syncStatesTablePath,
                                                           String dbNameRegExp, Set<String> skipDbNames) {
        List<MysqlSyncStates> syncStates;
        try {
            if (!skipDbNames.isEmpty()) {
                logger.info("skipping dbnames: {}", skipDbNames);
            }
            syncStates = ytOperator.selectRows(
                    String.format("%s, %s, %s FROM [%s]",
                            DBNAME_FIELD.getName(), LAST_TIMESTAMP_FIELD.getName(), GTID_SET_FIELD.getName(),
                            syncStatesTablePath), SYNC_TABLE_SELECT_TIMEOUT)
                    .getYTreeRows()
                    .stream()
                    .filter(n -> !skipDbNames.contains(n.getString(DBNAME_FIELD.getName())))
                    .filter(n -> n.getString(DBNAME_FIELD.getName()).matches(dbNameRegExp))
                    .map(n -> new MysqlSyncStates()
                            .withDbName(n.getString(DBNAME_FIELD.getName()))
                            .withLastTimestamp(n.getLong(LAST_TIMESTAMP_FIELD.getName()))
                            .withGtidSet(n.getString(GTID_SET_FIELD.getName())))
                    .collect(Collectors.toList());

        } catch (YtException | ClassCastException | NumberFormatException | ArrayIndexOutOfBoundsException e) {
            // Кластер недоступен или данные повреждены
            logger.error(String.format("YT cluster %s is unavailable", ytOperator.getCluster()), e);
            return Optional.empty();
        } catch (OperationRunningException e) {
            logger.warn(String.format("YT cluster %s is unavailable", ytOperator.getCluster()), e);
            return Optional.empty();
        }

        if (syncStates.isEmpty()) {
            // На кластере что-то не так
            logger.error("Sync states table {} is empty", syncStatesTablePath);
            return Optional.empty();
        }
        return Optional.of(syncStates);
    }

    public static int dbNameToShard(String dbName) {
        String numString = dbName.split(":")[1];
        return Integer.valueOf(numString);
    }
}
