package ru.yandex.direct.ytcomponents.service;

import java.time.Duration;
import java.time.ZonedDateTime;

import org.jooq.impl.TableImpl;

import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.direct.ytcomponents.repository.StatsDynClusterFreshnessRepository;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.dynamic.selector.ClusterFreshness;
import ru.yandex.direct.ytwrapper.dynamic.selector.FreshnessAndHashBasedClusterChooser;
import ru.yandex.direct.ytwrapper.dynamic.selector.QueryBasedClusterWeightFunction;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.utils.TableAvailabilityChecker;

public abstract class StatsDynContextProvider extends BaseDynContextProvider {

    protected StatsDynContextProvider(
            DirectYtDynamicConfig directYtDynamicConfig,
            YtProvider ytProvider,
            TableAvailabilityChecker tableAvailabilityChecker,
            StatsDynClusterFreshnessRepository statsDynClusterFreshnessRepository,
            TableImpl table,
            Duration timeout) {
        super(directYtDynamicConfig,
                ytProvider,
                () -> new StatClusterWeightFunc(tableAvailabilityChecker, statsDynClusterFreshnessRepository,
                        ytProvider, table),
                timeout);
    }


    /**
     * Весовая функция для определения веса кластера со статистикой.
     *
     * @see FreshnessAndHashBasedClusterChooser
     */
    private static class StatClusterWeightFunc extends QueryBasedClusterWeightFunction {
        // Смотрим на время обработки основного для нас лога — лога с показами и кликами
        // Свежие данные -- отстающие меньше, чем на N минут
        private final StatsDynClusterFreshnessRepository statsDynClusterFreshnessRepository;
        private static final Duration FRESH_DELAY = Duration.ofMinutes(60);

        private StatClusterWeightFunc(TableAvailabilityChecker tableAvailabilityChecker,
                                      StatsDynClusterFreshnessRepository statsDynClusterFreshnessRepository,
                                      YtProvider ytProvider, TableImpl table) {
            super(tableAvailabilityChecker, ytProvider, table);
            this.statsDynClusterFreshnessRepository = statsDynClusterFreshnessRepository;
        }

        @Override
        protected ClusterFreshness calcFreshness(YtCluster ytCluster) {
            // Подробнее про алгоритм определения свежести данных в кластере: https://st.yandex-team.ru/DIRECT-79301
            YtOperator ytOperator = ytProvider.getOperator(ytCluster);
            ZonedDateTime lastSyncTime =
                    statsDynClusterFreshnessRepository.getClusterFreshnessTimeForTable(ytOperator, table);

            ZonedDateTime now = ZonedDateTime.now();
            if (lastSyncTime.compareTo(now.minus(FRESH_DELAY)) > 0) {
                return ClusterFreshness.FRESH;
            } else {
                return ClusterFreshness.ROTTEN;
            }
        }
    }
}
