package ru.yandex.direct.ytcomponents.statistics.repository;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.Row2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.grid.schema.yt.tables.DirectgridstatBs;
import ru.yandex.direct.ytcomponents.model.CampaignDeal;
import ru.yandex.direct.ytcomponents.model.DealStatsResponse;
import ru.yandex.direct.ytcomponents.service.PhraseStatsDynContextProvider;
import ru.yandex.direct.ytcomponents.service.StatsDynContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;
import ru.yandex.yt.ytclient.wire.UnversionedValue;

import static org.jooq.impl.DSL.row;
import static org.jooq.impl.DSL.sum;
import static ru.yandex.direct.grid.schema.yt.Tables.DIRECTGRIDSTAT_BS;
import static ru.yandex.direct.ytwrapper.YtTableUtils.findColumnOrThrow;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.DECIMAL_MULT;

/**
 * Репозиторий, работающий с таблицей с позаказной статистикой
 */
@Repository
public class CampaignStatRepository {

    private static final DirectgridstatBs STAT = DIRECTGRIDSTAT_BS.as("S");

    private static final Field<Long> DEAL_ID = STAT.DEAL_ID.as("DealId");
    private static final Field<BigDecimal> SHOWS = sum(STAT.SHOWS).as("Shows");
    private static final Field<BigDecimal> CLICKS = sum(STAT.CLICKS).as("Clicks");
    private static final Field<BigDecimal> SPENT = sum(STAT.COST_TAX_FREE).as("Spent");

    private final StatsDynContextProvider dynContextProvider;

    @Autowired
    public CampaignStatRepository(PhraseStatsDynContextProvider dynContextProvider) {
        this.dynContextProvider = dynContextProvider;
    }

    public Map<Long, DealStatsResponse> getDealsStatistics(Collection<CampaignDeal> requests) {
        Collection<Row2<Long, Long>> idFilter =
                StreamEx.of(requests)
                        .map(r -> row(r.getCampaignId(), r.getDealId()))
                        .toList();

        UnversionedRowset rowset = dynContextProvider.getContext().executeSelect(
                YtDSL.ytContext()
                        .select(DEAL_ID, SHOWS, CLICKS, SPENT)
                        .from(STAT)
                        .where(row(STAT.EXPORT_ID, DEAL_ID).in(idFilter))
                        .groupBy(DEAL_ID)
        );

        TableSchema schema = rowset.getSchema();
        int dealIsColIdx = findColumnOrThrow(schema, DEAL_ID);
        int showsColIdx = findColumnOrThrow(schema, SHOWS);
        int clicksColIdx = findColumnOrThrow(schema, CLICKS);
        int spentColIdx = findColumnOrThrow(schema, SPENT);

        Function<UnversionedRow, DealStatsResponse> convertRow =
                row -> {
                    List<UnversionedValue> values = row.getValues();
                    return new DealStatsResponse()
                            .withDealId(values.get(dealIsColIdx).longValue())
                            .withShows(values.get(showsColIdx).longValue())
                            .withClicks(values.get(clicksColIdx).longValue())
                            .withSpent(BigDecimal.valueOf(values.get(spentColIdx).longValue()).divide(DECIMAL_MULT));
                };

        return StreamEx.of(rowset.getRows())
                .map(convertRow)
                .mapToEntry(DealStatsResponse::getDealId)
                .invert()
                .toMap();
    }
}
