package ru.yandex.direct.ytcore.entity.recommendation.service;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.log.container.LogType;
import ru.yandex.direct.common.log.service.CommonDataLogService;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.recommendation.RecommendationType;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationKey;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationLogInfo;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationQueueInfo;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationStatus;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationStatusInfo;
import ru.yandex.direct.core.entity.recommendation.repository.RecommendationOnlineRepository;
import ru.yandex.direct.core.entity.recommendation.repository.RecommendationQueueRepository;
import ru.yandex.direct.core.entity.recommendation.repository.RecommendationStatusRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.ytcore.entity.recommendation.service.typesupport.RecommendationTypeSupportDispatcher;

import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static ru.yandex.direct.core.entity.recommendation.RecommendationType.removePagesFromBlackList;
import static ru.yandex.direct.core.entity.recommendation.model.RecommendationStatus.CANCELLED;
import static ru.yandex.direct.core.entity.recommendation.model.RecommendationStatus.DONE;
import static ru.yandex.direct.core.entity.recommendation.model.RecommendationStatus.FAILED;
import static ru.yandex.direct.core.entity.recommendation.model.RecommendationStatus.IN_PROGRESS;
import static ru.yandex.direct.core.entity.recommendation.repository.RecommendationStatusRepository.ACTUALITY_DAYS_LIMIT;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@Component
@ParametersAreNonnullByDefault
public class RecommendationService {
    private static final Logger logger = LoggerFactory.getLogger(RecommendationService.class);
    private static final int MAX_RECOMMENDATATIONS_FOR_SYNCHRONOUS_APPLY = 30;

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final RecommendationQueueRepository recommendationQueueRepository;
    private final RecommendationStatusRepository recommendationStatusRepository;
    private final RecommendationOnlineRepository recommendationOnlineRepository;
    private final RecommendationTypeSupportDispatcher recommendationTypeSupportDispatcher;
    private RbacService rbacService;
    private CommonDataLogService commonDataLogService;

    @Autowired
    public RecommendationService(DslContextProvider dslContextProvider,
                                 ShardHelper shardHelper,
                                 CampaignRepository campaignRepository,
                                 RecommendationQueueRepository recommendationQueueRepository,
                                 RecommendationStatusRepository recommendationStatusRepository,
                                 RecommendationOnlineRepository recommendationOnlineRepository,
                                 RecommendationTypeSupportDispatcher recommendationTypeSupportDispatcher,
                                 RbacService rbacService, CommonDataLogService commonDataLogService) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.recommendationQueueRepository = recommendationQueueRepository;
        this.recommendationStatusRepository = recommendationStatusRepository;
        this.recommendationOnlineRepository = recommendationOnlineRepository;
        this.recommendationTypeSupportDispatcher = recommendationTypeSupportDispatcher;
        this.rbacService = rbacService;
        this.commonDataLogService = commonDataLogService;
    }

    /**
     * Добавляет рекомендации в очередь
     *
     * @param shard           шард
     * @param recommendations список рекомендаций
     */
    public void add(int shard, Collection<RecommendationQueueInfo> recommendations) {
        recommendationQueueRepository.add(shard, recommendations);

        List<RecommendationStatusInfo> recommendationStatuses = recommendations.stream()
                .map(recommendation -> convertToRecommendationStatusInfo(recommendation).withStatus(IN_PROGRESS))
                .collect(Collectors.toList());
        recommendationStatusRepository.add(shard, recommendationStatuses);
    }

    /**
     * Удаляет старые рекомендации
     *
     * @param shard шард
     */
    public void deleteOld(int shard) {
        long timestamp = LocalDateTime.now().minusDays(ACTUALITY_DAYS_LIMIT).toEpochSecond(ZoneOffset.UTC);
        int onlineDeletedRows = recommendationOnlineRepository.deleteOlderThan(shard, timestamp);
        int statusDeletedRows = recommendationStatusRepository.deleteOlderThan(shard, timestamp);
        if (onlineDeletedRows != 0 || statusDeletedRows != 0) {
            logger.info("Deleted {} rows from recommendations_online, {} rows from recommendations_status",
                    onlineDeletedRows, statusDeletedRows);
        }
    }

    /**
     * Применяет рекомендации или кладет их в очередь
     * Применяются только доступные для выполнения рекомендации (определяется из доступности кампаний)
     * Если рекомендаций меньше 30 - применять сразу, иначе - в очередь
     *
     * @param clientId        id клиента
     * @param uid             uid оператора
     * @param recommendations список рекомендаций
     */
    public Map<RecommendationQueueInfo, RecommendationStatus> execute(Long clientId,
                                                                      Long uid,
                                                                      Collection<RecommendationQueueInfo> recommendations) {
        int shard = shardHelper.getShardByClientId(ClientId.fromLong(clientId));

        Set<Long> campaignIds = recommendations.stream()
                .map(RecommendationQueueInfo::getCampaignId)
                .filter(Objects::nonNull)
                .collect(toSet());

        Set<Long> writableCampaignIds = rbacService.getWritableCampaigns(uid, campaignIds);
        Set<Long> archivedCampaignIds = campaignRepository.getArchivedCampaigns(shard, campaignIds);
        Set<RecommendationQueueInfo> availableRecommendations = recommendations.stream()
                .filter(rec -> rec.getCampaignId() == null || rec.getCampaignId() == 0
                        || (writableCampaignIds.contains(rec.getCampaignId())
                        && !archivedCampaignIds.contains(rec.getCampaignId())))
                .collect(toSet());

        List<RecommendationStatusInfo> recommendationStatuses = availableRecommendations.stream()
                .map(r -> convertToRecommendationStatusInfo(r).withStatus(IN_PROGRESS))
                .collect(Collectors.toList());
        int added = recommendationStatusRepository.add(shard, recommendationStatuses);

        // В таблице может быть запись со статусом "canselled"
        // В этом случае рекомендацию из обработки исключаем
        if (added < recommendationStatuses.size()) {
            recommendationStatuses.retainAll(recommendationStatusRepository.get(shard, recommendationStatuses));
        }

        Map<RecommendationQueueInfo, RecommendationStatus> statuses = new HashMap<>();

        if (canApplySynchronously(availableRecommendations)) {
            for (RecommendationQueueInfo recommendation : availableRecommendations) {
                RecommendationStatus newStatus = apply(shard, recommendation);
                statuses.put(recommendation, newStatus);
                recommendationStatusRepository.update(shard,
                        convertToRecommendationStatusInfo(recommendation).withStatus(newStatus));
            }
        } else {
            recommendationQueueRepository.add(shard, availableRecommendations);
            statuses = availableRecommendations.stream().collect(toMap(o -> o, o -> IN_PROGRESS));
        }

        return statuses;
    }

    // Применяем синхронно, если рекомендаций меньше граничного значения
    // и нет сложных для выполнения (пока удаление из черного списка)
    private boolean canApplySynchronously(Collection<RecommendationQueueInfo> recommendations) {
        //noinspection SimplifiableIfStatement
        if (recommendations.size() > MAX_RECOMMENDATATIONS_FOR_SYNCHRONOUS_APPLY) {
            return false;
        }

        return recommendations.stream()
                .map(r -> RecommendationType.fromId(r.getType()))
                .filter(t -> t == removePagesFromBlackList)
                .count() == 0;
    }

    /**
     * Применяет рекомендации из очереди
     * Несколько записей таблицы RECOMMENDATIONS_QUEUE захватывается текущим воркером
     * В таблице RECOMMENDATIONS_STATUS статус рекомендации меняется на IN_PROGRESS
     * Рекомендация применяется
     * В зависимости от результата статус рекомендации меняется на DONE или FAILED
     * Из таблицы RECOMMENDATIONS_QUEUE запись удаляется
     *
     * @param shard номер шарда
     * @param parId номер воркера
     */
    public void processFromQueue(int shard, int parId) {
        for (RecommendationQueueInfo recommendation : getForUpdate(shard, parId)) {
            RecommendationStatusInfo recommendationStatusInfo
                    = RecommendationService.convertToRecommendationStatusInfo(recommendation);
            recommendationStatusRepository.update(shard, recommendationStatusInfo.withStatus(IN_PROGRESS));

            RecommendationStatus newStatus = apply(shard, recommendation);

            // В транзакции изменяем статус в RECOMMENDATIONS_STATUS и удаляем из очереди RECOMMENDATIONS_QUEUE
            dslContextProvider.ppc(shard).transaction(configuration -> {
                DSLContext txContext = configuration.dsl();
                recommendationStatusRepository.update(txContext, recommendationStatusInfo.withStatus(newStatus));
                recommendationQueueRepository.delete(txContext, singleton(recommendation.getId()));
            });
        }
    }

    /**
     * Получает рекомендации для изменения
     * Берется несколько записей таблицы RECOMMENDATIONS_QUEUE с пустым значением parId
     * Затем для этих записей parId устанавливается значением текущего воркера
     * После этого получаются записи, где это значение удалось установить (другой воркер не сделал это раньше)
     *
     * @param shard номер шарда
     * @param parId номер воркера
     * @return {@link Collection} количество захваченных рекомендаций
     */
    private Collection<RecommendationQueueInfo> getForUpdate(int shard, int parId) {
        Collection<RecommendationQueueInfo> recommendations = recommendationQueueRepository.getPortion(shard, parId);
        List<Long> ids = recommendations.stream().map(RecommendationQueueInfo::getId).collect(toList());
        recommendationQueueRepository.lock(shard, parId, ids);
        return recommendationQueueRepository.getLocked(shard, parId);
    }

    /**
     * Применяет одну рекомендацию
     *
     * @param shard          номер шарда
     * @param recommendation рекомендация
     * @return {@link RecommendationStatus} DONE - успешно, FAILED - неуспешно
     */
    private RecommendationStatus apply(int shard, RecommendationQueueInfo recommendation) {
        boolean applied;
        try {
            logger.info("applying recommendation: {}", toJson(recommendation));
            applied = recommendationTypeSupportDispatcher
                    .getTypeSupport(RecommendationType.fromId(recommendation.getType())).apply(shard, recommendation);
        } catch (Exception e) {
            logger.warn("error during applying recommendation {}", toJson(recommendation), e);
            applied = false;
        }
        RecommendationStatus status = applied ? DONE : FAILED;

        commonDataLogService.log(LogType.RECOMMENDATIONS,
                singletonList(convertToRecommendationLogInfo(recommendation).withStatus(status)), null);

        return status;
    }

    static RecommendationStatusInfo convertToRecommendationStatusInfo(RecommendationKey recommendation) {
        return new RecommendationStatusInfo()
                .withType(recommendation.getType())
                .withClientId(defaultIfNull(recommendation.getClientId(), 0L))
                .withCampaignId(defaultIfNull(recommendation.getCampaignId(), 0L))
                .withAdGroupId(defaultIfNull(recommendation.getAdGroupId(), 0L))
                .withBannerId(defaultIfNull(recommendation.getBannerId(), 0L))
                .withUserKey1(defaultIfNull(recommendation.getUserKey1(), ""))
                .withUserKey2(defaultIfNull(recommendation.getUserKey2(), ""))
                .withUserKey3(defaultIfNull(recommendation.getUserKey3(), ""))
                .withTimestamp(defaultIfNull(recommendation.getTimestamp(), 0L));
    }

    private RecommendationLogInfo convertToRecommendationLogInfo(RecommendationQueueInfo recommendation) {
        return new RecommendationLogInfo()
                .withType(recommendation.getType())
                .withClientId(recommendation.getClientId())
                .withCampaignId(recommendation.getCampaignId())
                .withAdGroupId(recommendation.getAdGroupId())
                .withBannerId(recommendation.getBannerId())
                .withUserKey1(recommendation.getUserKey1())
                .withUserKey2(recommendation.getUserKey2())
                .withUserKey3(recommendation.getUserKey3())
                .withTimestamp(recommendation.getTimestamp())
                .withJsonData(recommendation.getJsonData());
    }

    /**
     * Проставляет указанным рекомендациям статус "cancelled", после чего они перестают отображаться в интерфейсе.
     * Возвращает кол-во добавленных/изменённых строк.
     */
    public int cancel(int shard, Set<RecommendationKey> keys) {
        if (keys.isEmpty()) {
            return 0;
        }

        return recommendationStatusRepository.add(shard,
                keys.stream()
                        .map(key -> convertToRecommendationStatusInfo(key).withStatus(CANCELLED))
                        .collect(toList()));
    }
}



