package ru.yandex.direct.ytcore.entity.recommendation.service.typesupport;

import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.recommendation.RecommendationType;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationQueueInfo;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchAddContainer;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchAddOperation;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.AddedModelId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;

/**
 * Включает автотаргетинг на группе
 */
@Component
@ParametersAreNonnullByDefault
public class RecommendationAutotargetingTypeSupport implements RecommendationTypeSupport {
    private static final Logger logger = LoggerFactory.getLogger(RecommendationAutotargetingTypeSupport.class);
    private final RelevanceMatchService relevanceMatchService;
    private final ClientService clientService;
    private final RecentStatisticsService recentStatisticsService;

    @Autowired
    public RecommendationAutotargetingTypeSupport(RelevanceMatchService relevanceMatchService,
                                                  ClientService clientService,
                                                  RecentStatisticsService recentStatisticsService) {
        this.relevanceMatchService = relevanceMatchService;
        this.clientService = clientService;
        this.recentStatisticsService = recentStatisticsService;
    }

    @Override
    public boolean apply(int shard, RecommendationQueueInfo recommendation) {
        Long campaignId = recommendation.getCampaignId();
        Long adGroupId = recommendation.getAdGroupId();
        ClientId clientId = ClientId.fromLong(recommendation.getClientId());
        Long uid = recommendation.getUid();

        MassResult<AddedModelId> result = switchAutotargetingOn(shard, clientId, adGroupId, uid);

        boolean success = result.getErrorCount() == 0;
        logger.info("autotargeting was " + (success ? "set" : "not set") + ": uid {}, clientId {}, cid {}, adgroup {}",
                uid, clientId, campaignId, adGroupId);
        if (!success) {
            logger.warn("Validation errors during applying autotargeting: {}",
                    result.getValidationResult().flattenErrors());
        }

        return success;
    }

    /**
     * Включает автотаргетинг для отдельной группы
     */
    private MassResult<AddedModelId> switchAutotargetingOn(int shard, ClientId clientId, Long adGroupId, long uid) {

        Currency currency = checkNotNull(clientService.getClient(clientId)).getWorkCurrency().getCurrency();
        RelevanceMatch relevanceMatch = new RelevanceMatch().withAdGroupId(adGroupId);

        RelevanceMatchAddContainer relevanceMatchAddOperationContainer =
                relevanceMatchService.createAddContainer(singletonList(relevanceMatch), uid, clientId, shard);

        ShowConditionAutoPriceParams autoPriceParams = new ShowConditionAutoPriceParams(
                ShowConditionFixedAutoPrices.ofGlobalFixedPrice(null), recentStatisticsService);

        RelevanceMatchAddOperation relevanceMatchAddOperation = relevanceMatchService.createFullAddOperation(currency,
                clientId, uid, singletonList(relevanceMatch), relevanceMatchAddOperationContainer,
                true, autoPriceParams);

        return relevanceMatchAddOperation.prepareAndApply();
    }

    @Override
    public Set<RecommendationType> getTypes() {
        return singleton(RecommendationType.switchOnAutotargeting);
    }
}



