package ru.yandex.direct.ytcore.entity.recommendation.service.typesupport;

import java.math.BigDecimal;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.communication.service.CommunicationChannelService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDayBudget;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignOperationService;
import ru.yandex.direct.core.entity.campaign.service.CampaignOptions;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.recommendation.RecommendationType;
import ru.yandex.direct.core.entity.recommendation.model.KpiDailyBudget;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationQueueInfo;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;

/**
 * Обновляет дневной бюджет у кампании
 */
@Component
@ParametersAreNonnullByDefault
public class RecommendationDailyBudgetTypeSupport implements RecommendationTypeSupport {
    private static final Logger logger = LoggerFactory.getLogger(RecommendationDailyBudgetTypeSupport.class);

    private final CampaignRepository campaignRepository;
    private final FeatureService featureService;
    private CommunicationChannelService communicationChannelService;
    private final CampaignOperationService campaignOperationService;

    @Autowired
    public RecommendationDailyBudgetTypeSupport(
            FeatureService featureService,
            CampaignRepository campaignRepository,
            CommunicationChannelService communicationChannelService,
            CampaignOperationService campaignOperationService) {
        this.featureService = featureService;
        this.campaignRepository = campaignRepository;
        this.communicationChannelService = communicationChannelService;
        this.campaignOperationService = campaignOperationService;
    }

    @Override
    public boolean apply(int shard, RecommendationQueueInfo recommendation) {
        Long campaignId = recommendation.getCampaignId();
        //Обновляем дневной бюджет только для кампании с дефолтной стратегией
        Campaign campaign = campaignRepository.getCampaigns(shard, singleton(campaignId)).get(0);
        if (featureService.isEnabledForClientId(
                ClientId.fromLong(campaign.getClientId()), FeatureName.DISABLE_OLD_DAILY_BUDGET_RECOMMENDATION)) {
            return communicationChannelService.apply(recommendation);
        }

        StrategyName strategyName = campaign.getStrategy().getStrategyName();
        KpiDailyBudget data = JsonUtils.fromJson(recommendation.getJsonData(), KpiDailyBudget.class);
        BigDecimal dailyBudget = data.getRecommendedDailyBudget();

        boolean wasUpdated;
        if (strategyName == StrategyName.DEFAULT_) {
            ModelChanges<CampaignWithDayBudget> changes = ModelChanges.build(campaignId,
                    CampaignWithDayBudget.class, CampaignWithDayBudget.DAY_BUDGET, dailyBudget);

            var options = new CampaignOptions();
            UidAndClientId uidAndClientId = UidAndClientId.of(campaign.getUserId(),
                    ClientId.fromLong(campaign.getClientId()));
            Long operatorUid = recommendation.getUid();

            Result<Long> result = campaignOperationService.createRestrictedCampaignUpdateOperation(
                            singletonList(changes), operatorUid, uidAndClientId, options)
                    .apply()
                    .get(0);

            if (result.isSuccessful()) {
                wasUpdated = true;
                logger.info("daily budget was changed to {} for cid {}", dailyBudget, campaignId);
            } else {
                wasUpdated = false;
                logger.info("daily budget was not changed for cid {}", campaignId);
            }
        } else {
            wasUpdated = false;
            logger.info("daily budget can not be changed for cid {} because strategy name is {}", campaignId,
                    strategyName);
        }

        return wasUpdated;
    }

    @Override
    public Set<RecommendationType> getTypes() {
        return singleton(RecommendationType.dailyBudget);
    }
}



