package ru.yandex.direct.ytcore.entity.recommendation.service.typesupport;

import java.math.BigDecimal;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.communication.service.CommunicationChannelService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignOperationService;
import ru.yandex.direct.core.entity.campaign.service.CampaignOptions;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.recommendation.RecommendationType;
import ru.yandex.direct.core.entity.recommendation.model.KpiWeeklyBudget;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationQueueInfo;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.Result;

import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static org.hibernate.validator.internal.util.CollectionHelper.asSet;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.NumberUtils.equalsByCompareTo;

/**
 * Обновляет недельный бюджет, целевые CPA и рентабельность у кампании
 */
@Component
@ParametersAreNonnullByDefault
public class RecommendationWeeklyBudgetTypeSupport implements RecommendationTypeSupport {
    private static final Logger logger = LoggerFactory.getLogger(RecommendationWeeklyBudgetTypeSupport.class);
    public static final String SUCCESS_LOG_TEMPLATE = "%s was changed from %s to %s for cid %s\n";
    public static final String FAILED_TO_APPLY_DATA_WAS_CHANGED_TEMPLATE =
            "failed to apply recommendation for cid %s, strategy data was already changed";

    private final CampaignRepository campaignRepository;
    private final CampaignOperationService campaignOperationService;
    private final FeatureService featureService;
    private final CommunicationChannelService communicationChannelService;

    @Autowired
    public RecommendationWeeklyBudgetTypeSupport(CampaignRepository campaignRepository,
                                                 CampaignOperationService campaignOperationService,
                                                 FeatureService featureService,
                                                 CommunicationChannelService communicationChannelService) {
        this.campaignRepository = campaignRepository;
        this.campaignOperationService = campaignOperationService;
        this.featureService = featureService;
        this.communicationChannelService = communicationChannelService;
    }

    @Override
    public boolean apply(int shard, RecommendationQueueInfo recommendation) {
        Long campaignId = recommendation.getCampaignId();
        Campaign campaign = campaignRepository.getCampaigns(shard, singleton(campaignId)).get(0);
        var type = RecommendationType.fromId(recommendation.getType());

        if (type == RecommendationType.increaseStrategyWeeklyBudget && featureService.isEnabledForClientId(
                ClientId.fromLong(campaign.getClientId()), FeatureName.DISABLE_OLD_WEEKLY_BUDGET_RECOMMENDATION)) {
            return communicationChannelService.apply(recommendation);
        }

        KpiWeeklyBudget data = fromJson(recommendation.getJsonData(), KpiWeeklyBudget.class);
        BigDecimal currentWeeklyBudget = data.getCurrentWeeklyBudget();
        BigDecimal recommendedWeeklyBudget = data.getRecommendedWeeklyBudget();
        BigDecimal currentTargetCPA = data.getCurrentTargetCPA();
        BigDecimal recommendedTargetCPA = data.getRecommendedTargetCPA();
        BigDecimal currentTargetROI = data.getCurrentTargetROI();
        BigDecimal recommendedTargetROI = data.getRecommendedTargetROI();

        DbStrategy strategy = campaign.getStrategy();
        UidAndClientId uidAndClientId = UidAndClientId.of(campaign.getUserId(),
                ClientId.fromLong(campaign.getClientId()));
        Long operatorUid = recommendation.getUid();

        StrategyData strategyData = strategy.getStrategyData();
        StringBuilder succesLog = new StringBuilder();
        StringBuilder failLog = new StringBuilder();

        //Применяем, только если текущие данные в рекомендации совпадают с данными в бд
        if (equalsByCompareTo(currentWeeklyBudget, strategyData.getSum())
                && equalsByCompareTo(currentTargetCPA, strategyData.getAvgCpa())
                && equalsByCompareTo(currentTargetROI, strategyData.getRoiCoef())) {

            if (!equalsByCompareTo(recommendedWeeklyBudget, strategyData.getSum())) {
                addLogs(succesLog, failLog, "weekly budget", campaignId, strategyData.getSum(),
                        recommendedWeeklyBudget);
                strategyData.setSum(recommendedWeeklyBudget);
            }
            if (!equalsByCompareTo(recommendedTargetCPA, strategyData.getAvgCpa())) {
                addLogs(succesLog, failLog, "target CPA", campaignId, strategyData.getAvgCpa(), recommendedTargetCPA);
                strategyData.setAvgCpa(recommendedTargetCPA);
            }
            if (!equalsByCompareTo(recommendedTargetROI, strategyData.getRoiCoef())) {
                addLogs(succesLog, failLog, "target ROI", campaignId, strategyData.getRoiCoef(), recommendedTargetROI);
                strategyData.setRoiCoef(recommendedTargetROI);
            }
        } else {
            logger.info(String.format(FAILED_TO_APPLY_DATA_WAS_CHANGED_TEMPLATE, campaignId));
            return false;
        }

        ModelChanges<CampaignWithCustomStrategy> changes = ModelChanges.build(campaignId,
                CampaignWithCustomStrategy.class, CampaignWithCustomStrategy.STRATEGY, strategy);

        var options = new CampaignOptions();
        Result<Long> result = campaignOperationService.createRestrictedCampaignUpdateOperation(
                singletonList(changes), operatorUid, uidAndClientId, options)
                .apply()
                .get(0);

        if (result.isSuccessful()) {
            logger.info(succesLog.toString());
        } else {
            logger.error("Unexpected error on applying recommendation: {}",
                result.getValidationResult().flattenErrors().toString());
        }

        logger.info(result.isSuccessful() && result.getValidationResult().hasAnyErrors()
                ? failLog.toString() : succesLog.toString());
        return result.isSuccessful();
    }

    private void addLogs(StringBuilder succesLog, StringBuilder failLog, String fieldName, Long campaignId,
                         BigDecimal oldValue, BigDecimal newValue) {
        succesLog.append(String.format(SUCCESS_LOG_TEMPLATE, fieldName, oldValue, newValue, campaignId));
    }

    @Override
    public Set<RecommendationType> getTypes() {
        return asSet(RecommendationType.weeklyBudget, RecommendationType.increaseStrategyWeeklyBudget,
                RecommendationType.increaseStrategyTargetCPA, RecommendationType.decreaseStrategyTargetROI);
    }
}
