package ru.yandex.direct.ytcore.entity.statistics.repository;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import org.jooq.Field;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.grid.schema.yt.tables.DirectconversioncostBs;
import ru.yandex.direct.utils.DateTimeUtils;
import ru.yandex.direct.ytcomponents.service.ConversionsStatsDynContextProvider;
import ru.yandex.direct.ytcomponents.statistics.model.ConversionsStatisticsResponse;
import ru.yandex.direct.ytcomponents.statistics.model.DateRange;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedValue;

import static org.jooq.impl.DSL.field;
import static ru.yandex.direct.grid.schema.yt.tables.DirectconversioncostBs.DIRECTCONVERSIONCOST_BS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;
import static ru.yandex.direct.ytwrapper.YtTableUtils.findColumnOrThrow;
import static ru.yandex.direct.ytwrapper.YtUtils.effectiveCampaignIdField;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL.toEpochSecondsAtStartOfDate;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtMappingUtils.fromMicros;

/**
 * Репозиторий для работы со статистикой по конверсиям
 */
@Repository
@ParametersAreNonnullByDefault
public class ConversionStatisticsRepository {
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");

    private static final DirectconversioncostBs STAT = DIRECTCONVERSIONCOST_BS.as("S");

    private static final Field<Long> EFFECTIVE_CAMPAIGN_ID = field("effectiveCampaignId", Long.class);
    private static final Field<Long> UPDATE_TIME = aliased(STAT.UPDATE_TIME);
    private static final Field<Long> GOAL_ID = aliased(STAT.GOAL_ID);
    private static final Field<Long> ATTRIBUTION_TYPE = aliased(STAT.ATTRIBUTION_TYPE);
    private static final Field<Long> COST_TAX_FREE = aliased(STAT.COST_TAX_FREE);
    private static final Field<Long> GOALS_NUM = aliased(STAT.GOALS_NUM);

    private final ConversionsStatsDynContextProvider contextProvider;

    @Autowired
    public ConversionStatisticsRepository(ConversionsStatsDynContextProvider contextProvider) {
        this.contextProvider = contextProvider;
    }

    public List<ConversionsStatisticsResponse> getConversionsStatistics(Collection<Long> campaignIds,
                                                                        Map<Long, Long> masterIdBySubId,
                                                                        DateRange dateRange) {
        var raw = contextProvider.getContext().executeSelect(
                YtDSL.ytContext()
                        .select(effectiveCampaignIdField(STAT.EXPORT_ID, masterIdBySubId).as(EFFECTIVE_CAMPAIGN_ID),
                                UPDATE_TIME, GOAL_ID, ATTRIBUTION_TYPE, COST_TAX_FREE, GOALS_NUM)
                        .from(STAT)
                        .where(STAT.EXPORT_ID.in(Sets.union(Set.copyOf(campaignIds), masterIdBySubId.keySet())))
                        .and(STAT.UPDATE_TIME.between(
                                toEpochSecondsAtStartOfDate(dateRange.getFromInclusive()),
                                toEpochSecondsAtStartOfDate(dateRange.getToInclusive())))
        );

        var schema = raw.getSchema();
        int campaignIdColIdx = findColumnOrThrow(schema, EFFECTIVE_CAMPAIGN_ID);
        int updateTimeColIdx = findColumnOrThrow(schema, UPDATE_TIME);
        int goalIdColIdx = findColumnOrThrow(schema, GOAL_ID);
        int attrTypeColIdx = findColumnOrThrow(schema, ATTRIBUTION_TYPE);
        int costTaxFreeColIdx = findColumnOrThrow(schema, COST_TAX_FREE);
        int goalsNumColIdx = findColumnOrThrow(schema, GOALS_NUM);

        Function<UnversionedRow, ConversionsStatisticsResponse> responseConverter = row -> {
            List<UnversionedValue> values = row.getValues();
            return new ConversionsStatisticsResponse()
                    .withCampaignId(values.get(campaignIdColIdx).longValue())
                    .withUpdateTime(formatDate(values.get(updateTimeColIdx).longValue()))
                    .withGoalId(values.get(goalIdColIdx).longValue())
                    .withAttributionType(values.get(attrTypeColIdx).longValue())
                    .withCostTaxFree(parseCost(values.get(costTaxFreeColIdx).longValue()))
                    .withGoalsNum(values.get(goalsNumColIdx).longValue());
        };

        return mapList(raw.getRows(), responseConverter);
    }

    private static String formatDate(Long seconds) {
        var date = LocalDate.ofInstant(Instant.ofEpochSecond(seconds), DateTimeUtils.MSK);
        return date.format(DATE_FORMATTER);
    }

    private static BigDecimal parseCost(long rawCost) {
        var converted = fromMicros(rawCost);
        return converted.compareTo(BigDecimal.ZERO) < 0 ? BigDecimal.ZERO.setScale(2, RoundingMode.DOWN) : converted;
    }
}
