package ru.yandex.direct.ytcore.entity.statistics.repository;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.Row3;
import org.jooq.Row4;
import org.jooq.types.ULong;

import ru.yandex.direct.grid.schema.yt.tables.Directphrasestatv2Bs;
import ru.yandex.direct.ytcomponents.service.StatsDynContextProvider;
import ru.yandex.direct.ytcomponents.statistics.model.DateRange;
import ru.yandex.direct.ytcomponents.statistics.model.PhraseStatisticsRequest;
import ru.yandex.direct.ytcomponents.statistics.model.PhraseStatisticsResponse;
import ru.yandex.direct.ytcomponents.statistics.model.RetargetingStatisticsRequest;
import ru.yandex.direct.ytcomponents.statistics.model.ShowConditionStatisticsRequest;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;
import ru.yandex.yt.ytclient.wire.UnversionedValue;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.jooq.impl.DSL.not;
import static org.jooq.impl.DSL.row;
import static org.jooq.impl.DSL.sum;
import static org.jooq.impl.DSL.val;
import static org.jooq.types.Unsigned.ulong;
import static ru.yandex.direct.grid.schema.yt.Tables.DIRECTPHRASESTATV2_BS;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;
import static ru.yandex.direct.ytwrapper.YtTableUtils.findColumnOrThrow;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL.isNull;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL.toEpochSecondsAtStartOfDate;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL.toULongLiteral;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL.ytFalse;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL.ytIf;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL.ytTrue;

/**
 * Репозиторий, умеющий доставать обобщённую статитику по показам из YT.
 */
public class RecentStatisticsRepository {

    private static final Directphrasestatv2Bs STAT = DIRECTPHRASESTATV2_BS.as("S");

    // ID'шки из Директа
    private static final Field<Long> CAMPAIGN_ID = aliased(STAT.EXPORT_ID);
    private static final Field<Long> AD_GROUP_ID = aliased(STAT.GROUP_EXPORT_ID);
    private static final Field<Long> BANNER_ID = aliased(STAT.DIRECT_BANNER_ID);
    private static final Field<ULong> PHRASE_ID = aliased(STAT.PHRASE_EXPORT_ID);
    private static final Field<ULong> BS_PHRASE_ID = aliased(STAT.PHRASE_ID);
    private static final Field<ULong> GOAL_CONTEXT_ID = aliased(STAT.GOAL_CONTEXT_ID);
    private static final Field<Long> IS_MOBILE = aliased(STAT.IS_MOBILE);

    /*
     * Правила учёта статистики показов/кликов:
     * - для мобильных баннеров берётся только мобильная,
     * - для РМП суммарная,RecentStatisticsRepositoryTest.java
     * - для остальных — только десктопная
     *
     * YaSearch{Shows,Clicks} содержат данные о показах/кликах только на поиске Яндекс.
     * Прочие поисковые площадки не учитываются
     */
    private static final Field<BigDecimal> YA_SEARCH_SHOWS =
            sum(STAT.YA_SEARCH_SHOWS)
                    .as("YaSearchShows");

    private static final Field<BigDecimal> YA_SEARCH_CLICKS =
            sum(STAT.YA_SEARCH_CLICKS)
                    .as("YaSearchClicks");

    private static final Field<BigDecimal> YA_SEARCH_ESHOWS =
            sum(ytIf(isNull(STAT.YA_SEARCH_ESHOWS), val(BigDecimal.ZERO), STAT.YA_SEARCH_ESHOWS))
                    .as("YaSearchEshows");

    private static final Field<BigDecimal> SEARCH_SHOWS =
            sum(ytIf(STAT.IS_FLAT.eq(ytFalse()), STAT.SHOWS, val(0L)))
                    .as("SearchShows");

    private static final Field<BigDecimal> SEARCH_CLICKS =
            sum(ytIf(STAT.IS_FLAT.eq(ytFalse()), STAT.CLICKS, val(0L)))
                    .as("SearchClicks");

    private static final Field<BigDecimal> SEARCH_ESHOWS =
            sum(ytIf(STAT.IS_FLAT.eq(ytFalse()).and(not(isNull(STAT.ESHOWS))), STAT.ESHOWS, val(BigDecimal.ZERO)))
                    .as("SearchEshows");

    private static final Field<BigDecimal> NETWORK_SHOWS =
            sum(ytIf(STAT.IS_FLAT.eq(ytTrue()), STAT.SHOWS, val(0L)))
                    .as("NetworkShows");

    private static final Field<BigDecimal> NETWORK_CLICKS =
            sum(ytIf(STAT.IS_FLAT.eq(ytTrue()), STAT.CLICKS, val(0L)))
                    .as("NetworkClicks");

    private static final Field<BigDecimal> NETWORK_ESHOWS =
            sum(ytIf(STAT.IS_FLAT.eq(ytTrue()).and(not(isNull(STAT.ESHOWS))), STAT.ESHOWS, val(BigDecimal.ZERO)))
                    .as("NetworkEshows");

    private final StatsDynContextProvider dynContextProvider;

    public RecentStatisticsRepository(StatsDynContextProvider dynContextProvider) {
        this.dynContextProvider = dynContextProvider;
    }

    /**
     * Для указанных условий ретаргетинга возвращает статистику за выбранный период.
     */
    public List<PhraseStatisticsResponse> getRetargetingStatistics(Collection<RetargetingStatisticsRequest> requests,
                                                                   DateRange dateRange) {
        Collection<Row3<Long, Long, ULong>> idFilter = requests.stream()
                .map(r -> row(r.getCampaignId(), r.getAdGroupId(), ULong.valueOf(r.getRetargetingConditionId())))
                .collect(toList());
        Condition idFilterCondition = row(CAMPAIGN_ID, AD_GROUP_ID, GOAL_CONTEXT_ID).in(idFilter);
        return getStatistics(idFilterCondition, dateRange);
    }

    /**
     * Для указанных условий показа возвращает статистику за выбранный период.
     */
    public List<PhraseStatisticsResponse> getShowConditionStatistics(
            Collection<ShowConditionStatisticsRequest> requests,
            DateRange dateRange) {
        Collection<Row3<Long, Long, ULong>> idFilter = requests.stream()
                .map(r -> row(r.getCampaignId(), r.getAdGroupId(), ULong.valueOf(r.getShowConditionId())))
                .collect(toList());
        Condition idFilterCondition = row(CAMPAIGN_ID, AD_GROUP_ID, PHRASE_ID).in(idFilter);
        return getStatistics(idFilterCondition, dateRange);
    }

    /**
     * Для указанных фраз возвращает статистику за выбранный период.
     */
    public List<PhraseStatisticsResponse> getPhraseStatistics(Collection<PhraseStatisticsRequest> requests,
                                                              DateRange dateRange) {
        List<PhraseStatisticsResponse> result = new ArrayList<>(requests.size());

        // Создаём разные Condition'ы для фильтрации по разным полям индекса
        // Для одного _типа_ запросов будет один Condition
        Collection<Condition> idFilterConditions = createIdFilterConditions(requests);
        // Для разных условий на индексные колонки -- разные запросы в YT
        for (Condition idFilterCondition : idFilterConditions) {
            result.addAll(getStatistics(idFilterCondition, dateRange));
        }

        return result;
    }

    /**
     * Получает из YT статистику за указанный интервал времени для условий показа,
     * попадающих под критерий {@code idFilterCondition}
     */
    private List<PhraseStatisticsResponse> getStatistics(Condition idFilterCondition,
                                                         DateRange dateRange) {
        // Описание запроса в тикете https://st.yandex-team.ru/DIRECT-74239#1516199576000
        UnversionedRowset rowset = dynContextProvider.getContext().executeSelect(
                YtDSL.ytContext()
                        .select(CAMPAIGN_ID, AD_GROUP_ID, BANNER_ID, PHRASE_ID, BS_PHRASE_ID, GOAL_CONTEXT_ID,
                                IS_MOBILE,
                                YA_SEARCH_SHOWS, YA_SEARCH_CLICKS, YA_SEARCH_ESHOWS,
                                SEARCH_SHOWS, SEARCH_CLICKS, SEARCH_ESHOWS,
                                NETWORK_SHOWS, NETWORK_CLICKS, NETWORK_ESHOWS)
                        .from(STAT)
                        .where(idFilterCondition)
                        .and(STAT.UPDATE_TIME.between(
                                toEpochSecondsAtStartOfDate(dateRange.getFromInclusive()),
                                toEpochSecondsAtStartOfDate(dateRange.getToInclusive())))
                        // todo maxlog: подумать про то, чтобы убрать этот WA (https://st.yandex-team.ru/BSDEV-68944#1523031902000)
                        .and(not(isNull(STAT.SHOWS)))
                        .groupBy(CAMPAIGN_ID, AD_GROUP_ID, BANNER_ID, PHRASE_ID, BS_PHRASE_ID, GOAL_CONTEXT_ID,
                                IS_MOBILE)
        );

        TableSchema schema = rowset.getSchema();
        int campaignIdColIdx = findColumnOrThrow(schema, CAMPAIGN_ID);
        int adGroupIdColIdx = findColumnOrThrow(schema, AD_GROUP_ID);
        int bannerIdColIdx = findColumnOrThrow(schema, BANNER_ID);
        int phraseIdColIdx = findColumnOrThrow(schema, PHRASE_ID);
        int bsPhraseIdColIdx = findColumnOrThrow(schema, BS_PHRASE_ID);
        int goalContextIdColIdx = findColumnOrThrow(schema, GOAL_CONTEXT_ID);
        int isMobileColIdx = findColumnOrThrow(schema, IS_MOBILE);
        int yaSearchShowsColIdx = findColumnOrThrow(schema, YA_SEARCH_SHOWS);
        int yaSearchClicksColIdx = findColumnOrThrow(schema, YA_SEARCH_CLICKS);
        int yaSearchEshowsColIdx = findColumnOrThrow(schema, YA_SEARCH_ESHOWS);
        int searchShowsColIdx = findColumnOrThrow(schema, SEARCH_SHOWS);
        int searchClicksColIdx = findColumnOrThrow(schema, SEARCH_CLICKS);
        int searchEshowsColIdx = findColumnOrThrow(schema, SEARCH_ESHOWS);
        int networkShowsColIdx = findColumnOrThrow(schema, NETWORK_SHOWS);
        int networkClicksColIdx = findColumnOrThrow(schema, NETWORK_CLICKS);
        int networkEshowsColIdx = findColumnOrThrow(schema, NETWORK_ESHOWS);

        Function<UnversionedRow, PhraseStatisticsResponse> convertRow =
                row -> {
                    List<UnversionedValue> values = row.getValues();
                    return new PhraseStatisticsResponse()
                            .withCampaignId(values.get(campaignIdColIdx).longValue())
                            .withAdGroupId(values.get(adGroupIdColIdx).longValue())
                            .withBannerId(values.get(bannerIdColIdx).longValue())
                            .withPhraseId(values.get(phraseIdColIdx).longValue())
                            .withBsPhraseId(ulong(values.get(bsPhraseIdColIdx).longValue()))
                            .withGoalContextId(ulong(values.get(goalContextIdColIdx).longValue()))
                            .withMobile(values.get(isMobileColIdx).booleanValue())
                            .withSearchShows(values.get(searchShowsColIdx).longValue())
                            .withSearchClicks(values.get(searchClicksColIdx).longValue())
                            .withSearchEshows(values.get(searchEshowsColIdx).doubleValue())
                            .withNetworkShows(values.get(networkShowsColIdx).longValue())
                            .withNetworkClicks(values.get(networkClicksColIdx).longValue())
                            .withNetworkEshows(values.get(networkEshowsColIdx).doubleValue())
                            .withYaSearchShows(values.get(yaSearchShowsColIdx).longValue())
                            .withYaSearchClicks(values.get(yaSearchClicksColIdx).longValue())
                            .withYaSearchEshows(values.get(yaSearchEshowsColIdx).doubleValue());
                };

        return StreamEx.of(rowset.getRows())
                .map(convertRow)
                .toList();
    }

    /**
     * В запросе может присутствовать bannerId, а может и отсутствовать. Генерируем разные условия для этих случаев.
     * Также возможны варианты запроса: с {@code bsPhraseId} или с {@code phraseId}.
     * <p>
     * Если все запросы из {@code requests} одного типа (например, без {@code bannerId} и с {@code phraseId}),
     * вернётся коллекция из одного элемента.
     */
    private Collection<Condition> createIdFilterConditions(Collection<PhraseStatisticsRequest> requests) {
        if (requests.isEmpty()) {
            return emptyList();
        }
        Collection<Row4<Long, Long, Long, ULong>> withBsPhraseIdAndBannerId = new ArrayList<>();
        Collection<Row4<Long, Long, Long, ULong>> withPhraseIdAndBannerId = new ArrayList<>();
        Collection<Row3<Long, Long, ULong>> withBsPhraseIdAndWithoutBannerId = new ArrayList<>();
        Collection<Row3<Long, Long, ULong>> withPhraseIdAndWithoutBannerId = new ArrayList<>();

        for (PhraseStatisticsRequest request : requests) {
            Long campaignId = request.getCampaignId();
            Long adGroupId = request.getAdGroupId();
            Long bannerId = request.getBannerId();
            ULong bsPhraseId = request.getBsPhraseId();
            ULong phraseId = request.getPhraseId() != null ? ULong.valueOf(request.getPhraseId()) : null;
            if (bannerId != null) {
                if (phraseId != null) {
                    withPhraseIdAndBannerId.add(row(campaignId, adGroupId, bannerId, phraseId));
                } else if (bsPhraseId != null) {
                    withBsPhraseIdAndBannerId
                            .add(row(val(campaignId), val(adGroupId), val(bannerId), toULongLiteral(bsPhraseId)));
                } else {
                    throw new IllegalArgumentException("Both bsPhraseId and phraseId are null");
                }
            } else {
                if (phraseId != null) {
                    withPhraseIdAndWithoutBannerId.add(row(campaignId, adGroupId, phraseId));
                } else if (bsPhraseId != null) {
                    withBsPhraseIdAndWithoutBannerId
                            .add(row(val(campaignId), val(adGroupId), toULongLiteral(bsPhraseId)));
                } else {
                    throw new IllegalArgumentException("Both bsPhraseId and phraseId are null");
                }
            }
        }

        Collection<Condition> result = new ArrayList<>();
        if (!withBsPhraseIdAndBannerId.isEmpty()) {
            result.add(row(CAMPAIGN_ID, AD_GROUP_ID, BANNER_ID, BS_PHRASE_ID).in(withBsPhraseIdAndBannerId));
        }

        if (!withPhraseIdAndBannerId.isEmpty()) {
            result.add(row(CAMPAIGN_ID, AD_GROUP_ID, BANNER_ID, PHRASE_ID).in(withPhraseIdAndBannerId));
        }

        if (!withBsPhraseIdAndWithoutBannerId.isEmpty()) {
            result.add(row(CAMPAIGN_ID, AD_GROUP_ID, BS_PHRASE_ID).in(withBsPhraseIdAndWithoutBannerId));
        }

        if (!withPhraseIdAndWithoutBannerId.isEmpty()) {
            result.add(row(CAMPAIGN_ID, AD_GROUP_ID, PHRASE_ID).in(withPhraseIdAndWithoutBannerId));
        }

        return result;
    }

}
