package ru.yandex.direct.ytcore.entity.statistics.service;

import java.time.LocalDate;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.ytcomponents.statistics.model.ConversionsStatisticsResponse;
import ru.yandex.direct.ytcomponents.statistics.model.DateRange;
import ru.yandex.direct.ytcore.entity.statistics.repository.ConversionStatisticsRepository;

import static java.util.function.Function.identity;

@Service
@ParametersAreNonnullByDefault
public class ConversionStatisticsService {
    private final ConversionStatisticsRepository repository;
    private final CampaignService campaignService;

    @Autowired
    public ConversionStatisticsService(ConversionStatisticsRepository repository,
                                       CampaignService campaignService) {
        this.repository = repository;
        this.campaignService = campaignService;
    }

    /**
     * Возвращает статистику по конверсиям для запрошенных кампаний.
     * Результат — {@link Map}, в котором ключи представлены cid'ами кампаний,
     * а значения — списки {@link ConversionsStatisticsResponse} с данными по конверсиям.
     * <p>
     * Этот метод возвращает статистику за последний год.
     */
    public Map<Long, List<ConversionsStatisticsResponse>> getConversionStatistics(Collection<Long> cids) {
        LocalDate now = LocalDate.now();
        return getConversionStatistics(cids, now.minusYears(1), now);
    }

    private Map<Long, List<ConversionsStatisticsResponse>> getConversionStatistics(
            Collection<Long> cids, LocalDate fromInclusive, LocalDate toInclusive) {
        DateRange dateRange = new DateRange()
                .withFromInclusive(fromInclusive)
                .withToInclusive(toInclusive);
        var masterIdBySubId = campaignService.getSubCampaignIdsWithMasterIds(cids);
        var stats = repository.getConversionsStatistics(cids, masterIdBySubId, dateRange);

        return StreamEx.of(stats)
                .mapToEntry(ConversionsStatisticsResponse::getCampaignId, identity())
                .grouping();
    }
}
