package ru.yandex.direct.ytcore.entity.statistics.service;

import java.time.LocalDate;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithIsMobile;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.model.KeywordRecentStatistics;
import ru.yandex.direct.core.entity.keyword.service.KeywordRecentStatisticsProvider;
import ru.yandex.direct.ytcomponents.statistics.model.DateRange;
import ru.yandex.direct.ytcomponents.statistics.model.PhraseStatisticsRequest;
import ru.yandex.direct.ytcomponents.statistics.model.PhraseStatisticsResponse;
import ru.yandex.direct.ytcomponents.statistics.model.RetargetingStatisticsRequest;
import ru.yandex.direct.ytcomponents.statistics.model.ShowConditionStatisticsRequest;
import ru.yandex.direct.ytcomponents.statistics.model.StatValueAggregator;
import ru.yandex.direct.ytcore.entity.statistics.repository.RecentStatisticsRepository;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.getValueIfAssignable;
import static ru.yandex.direct.core.entity.banner.service.validation.type.BannerTypeValidationPredicates.isMobileAppBanner;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис, умеющий доставать обобщённую статитику по показам из YT.
 */
@Service
public class RecentStatisticsService implements KeywordRecentStatisticsProvider {
    public static final int DEFAULT_STATISTIC_PERIOD = 28;

    private static final Logger logger = LoggerFactory.getLogger(RecentStatisticsService.class);

    private final RecentStatisticsRepository statisticsRepository;
    private final BannerService bannerService;

    @Autowired
    public RecentStatisticsService(RecentStatisticsRepository statisticsRepository,
                                   BannerService bannerService) {
        this.statisticsRepository = statisticsRepository;
        this.bannerService = bannerService;
    }

    /**
     * Для указанных фраз возвращает статистику за выбранный период.
     * <p>
     * Особенности (подробнее в тикете <a href="https://st.yandex-team.ru/DIRECT-74239">DIRECT-74239</a>):
     * <ul>
     * <li>Из поисковой учитывается только статистика с поиска Яндекс (без других поисковых площадок).</li>
     * <li>Из поисковой статистики для мобильных баннеров учитывается только мобильная статистика.</li>
     * <li>Из поисковой статистики для РМП баннеров учитывается как мобильная, так и десктопная статистика.</li>
     * <li>Из поисковой статистики для не мобильных и не-РМП баннеров учитывается только десктопная статистика.</li>
     * </ul>
     *
     * @see RecentStatisticsRepository#getPhraseStatistics(Collection, DateRange)
     */
    public Map<PhraseStatisticsResponse.PhraseStatIndex, StatValueAggregator> getPhraseStatistics(
            Collection<PhraseStatisticsRequest> requests,
            DateRange dateRange) {
        return getPhraseStatistics(requests, dateRange, PhraseStatisticsResponse::phraseStatIndex);
    }

    public Map<Long, KeywordRecentStatistics> getKeywordRecentStatistics(
            Collection<Keyword> keywordRequests) {
        DateRange dateRange = new DateRange()
                .withFromInclusive(LocalDate.now().minusDays(DEFAULT_STATISTIC_PERIOD))
                .withToExclusive(LocalDate.now());
        Collection<PhraseStatisticsRequest> phraseRequests = mapList(
                keywordRequests, kr -> new PhraseStatisticsRequest.Builder()
                        .withCampaignId(kr.getCampaignId())
                        .withAdGroupId(kr.getAdGroupId())
                        .withPhraseId(kr.getId())
                        .build()
        );

        Map<PhraseStatisticsResponse.PhraseStatIndex, StatValueAggregator> response =
                getPhraseStatistics(phraseRequests, dateRange);

        Map<Long, Optional<StatValueAggregator>> responseGroupedByPhrase = EntryStream.of(response)
                .mapKeys(PhraseStatisticsResponse.PhraseStatIndex::getPhraseId)
                .grouping(Collectors.reducing(StatValueAggregator::add));

        return EntryStream.of(responseGroupedByPhrase)
                .mapValues(Optional::get)
                .mapValues(statistics -> ((KeywordRecentStatistics) statistics))
                .toMap();
    }

    /**
     * Для указанных фраз возвращает статистиску за период с учётом задаваемого ключа.
     */
    public <T> Map<T, StatValueAggregator> getPhraseStatistics(
            Collection<PhraseStatisticsRequest> requests,
            DateRange dateRange,
            Function<PhraseStatisticsResponse, T> extractStatIndex) {
        List<PhraseStatisticsResponse> phraseStatistics = statisticsRepository.getPhraseStatistics(requests, dateRange);

        Set<Long> bannerIds = phraseStatistics.stream()
                .map(PhraseStatisticsResponse::getBannerId)
                .filter(Objects::nonNull)
                .collect(toSet());
        Map<Long, Banner> bannersByIds = listToMap(bannerService.getBannersByIds(bannerIds), Banner::getId);

        Map<T, Optional<StatValueAggregator>> grouping = StreamEx.of(phraseStatistics)
                .mapToEntry(extractStatIndex, identity())
                .mapValues(statCalculationByMobileForPhrase(bannersByIds))
                .grouping(Collectors.reducing(StatValueAggregator::add));

        return EntryStream.of(grouping)
                .filterValues(Optional::isPresent)
                .mapValues(Optional::get)
                .toMap();
    }

    /**
     * Возвращет функцию, которая реализует логику правильного подсчёта статистики для фраз
     * в зависимости от мобильности баннера
     *
     * @see #statCalculationByMobile(Map)
     */
    private Function<PhraseStatisticsResponse, StatValueAggregator> statCalculationByMobileForPhrase(
            Map<Long, Banner> bannersByIds) {
        return resp -> {
            Long bannerId = resp.getBannerId();
            Banner banner = bannersByIds.get(bannerId);

            boolean useNetwork;
            boolean useMobile;
            boolean useNonMobile;
            if (banner != null) {
                useNetwork = true;
                Boolean isMobile = getValueIfAssignable(banner, BannerWithIsMobile.IS_MOBILE);
                if (isMobile != null && isMobile) {
                    // для мобильных баннеров учитываем только показы на мобильных
                    useMobile = true;
                    useNonMobile = false;
                } else if (isMobileAppBanner(banner)) {
                    // для РМП учитываем все показы
                    useMobile = true;
                    useNonMobile = true;
                } else {
                    // для остальных -- только на desktop
                    useMobile = false;
                    useNonMobile = true;
                }
            } else if (bannerId == 0L) {
                // Не учитываем статистику для bannerId == 0
                useNetwork = false;
                useMobile = false;
                useNonMobile = false;
            } else {
                logger.warn("Statistic for absent banner (id={}) found", bannerId);
                // Нет баннера -- ситуация возможная только в тестовом окружении.
                // Считаем немобильную статистику
                useNetwork = true;
                useMobile = false;
                useNonMobile = true;
            }

            StatValueAggregator agg = new StatValueAggregator();
            if (useNetwork) {
                agg.addNetworkClicks(resp.getNetworkClicks());
                agg.addNetworkShows(resp.getNetworkShows());
                agg.addNetworkEshows(resp.getNetworkEshows());
            }
            if ((useMobile && resp.isMobile())
                    || useNonMobile && !resp.isMobile()) {
                agg.addSearchClicks(resp.getYaSearchClicks());
                agg.addSearchShows(resp.getYaSearchShows());
                agg.addSearchEshows(resp.getYaSearchEshows());
            }
            return agg;
        };
    }

    /**
     * Для указанных условий показа возвращает статистику за выбранный период.
     */
    public Map<PhraseStatisticsResponse.ShowConditionStatIndex, StatValueAggregator> getShowConditionStatistics(
            Collection<ShowConditionStatisticsRequest> showConditionStaticticsRequests,
            DateRange dateRange) {
        List<PhraseStatisticsResponse> phraseStatistics =
                statisticsRepository.getShowConditionStatistics(showConditionStaticticsRequests, dateRange);

        Set<Long> bannerIds = phraseStatistics.stream()
                .map(PhraseStatisticsResponse::getBannerId)
                .filter(Objects::nonNull)
                .collect(toSet());
        Map<Long, Banner> bannersByIds = listToMap(bannerService.getBannersByIds(bannerIds), Banner::getId);

        Map<PhraseStatisticsResponse.ShowConditionStatIndex, Optional<StatValueAggregator>> grouping =
                StreamEx.of(phraseStatistics)
                        .mapToEntry(PhraseStatisticsResponse::showConditionStatIndex, identity())
                        .mapValues(statCalculationByMobile(bannersByIds))
                        .grouping(Collectors.reducing(StatValueAggregator::add));

        return EntryStream.of(grouping)
                .filterValues(Optional::isPresent)
                .mapValues(Optional::get)
                .toMap();
    }

    /**
     * Для указанных условий ретаргетинга возвращает статистику за выбранный период.
     */
    public Map<PhraseStatisticsResponse.GoalContextStatIndex, StatValueAggregator> getRetargetingStatistics(
            Collection<RetargetingStatisticsRequest> retargetingStatisticRequests, DateRange dateRange) {
        List<PhraseStatisticsResponse> phraseStatistics =
                statisticsRepository.getRetargetingStatistics(retargetingStatisticRequests, dateRange);

        Set<Long> bannerIds = phraseStatistics.stream()
                .map(PhraseStatisticsResponse::getBannerId)
                .filter(Objects::nonNull)
                .collect(toSet());
        Map<Long, Banner> bannersByIds = listToMap(bannerService.getBannersByIds(bannerIds), Banner::getId);

        Map<PhraseStatisticsResponse.GoalContextStatIndex, Optional<StatValueAggregator>> grouping =
                StreamEx.of(phraseStatistics)
                        .mapToEntry(PhraseStatisticsResponse::goalContextStatIndex, identity())
                        .mapValues(statCalculationByMobile(bannersByIds))
                        .grouping(Collectors.reducing(StatValueAggregator::add));

        return EntryStream.of(grouping)
                .filterValues(Optional::isPresent)
                .mapValues(Optional::get)
                .toMap();
    }

    /**
     * Возвращет функцию, которая реализует логику правильного подсчёта статистики для условий показа
     * в зависимости от мобильности баннера
     *
     * @see #statCalculationByMobileForPhrase(Map)
     */
    private Function<PhraseStatisticsResponse, StatValueAggregator> statCalculationByMobile(
            Map<Long, Banner> bannersByIds) {
        return resp -> {
            Long bannerId = resp.getBannerId();
            Banner banner = bannersByIds.get(bannerId);

            boolean useNetwork;
            boolean useMobile;
            boolean useNonMobile;

            // Если баннер известен, и он мобильный или РМП, то учитываем только мобильную статистику
            if (banner != null) {
                useNetwork = true;
                Boolean isMobile = getValueIfAssignable(banner, BannerWithIsMobile.IS_MOBILE);
                if (isMobile != null && isMobile) {
                    // для мобильных баннеров учитываем только показы на мобильных
                    useMobile = true;
                    useNonMobile = false;
                } else if (isMobileAppBanner(banner)) {
                    // для РМП учитываем тоже только мобильную статистику (отличие от фраз)
                    useMobile = true;
                    useNonMobile = false;
                } else {
                    // для остальных -- только на desktop
                    useMobile = false;
                    useNonMobile = true;
                }
            } else if (bannerId == 0L) {
                // todo maxlog: временный WA: не находим баннер -- используем немобильную статистику (https://st
                //  .yandex-team.ru/BSDEV-68944#1523014219000)
                useNetwork = true;
                useMobile = false;
                useNonMobile = true;
            } else {
                logger.warn("Statistic for absent banner (id={}) found", bannerId);
                // Нет баннера -- ситуация возможная только в тестовом окружении.
                // Считаем немобильную статистику
                useNetwork = true;
                useMobile = false;
                useNonMobile = true;
            }

            StatValueAggregator agg = new StatValueAggregator();
            if (useNetwork) {
                agg.addNetworkClicks(resp.getNetworkClicks());
                agg.addNetworkShows(resp.getNetworkShows());
                agg.addNetworkEshows(resp.getNetworkEshows());
            }
            if ((useMobile && resp.isMobile())
                    || useNonMobile && !resp.isMobile()) {
                agg.addSearchClicks(resp.getSearchClicks());
                agg.addSearchShows(resp.getSearchShows());
                agg.addSearchEshows(resp.getSearchEshows());
            }
            return agg;
        };
    }
}
