package ru.yandex.direct.advq;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import one.util.streamex.StreamEx;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.advq.checkminhits.CheckMinHitsItem;
import ru.yandex.direct.advq.checkminhits.CheckMinHitsResponse;
import ru.yandex.direct.advq.exception.AdvqClientException;
import ru.yandex.direct.advq.search.AdvqRequestKeyword;
import ru.yandex.direct.advq.search.AdvqSearchRequest;
import ru.yandex.direct.advq.search.SearchChunkedRequests;
import ru.yandex.direct.advq.search.SearchItem;
import ru.yandex.direct.advq.search.SearchParsableRequest;
import ru.yandex.direct.advq.search.SearchResponse;
import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableRequest;
import ru.yandex.direct.asynchttp.ParsableStringRequest;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceChild;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.AutoCloseableList;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.Transient;

import static com.google.common.collect.Lists.partition;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.tracing.util.TraceUtil.X_YANDEX_TRACE;
import static ru.yandex.direct.tracing.util.TraceUtil.traceToHeader;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

/**
 * Клиент для сервиса ADVQ - количество запросов на поиске по ключевым фразам
 */
@ParametersAreNonnullByDefault
public class AdvqClient {
    public static final String DEVICES_ALL = "all";
    public static final String DEVICES_PHONE = "phone";
    public static final String DEVICES_TABLET = "tablet";
    public static final String DEVICES_DESKTOP = "desktop";

    public static final String X_ADVQ_CUSTOMER_HEADER = "X-Advq-Customer";

    /**
     * Когда отправляем в ADVQ/search запрос с параметром timeout, на своей стороне
     * ждём ответа большее в {@code SEARCH_QUERY_OUTER_TIMEOUT_MULTIPLIER} раз
     */
    private static final long SEARCH_QUERY_OUTER_TIMEOUT_MULTIPLIER = 2L;

    private static final Logger logger = LoggerFactory.getLogger(AdvqClient.class);

    private final AdvqClientSettings settings;
    private String searchUrl;
    private String videoSearchUrl;
    private final ParallelFetcherFactory parallelFetcherFactory;

    /**
     * Создает экземпляр клиента
     *
     * @param settings Установки клиента
     */
    public AdvqClient(AdvqClientSettings settings, AsyncHttpClient asyncHttpClient) {
        this.settings = settings;
        this.searchUrl = settings.getSearchUrl();
        this.videoSearchUrl = settings.getVideoSearchUrl();
        this.parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient, settings.getFetcherSettings());
    }

    /**
     * Возвращает, были ли показы на поиске по ключевым фразам в определенных регионах
     *
     * @param keywords  набор ключевых фраз
     * @param regionIds набор регионов, поддерживает минус-регионы для исключения
     * @return отображение исходной ключевой фразы на результат
     */
    public CheckMinHitsResult checkMinHits(List<String> keywords, List<Long> regionIds, List<String> devices) {
        logger.trace("Starting processing request of keywords {}, regionIds {}, devices {}",
                keywords, regionIds, devices);
        try (TraceProfile profile = Trace.current().profile("checkMinHits");
             AutoCloseableList<TraceChild> traces = new AutoCloseableList<>();
             ParallelFetcher<String> parallelFetcher = parallelFetcherFactory.getParallelFetcherWithMetricRegistry(
                     SolomonUtils.getParallelFetcherMetricRegistry(profile.getFunc()))) {
            List<ParsableRequest<String>> requests = new ArrayList<>();

            long id = 0L;
            Multimap<String, Long> deviceMap = HashMultimap.create();
            for (String device : devices) {
                for (RequestBuilder requestBuilder : prepareRequests(keywords, regionIds, device)) {
                    TraceChild traceChild = Trace.current().child("advq", "checkMinHits");
                    requestBuilder.setHeader(X_YANDEX_TRACE, traceToHeader(traceChild));
                    requestBuilder.setHeader(X_ADVQ_CUSTOMER_HEADER,
                            nvl(settings.getCustomerName(), AdvqClientSettings.DEFAULT_CUSTOMER_NAME));
                    traces.add(new Transient<>(traceChild));

                    requests.add(new ParsableStringRequest(id, requestBuilder.build()));
                    deviceMap.put(device, id);
                    id++;
                }
            }

            Map<Long, Result<String>> results = parallelFetcher.execute(requests);
            List<Result<String>> failed = filterList(results.values(), v -> v.getSuccess() == null);
            if (failed.size() > 0) {
                logger.info("Advq response with errors: {}", results);
                return CheckMinHitsResult.failure(flatMap(failed, Result::getErrors));
            }

            logger.debug("Advq response: {}", results);
            return CheckMinHitsResult.success(formResults(results, deviceMap));
        } catch (InterruptedException ex) {
            logger.error("Interrupted while processing request with keywords {}, regionIds {}, devices {}",
                    keywords, regionIds, devices);
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        } finally {
            logger.trace("Finished processing request of keywords {}, regionIds {}, devices {}",
                    keywords, regionIds, devices);
        }
    }

    public IdentityHashMap<SearchRequest, Map<AdvqRequestKeyword, SearchKeywordResult>> search(Collection<SearchRequest> requests)
            throws AdvqClientException {
        return search(requests, new AdvqSearchOptions());
    }

    /**
     * Вызов {@link #searchBase(Collection, AdvqSearchOptions, Duration, Integer, String)}
     * для получения статистики по ключевым фразам без задания таймаута.
     *
     * @see #searchBase(Collection, AdvqSearchOptions, Duration, Integer, String)
     */
    public IdentityHashMap<SearchRequest, Map<AdvqRequestKeyword, SearchKeywordResult>> search(
            Collection<SearchRequest> requests,
            AdvqSearchOptions options) throws AdvqClientException {
        return search(requests, options, null);
    }

    /**
     * Вызов {@link #searchBase(Collection, AdvqSearchOptions, Duration, Integer, String)}
     * для получения статистики по ключевым фразам.
     *
     * @see #searchBase(Collection, AdvqSearchOptions, Duration, Integer, String)
     */
    public IdentityHashMap<SearchRequest, Map<AdvqRequestKeyword, SearchKeywordResult>> search(
            Collection<SearchRequest> requests,
            AdvqSearchOptions options,
            @Nullable Duration timeout) throws AdvqClientException {
        return searchBase(requests, options, timeout, settings.getSearchChunkSize(), searchUrl);
    }

    /**
     * Вызов {@link #searchBase(Collection, AdvqSearchOptions, Duration, Integer, String)}
     * для получения статистики видео по ключевым фразам без задания таймаута.
     *
     * @see #searchBase(Collection, AdvqSearchOptions, Duration, Integer, String)
     */
    public IdentityHashMap<SearchRequest, Map<AdvqRequestKeyword, SearchKeywordResult>> searchVideo(
            Collection<SearchRequest> requests,
            AdvqSearchOptions options) throws AdvqClientException {
        return searchVideo(requests, options, null);
    }

    /**
     * Вызов {@link #searchBase(Collection, AdvqSearchOptions, Duration, Integer, String)}
     * для получения статистики видео по ключевым фразам.
     *
     * @see #searchBase(Collection, AdvqSearchOptions, Duration, Integer, String)
     */
    public IdentityHashMap<SearchRequest, Map<AdvqRequestKeyword, SearchKeywordResult>> searchVideo(
            Collection<SearchRequest> requests,
            AdvqSearchOptions options,
            @Nullable Duration timeout) throws AdvqClientException {
        return searchBase(requests, options, timeout, settings.getVideoSearchChunkSize(), videoSearchUrl);
    }

    /**
     * Вызывает метод search в ADVQ, возвращающий для ключевой фразы и набора регионов статистику
     * по этой фразе в этих регионах.
     * <p>
     * Для каждого исходного запроса всегда возвращает ответ, содержащий все ключевые фразы, для каждой фразы
     * возвращается один из трех вариантов - статистика, список ошибок, или "пустой" ответ, если ADVQ не вернуло
     * ничего.
     *
     * @param requests  список запросов для отправки в ADVQ. Запрос представляет собой объединение набора ключевых
     *                  фраз и набора регионов
     * @param options   дополнительные параметры для запроса в ADVQ
     * @param timeout   таймаут на весь вызов. {@code null} означает отсутствие ограничения
     * @param chunkSize размер чанка для запроса в ADVQ
     * @param searchUrl URL, по которому делается запрос в ADVQ
     * @return соответствие исходный запрос -> мэппинг (ключевая фраза из запроса) -> результат advq.
     */
    private IdentityHashMap<SearchRequest, Map<AdvqRequestKeyword, SearchKeywordResult>> searchBase(
            Collection<SearchRequest> requests,
            AdvqSearchOptions options,
            @Nullable Duration timeout,
            Integer chunkSize,
            String searchUrl) throws AdvqClientException {
        logger.trace("Starting processing requests of {}", requests);
        FetcherSettings fetcherSettings = getSearchFetcherSettings(timeout);

        int phrasesCount = requests.stream().map(SearchRequest::getKeywords).mapToInt(List::size).sum();
        try (TraceProfile profile = Trace.current().profile("advq:advq_request", "search", phrasesCount);
             AutoCloseableList<TraceChild> traces = new AutoCloseableList<>();
             ParallelFetcher<SearchResponse> parallelFetcher =
                     parallelFetcherFactory.getParallelFetcher(
                             fetcherSettings.withMetricRegistry(SolomonUtils.getParallelFetcherMetricRegistry(profile.getFunc())))) {
            SearchChunkedRequests searchChunkedRequests = getSearchChunkedRequests(requests, options, chunkSize);

            Map<Long, Result<SearchResponse>> advqResults = executeSearchRequests(traces, parallelFetcher,
                    searchChunkedRequests, searchUrl);

            return getSearchResults(requests, searchChunkedRequests, advqResults);
        } catch (InterruptedException ex) {
            if (requests.size() > 10) {
                logger.error("Interrupted while processing {} requests, the first 10 requests {}", requests.size(),
                        requests.stream().limit(10).toArray());
            } else {
                logger.error("Interrupted while processing {} requests {}", requests.size(), requests);
            }
            Thread.currentThread().interrupt();
            throw new AdvqClientException("Interrupted while processing ADVQ/search request", ex);
        } finally {
            logger.trace("Finished processing requests {}", requests);
        }
    }

    private FetcherSettings getSearchFetcherSettings(@Nullable Duration timeout) {
        // В метод /search передаём timeout, который использует ADVQ. Реальное время может получиться
        // чуть больше (сеть + ожидание в очереди), поэтому с нашей стороны устанавливаем таймаут на запрос вдвое
        // большим
        Duration outerRequestTimeout = settings.getFetcherSettings().getRequestTimeout()
                .multipliedBy(SEARCH_QUERY_OUTER_TIMEOUT_MULTIPLIER);
        return new FetcherSettings(settings.getFetcherSettings())
                .withGlobalTimeout(timeout)
                .withRequestTimeout(outerRequestTimeout);
    }

    /**
     * Преобразование списка запросов в ADVQ в запросы, разбитые по чанкам.
     *
     * @param requests  Исходный список запросов.
     * @param options   Дополнительные параметры для запроса в ADVQ.
     * @param chunkSize Размер чанка для запроса в ADVQ.
     * @return Запросы, разбитые по чанкам, и мапы id запроса к исходному запросу и к чанку.
     */
    private SearchChunkedRequests getSearchChunkedRequests(
            Collection<SearchRequest> requests,
            AdvqSearchOptions options,
            Integer chunkSize) {
        Map<Long, SearchRequest> idToRequest = new HashMap<>();
        Map<Long, List<AdvqRequestKeyword>> idToChunk = new HashMap<>();
        List<AdvqSearchRequest> chunkedRequests = new ArrayList<>();
        long id = 0;
        for (SearchRequest request : requests) {
            // Уникализируем ключевые фразы внутри запроса
            List<AdvqRequestKeyword> uniqueKeywords = StreamEx.of(request.getKeywords()).distinct().toList();
            List<List<AdvqRequestKeyword>> chunks = partition(uniqueKeywords, chunkSize);
            for (List<AdvqRequestKeyword> chunk : chunks) {
                chunkedRequests.add(formAdvqSearchRequest(id, chunk, request, options));
                idToRequest.put(id, request);
                idToChunk.put(id, chunk);

                id++;
            }
        }
        return new SearchChunkedRequests(idToRequest, idToChunk, chunkedRequests);
    }

    private Map<Long, Result<SearchResponse>> executeSearchRequests(AutoCloseableList<TraceChild> traces,
                                                                    ParallelFetcher<SearchResponse> parallelFetcher,
                                                                    SearchChunkedRequests searchChunkedRequests,
                                                                    String videoSearchUrl) throws InterruptedException {
        List<SearchParsableRequest> parsableRequests = mapList(searchChunkedRequests.getChunkedRequests(), request -> {
            TraceChild child = Trace.current().child("advq", "search");
            traces.add(new Transient<>(child));
            return new SearchParsableRequest(request.getId(), request.formRequest(videoSearchUrl,
                    traceToHeader(child), settings.getCustomerName()));
        });

        Map<Long, Result<SearchResponse>> advqResults = parallelFetcher.execute(parsableRequests);
        logger.debug("advq response: {}", advqResults);
        return advqResults;
    }

    private IdentityHashMap<SearchRequest, Map<AdvqRequestKeyword, SearchKeywordResult>> getSearchResults(
            Collection<SearchRequest> requests,
            SearchChunkedRequests searchChunkedRequests,
            Map<Long, Result<SearchResponse>> advqResults) {
        IdentityHashMap<SearchRequest, Map<AdvqRequestKeyword, SearchKeywordResult>> results = new IdentityHashMap<>();
        for (SearchRequest request : requests) {
            results.put(request, new HashMap<>());
        }

        for (Long resultId : advqResults.keySet()) {
            SearchRequest request = searchChunkedRequests.getIdToRequest().get(resultId);
            List<AdvqRequestKeyword> chunk = searchChunkedRequests.getIdToChunk().get(resultId);
            Result<SearchResponse> advqResult = advqResults.get(resultId);
            Map<AdvqRequestKeyword, SearchKeywordResult> requestResult = results.get(request);

            if (advqResult.getSuccess() != null) {
                Map<String, SearchItem> searchItemByKeyword =
                        listToMap(advqResult.getSuccess().getSearchItems(), SearchItem::getReq);
                for (AdvqRequestKeyword keyword : chunk) {
                    if (searchItemByKeyword.containsKey(keyword.getPhrase())) { // нашли, вносим ответ
                        requestResult.put(keyword,
                                SearchKeywordResult.success(searchItemByKeyword.get(keyword.getPhrase())));
                    } else { // не нашли, вносим пустой результат, странно, логаем
                        logger.warn("Keyword '{}' is not found in advq response {}.", keyword, advqResults);
                        requestResult.put(keyword, SearchKeywordResult.empty());
                    }
                }
            } else { // ошибка, вносим результаты-ошибки
                for (AdvqRequestKeyword keyword : chunk) {
                    requestResult.put(keyword, SearchKeywordResult.failure(advqResult.getErrors()));
                }
            }
        }

        return results;
    }

    private AdvqSearchRequest formAdvqSearchRequest(long id,
                                                    List<AdvqRequestKeyword> keywords,
                                                    SearchRequest searchRequest,
                                                    AdvqSearchOptions options) {
        Duration requestTimeout = settings.getFetcherSettings().getRequestTimeout();
        return new AdvqSearchRequest()
                .withParserType(options.getParserType())
                .withId(id)
                .withTimeout(requestTimeout)
                .withWords(mapList(keywords, AdvqRequestKeyword::getPhrase))
                .withRegions(searchRequest.getRegionIds())
                .withDevices(searchRequest.getDeviceTypes())
                .withAssocs(options.getAssocs())
                .withCalcTotalHits(options.getCalcTotalHits())
                .withFastMode(options.getFastMode())
                .withPhPage(options.getPhPage())
                .withPhPageSize(options.getPhPageSize());

    }

    private List<RequestBuilder> prepareRequests(List<String> keywords,
                                                 List<Long> regionIds,
                                                 String device) {
        return mapList(partition(keywords, settings.getCheckMinHitsChunkSize()), chunkKeywords -> {
            RequestBuilder builder = new RequestBuilder("POST")
                    .setUrl(settings.getCheckMinHitsUrl())
                    .addFormParam("format", "json")
                    .addFormParam("regions", String.join(",", mapList(regionIds, Object::toString)))
                    .addFormParam("devices", device)
                    .setCharset(UTF_8);
            for (String keyword : chunkKeywords) {
                builder.addFormParam("words", keyword);
            }
            return builder;
        });
    }

    private Map<String, Map<String, CheckMinHitsItem>> formResults(
            Map<Long, Result<String>> results,
            Multimap<String, Long> deviceMap) {
        return deviceMap.keySet().stream().collect(toMap(
                key -> key,
                key -> deviceMap.get(key).stream()
                        .map(results::get)
                        .map(result -> fromJson(result.getSuccess(), CheckMinHitsResponse.class))
                        .flatMap(r -> r.getRequests().stream())
                        .collect(toMap(CheckMinHitsItem::getReq, Function.identity()))));
    }
}
