package ru.yandex.direct.advq;

import org.asynchttpclient.uri.Uri;

import ru.yandex.direct.asynchttp.FetcherSettings;

/**
 * Основные настройки ADVQ-клиента
 */
public class AdvqClientSettings {
    public static final String DEFAULT_CUSTOMER_NAME = "advq-java-client";

    private String host;
    private int port = 80;
    private String customerName = null;

    private String checkMinHitsPath;
    private Integer checkMinHitsChunkSize;
    private String searchPath;
    private Integer searchChunkSize;

    private String videoHost;
    private int videoPort;
    private String videoSearchPath;
    private Integer videoSearchChunkSize;

    private FetcherSettings fetcherSettings;

    /**
     * @param host хост ADVQ
     */
    public AdvqClientSettings withHost(String host) {
        this.host = host;
        return this;
    }


    public AdvqClientSettings withPort(int port) {
        this.port = port;
        return this;
    }

    public AdvqClientSettings withCustomerName(String customerName) {
        this.customerName = customerName;
        return this;
    }

    /**
     * @param checkMinHitsPath путь до сервиса check_min_hits на сервере ADVQ
     */
    public AdvqClientSettings withCheckMinHitsPath(String checkMinHitsPath) {
        this.checkMinHitsPath = checkMinHitsPath;
        return this;
    }

    /**
     * @param checkMinHitsChunkSize сколько ключевых слов мы отправляем в ADVQ одним запросом в метод check_min_hits
     *                              (влияет на скорость ответа ADVQ)
     */
    public AdvqClientSettings withCheckMinHitsChunkSize(Integer checkMinHitsChunkSize) {
        this.checkMinHitsChunkSize = checkMinHitsChunkSize;
        return this;
    }

    /**
     * @param searchPath путь до сервиса search на сервере ADVQ
     */
    public AdvqClientSettings withSearchPath(String searchPath) {
        this.searchPath = searchPath;
        return this;
    }

    /**
     * @param searchChunkSize сколько ключевых слов мы отправляем в ADVQ одним запросом в метод search
     *                        (влияет на скорость ответа ADVQ)
     */
    public AdvqClientSettings withSearchChunkSize(Integer searchChunkSize) {
        this.searchChunkSize = searchChunkSize;
        return this;
    }

    /**
     * @param videoHost хост сервера video ADVQ
     */
    public AdvqClientSettings withVideoHost(String videoHost) {
        this.videoHost = videoHost;
        return this;
    }

    /**
     * @param videoPort порт сервера video ADVQ
     */
    public AdvqClientSettings withVideoPort(int videoPort) {
        this.videoPort = videoPort;
        return this;
    }

    /**
     * @param videoSearchPath путь до сервиса video на сервере video ADVQ
     */
    public AdvqClientSettings withVideoSearchPath(String videoSearchPath) {
        this.videoSearchPath = videoSearchPath;
        return this;
    }

    /**
     * @param videoSearchChunkSize сколько ключевых слов мы отправляем в video ADVQ одним запросом в метод search
     *                             (влияет на скорость ответа ADVQ)
     */
    public AdvqClientSettings withVideoSearchChunkSize(Integer videoSearchChunkSize) {
        this.videoSearchChunkSize = videoSearchChunkSize;
        return this;
    }

    /**
     * @param fetcherSettings Настройки async http-клиента, который будет использоваться клиентом ADVQ
     */
    public AdvqClientSettings withFetcherSettings(FetcherSettings fetcherSettings) {
        this.fetcherSettings = fetcherSettings;
        return this;
    }

    /**
     * @return URL сервиса check_min_hits
     */
    String getCheckMinHitsUrl() {
        return new Uri(Uri.HTTP, null, host, port, checkMinHitsPath, null, null).toUrl();
    }

    /**
     * @return размер чанка для отправки в ADVQ
     */
    Integer getCheckMinHitsChunkSize() {
        return checkMinHitsChunkSize;
    }

    /**
     * @return URL сервиса search
     */
    String getSearchUrl() {
        return new Uri(Uri.HTTP, null, host, port, searchPath, null, null).toUrl();
    }

    String getCustomerName() {
        return customerName;
    }

    Integer getSearchChunkSize() {
        return searchChunkSize;
    }

    /**
     * @return URL сервиса search в video ADVQ
     */
    public String getVideoSearchUrl() {
        return new Uri(Uri.HTTP, null, videoHost, videoPort, videoSearchPath, null, null).toUrl();
    }

    /**
     * @return размер чанка для отправки в video ADVQ
     */
    public Integer getVideoSearchChunkSize() {
        return videoSearchChunkSize;
    }

    /**
     * @return настройки async http-клиента, который будет использоваться клиентом ADVQ
     */
    FetcherSettings getFetcherSettings() {
        return fetcherSettings;
    }
}
