package ru.yandex.direct.advq.search;

import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Collection;
import java.util.Objects;

import one.util.streamex.StreamEx;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;

import ru.yandex.direct.advq.Device;

import static ru.yandex.direct.advq.AdvqClient.X_ADVQ_CUSTOMER_HEADER;
import static ru.yandex.direct.tracing.util.TraceUtil.X_YANDEX_TRACE;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Посмотреть справку по параметрам ручек ADVQ можно на странице http://back-normal.advq.yandex.net/advq/help.
 * Ссылка работает, если зайти с ppcdev
 */
public class AdvqSearchRequest {

    private static final double MILLIS_PER_SECOND = 1_000.;

    private ParserType parserType;

    private long id;
    private Duration timeout;

    private Collection<String> words;
    private Collection<Long> regions;
    // todo maxlog (DIRECT-73232): этот параметр принимает только значения, упомянутые в /advq/info (rus,rus-robots,tur,tur-robots)
    private String dbName;
    private Collection<Device> devices;
    // todo maxlog (DIRECT-73232): этот параметр assocs не известен в ADVQ (/advq/help)
    private Boolean calcTotalHits;
    // todo maxlog (DIRECT-73232): этот параметр assocs не известен в ADVQ (/advq/help)
    private Boolean assocs;
    private Boolean fastMode;
    private Integer phPage;
    private Integer phPageSize;

    public Long getId() {
        return id;
    }

    public AdvqSearchRequest withTimeout(Duration timeout) {
        this.timeout = timeout;
        return this;
    }

    public AdvqSearchRequest withParserType(ParserType parserType) {
        this.parserType = parserType;
        return this;
    }

    public AdvqSearchRequest withId(long id) {
        this.id = id;
        return this;
    }

    public AdvqSearchRequest withWords(Collection<String> words) {
        this.words = words;
        return this;
    }

    public AdvqSearchRequest withRegions(Collection<Long> regions) {
        this.regions = regions;
        return this;
    }

    public AdvqSearchRequest withDbName(String dbName) {
        this.dbName = dbName;
        return this;
    }

    public AdvqSearchRequest withDevices(Collection<Device> devices) {
        this.devices = devices;
        return this;
    }

    public AdvqSearchRequest withCalcTotalHits(Boolean calcTotalHits) {
        this.calcTotalHits = calcTotalHits;
        return this;
    }

    public AdvqSearchRequest withAssocs(Boolean assocs) {
        this.assocs = assocs;
        return this;
    }

    public AdvqSearchRequest withFastMode(Boolean fastMode) {
        this.fastMode = fastMode;
        return this;
    }

    public AdvqSearchRequest withPhPage(Integer phPage) {
        this.phPage = phPage;
        return this;
    }

    public AdvqSearchRequest withPhPageSize(Integer phPageSize) {
        this.phPageSize = phPageSize;
        return this;
    }

    public Request formRequest(String url, String yandexTrace, String advqCustomerName) {
        RequestBuilder builder = new RequestBuilder("POST")
                .setUrl(url)
                .addFormParam("format", "json")
                .addFormParam("parser", nvl(parserType, ParserType.ADVQ).getValue())
                .setCharset(StandardCharsets.UTF_8);

        if (timeout != null) {
            // Advq принимает в параметр timeout числа с плавающей запятой
            long millis = timeout.toMillis();
            double timeoutSeconds = millis / MILLIS_PER_SECOND;
            builder.addFormParam("timeout", Double.toString(timeoutSeconds));
        }

        if (regions != null && !regions.isEmpty()) {
            builder.addFormParam("regions", StreamEx.of(regions).joining(","));
        }

        if (devices != null && !devices.isEmpty()) {
            builder.addFormParam("devices", StreamEx.of(devices).map(Device::getValue).joining(","));
        }

        if (dbName != null && !dbName.isEmpty()) {
            builder.addFormParam("dbname", dbName);
        }

        if (calcTotalHits != null) {
            builder.addFormParam("calc_total_hits", calcTotalHits ? "1" : "0");
        }

        if (assocs != null) {
            builder.addFormParam("assocs", assocs ? "1" : "0");
        }

        if (fastMode != null) {
            builder.addFormParam("fast_mode", fastMode ? "1" : "0");
        }

        if (phPage != null) {
            builder.addFormParam("ph_page", phPage.toString());
        }

        if (phPageSize != null) {
            builder.addFormParam("ph_page_size", phPageSize.toString());
        }

        for (String word : Objects.requireNonNull(words, "words must be filled")) {
            builder.addFormParam("words", word);
        }

        builder.setHeader(X_YANDEX_TRACE, yandexTrace);
        builder.setHeader(X_ADVQ_CUSTOMER_HEADER, advqCustomerName);
        return builder.build();
    }
}
