package ru.yandex.direct.ansiblejuggler;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ansiblejuggler.model.PlayRecap;
import ru.yandex.direct.process.ProcessCommunicator;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.ansiblejuggler.AnsibleWrapperConfiguration.DEFAULT_ANSIBLE_EXECUTION_TIMEOUT;
import static ru.yandex.direct.ansiblejuggler.AnsibleWrapperConfiguration.LINUX_ANSIBLE_PLAYBOOK_PATH;
import static ru.yandex.direct.ansiblejuggler.AnsibleWrapperConfiguration.NULL_PATH;

/**
 * Обертка над ansible-playbook для проверки и применения плейбуков.
 * <pre>{@code
 *  File playFile = playbook.saveToTempFile();
 *  AnsibleWrapper wrapper = new AnsibleWrapper(playFile);
 *  PlayRecap checkRecap = wrapper.processPlaybook(true);
 *  if (checkRecap != null && checkRecap.getFailed() == 0 && checkRecap.getChanged() != 0) {
 *      wrapper.processPlaybook(false);
 *  }
 * }
 * </pre>
 */
@ParametersAreNonnullByDefault
public class AnsibleWrapper {
    /**
     * Параметры запуска ansible-playbook по умолчанию:
     * <ul>
     * <li>путь до ansible-playbook - {@value AnsibleWrapperConfiguration#LINUX_ANSIBLE_PLAYBOOK_PATH}</li>
     * <li>инвентори файл ansible - {@value AnsibleWrapperConfiguration#NULL_PATH}</li>
     * <li>лог-файл ansible - {@value AnsibleWrapperConfiguration#NULL_PATH}</li>
     * <li>таймаут на исполнение ansible-playbook -
     * {@link AnsibleWrapperConfiguration#DEFAULT_ANSIBLE_EXECUTION_TIMEOUT}</li>
     * </ul>
     */
    public static final AnsibleWrapperConfiguration DEFAULT_CONFIGURATION = new AnsibleWrapperConfiguration.Builder()
            .withAnsiblePlaybookCmd(LINUX_ANSIBLE_PLAYBOOK_PATH)
            .withExecutionTimeout(DEFAULT_ANSIBLE_EXECUTION_TIMEOUT)
            .withAnsibleLogPath(NULL_PATH)
            .withInventoryFile(NULL_PATH)
            .build();

    /**
     * Исключение, которое будет выброшено при ошибке синхронизации плейбука.
     * Возможные причины:
     * <ul>
     * <li>отсутствие ansible</li>
     * <li>отсутствие ansible-juggler</li>
     * <li>ошибка запуска ansible-playbook (например при неправильном пути до исполняемого файла или плейбука)</li>
     * <li>ошибки межпроцессного взаимодействия, ввода-вывода</li>
     * <li>невозмонжость распарсить вывод ansible-playbook (бывает если плейбук синтаксически некорректен)</li>
     * </ul>
     */
    public static class PlaybookProcessingException extends Exception {
        PlaybookProcessingException(Throwable cause) {
            super(cause);
        }

        PlaybookProcessingException(String message) {
            super(message);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(AnsibleWrapper.class);
    private static final String RECAP_LINE_MARKER = "PLAY RECAP ***";

    private final AnsibleWrapperConfiguration configuration;
    private final File playbook;

    /**
     * Создает обертку для синхронизации плейбука c настройками по умолчанию {@link #DEFAULT_CONFIGURATION}
     *
     * @param playbook файл синхронизируемого плейбука
     * @throws NullPointerException  если {@code playbook} равен {@code null}
     * @throws IllegalStateException если не существуют {@code configuration.getAnsiblePlaybookCmd()}
     *                               или {@code playbook}
     */
    public AnsibleWrapper(File playbook) {
        this(DEFAULT_CONFIGURATION, playbook);
    }

    /**
     * Создает обертку для синхронизации плейбука.
     *
     * @param configuration конфигурация для ansible-playbook
     * @param playbook      файл синхронизируемого плейбука
     * @throws NullPointerException     если равны {@code null} любой из переданных аргументов,
     *                                  {@code configuration.getAnsiblePlaybookCmd()},
     *                                  {@code configuration.getExecutionTimeout()}
     * @throws IllegalStateException    если не существуют {@code configuration.getAnsiblePlaybookCmd()}
     *                                  {@code playbook}
     * @throws IllegalArgumentException при нулевых или отрицательных значениях
     *                                  {@code configuration.getExecutionTimeout()}
     */
    public AnsibleWrapper(AnsibleWrapperConfiguration configuration, File playbook) {
        checkNotNull(configuration);
        checkNotNull(playbook);
        checkNotNull(configuration.getAnsiblePlaybookCmd(), "ansiblePlaybookCommand is not specified in configuration");
        checkNotNull(configuration.getExecutionTimeout(), "executionTimeout is not specified in configuration");

        checkArgument(
                !(configuration.getExecutionTimeout().isNegative() || configuration.getExecutionTimeout().isZero()),
                "executionTimeout must be positive");
        checkState(new File(configuration.getAnsiblePlaybookCmd()).exists(), "ansiblePlaybookCommand not found");
        checkState(playbook.exists(), "playbook file must exists");

        this.configuration = configuration;
        this.playbook = playbook;
    }

    /**
     * Проверить плейбук.
     * Не вносит реальных изменений в конфигурацию juggler-сервера.
     *
     * @return новый объект {@link PlayRecap} со статистикой проверки плейбука
     * @throws PlaybookProcessingException при ошибках применения
     * @see #processPlaybook(boolean)
     */
    public PlayRecap checkPlaybook() throws PlaybookProcessingException {
        return processPlaybook(true);
    }

    /**
     * Синхронизировать плейбук.
     *
     * @return новый объект {@link PlayRecap} со статистикой применения плейбука
     * @throws PlaybookProcessingException при ошибках применения
     * @see #processPlaybook(boolean)
     */
    public PlayRecap syncPlaybook() throws PlaybookProcessingException {
        return processPlaybook(false);
    }

    /**
     * Выставляет переменные окружения, запускает дочерним процессом {@code ansible-playbook} и обрабатывает его вывод.
     * Время жизни дочернего процесса ограничено {@code configuration.getExecutionTimeout()},
     * после этого процесс будет убит.
     * Весь STDOUT дочернего процесса логируется с уровнем TRACE, STDERR - с уровнем ERROR.
     *
     * @param checkOnly выполнить только проверку {@code true} или синхронизацию {@code false} плейбука
     * @return новый объект {@link PlayRecap} со статистикой обработки плейбука
     * @throws PlaybookProcessingException при ошибках применения
     * @see PlaybookProcessingException
     */
    @SuppressWarnings("WeakerAccess")
    protected PlayRecap processPlaybook(boolean checkOnly) throws PlaybookProcessingException {
        List<String> cmd = new ArrayList<>(3);
        cmd.add(configuration.getAnsiblePlaybookCmd());
        if (checkOnly) {
            cmd.add("--check");
        }
        cmd.add(playbook.getAbsolutePath());

        ProcessBuilder builder = new ProcessBuilder(cmd);
        if (configuration.getAnsibleLogPath() != null) {
            builder.environment().put("ANSIBLE_LOG_PATH", configuration.getAnsibleLogPath());
        }
        if (configuration.getAnsibleLibrary() != null) {
            builder.environment().put("ANSIBLE_LIBRARY", configuration.getAnsibleLibrary());
        }
        builder.environment().put("ANSIBLE_RETRY_FILES_ENABLED", "False");

        File temporaryInventory = null;
        switch (configuration.getInventoryFileType()) {
            case TEMPORARY:
                try {
                    logger.debug("Creating temporary file for inventory");
                    temporaryInventory = File.createTempFile("ansible-inventory-", "tmp");
                    temporaryInventory.deleteOnExit();
                    FileWriter writer = new FileWriter(temporaryInventory);
                    logger.trace("Write inventory content to {}", temporaryInventory.getAbsoluteFile());
                    writer.write(configuration.getAnsibleInventoryContent());
                    writer.close();

                    builder.environment().put("ANSIBLE_INVENTORY", temporaryInventory.getAbsolutePath());
                } catch (IOException e) {
                    logger.error("Failed to create inventory temporary file", e);
                    throw new PlaybookProcessingException(e);
                }
                break;
            case REGULAR:
                builder.environment().put("ANSIBLE_INVENTORY", configuration.getAnsibleInventoryPath());
                break;
            default:
                // никаких дополнительных настроек не делаем
                break;
        }

        OutConsumer outConsumer = new OutConsumer();

        logger.info("Execute: {}", builder.command());
        try (ProcessCommunicator communicator = new ProcessCommunicator.Builder(builder)
                .withThreadNamePrefix("ansible")
                .withStdoutReader(new ProcessCommunicator.LineReader(StandardCharsets.UTF_8, outConsumer))
                .withStderrReader(new ProcessCommunicator.LineReader(StandardCharsets.UTF_8, new ErrConsumer()))
                .withRaiseExceptionOnNonZeroExitCode(false)
                .build()
        ) {
            if (!communicator.waitFor(configuration.getExecutionTimeout())) {
                throw new PlaybookProcessingException("Timed out waiting for ansible output");
            }
        } catch (InterruptedException e) {
            logger.error("Interrupted during wait sub-process exit", e);
            Thread.currentThread().interrupt();
            throw new PlaybookProcessingException(e);
        } catch (RuntimeException e) {
            logger.error("Unknown error", e);
            throw new PlaybookProcessingException(e);
        } finally {
            if (temporaryInventory != null) {
                logger.debug("Delete temporary inventory file");
                if (!temporaryInventory.delete()) {
                    logger.warn("Failed to inventory temporary file. Will be deleted at JVM exit");
                }
            }
        }

        PlayRecap recap = outConsumer.getRecap();
        if (recap == null) {
            logger.error("Recap absent");
            throw new PlaybookProcessingException("Recap absent. Check log for details");
        } else {
            logger.info("Playbook stat: {}", recap);
            return recap;
        }
    }

    /**
     * Класс для обработки STDOUT от ansible-playbook.
     * <p>
     * В получаемых строках ожидает {@value AnsibleWrapper#RECAP_LINE_MARKER},
     * для создания объекта {@code PlayRecap} из первой встреченной строки соответствующего вида
     */
    static class OutConsumer implements Consumer<String> {
        private static final Logger logger = LoggerFactory.getLogger(OutConsumer.class);
        private static final String ERROR_LINE_MARKER = "ERROR:";
        private static final String TASK_LINE_MARKER = "TASK:";
        private static final String FAILED_TASK_LINE_MARKER = "failed:";

        private boolean seenErrorMarker;
        private boolean seenRecapMarker;
        private String previousLine;
        private PlayRecap recap;

        OutConsumer() {
            seenErrorMarker = false;
            seenRecapMarker = false;
            previousLine = null;
            recap = null;
        }

        /**
         * @param line строка вывода ansible-playbook
         */
        @Override
        public void accept(String line) {
            if (line.isEmpty()) {
                seenErrorMarker = false;
                return;
            } else if (line.startsWith(ERROR_LINE_MARKER) || seenErrorMarker) {
                seenErrorMarker = true;
                logger.error(line);
            } else if (previousLine != null) {
                // workaround для логирования зафейленных тасков с уровнем "ошибка"
                if (line.startsWith(FAILED_TASK_LINE_MARKER)) {
                    logger.error(previousLine);
                    logger.error(line);
                    seenErrorMarker = true;
                } else {
                    logger.trace(previousLine);
                    logger.trace(line);
                }
                previousLine = null;
            } else if (line.startsWith(TASK_LINE_MARKER)) {
                previousLine = line;
                // залогируем строку вместе со следующей
            } else {
                logger.trace(line);
            }
            if (!seenRecapMarker && line.startsWith(RECAP_LINE_MARKER)) {
                seenRecapMarker = true;
            }
            if (seenRecapMarker && recap == null && PlayRecap.getStatMatcher(line).find()) {
                recap = new PlayRecap(line);
            }
        }

        /**
         * Получить статистику обработки плейбка на основе обработанного вывода.
         *
         * @return новый объект {@link PlayRecap} или {@code null}, если статистика не была найдена
         */
        @Nullable
        PlayRecap getRecap() {
            return recap;
        }
    }

    /**
     * Класс для обработки STDERR от ansible-playbook
     */
    static class ErrConsumer implements Consumer<String> {
        private static final Logger logger = LoggerFactory.getLogger(ErrConsumer.class);

        @Override
        public void accept(String line) {
            if (!line.isEmpty()) {
                logger.error(line);
            }
        }
    }
}
