package ru.yandex.direct.ansiblejuggler;

import java.time.Duration;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Класс описывающий для {@link AnsibleWrapper} параметры запуска ansible-playbook
 */
@SuppressWarnings("WeakerAccess")
public class AnsibleWrapperConfiguration {
    /**
     * Путь по умолчанию к ansible-playbook на linux-системах (при установке из deb-пактов)
     */
    public static final String LINUX_ANSIBLE_PLAYBOOK_PATH = "/bin/ansible-juggler";

    /**
     * Путь {@literal /dev/null}, для использования в качестве значений параметров,
     * например logPath или inventoryFile
     */
    public static final String NULL_PATH = "/dev/null";

    /**
     * Умолчание на максимально-допустимое время выполнения ansible-playbook
     */
    public static final Duration DEFAULT_ANSIBLE_EXECUTION_TIMEOUT = Duration.ofMinutes(10);

    private final InventoryFileType inventoryType;
    private final String ansiblePlaybookCmd;
    private final String ansibleLogPath;
    private final String ansibleInventoryPath;
    private final String ansibleInventoryContent;
    private final String ansibleLibrary;
    private final Duration executionTimeout;

    private AnsibleWrapperConfiguration(Builder builder) {
        ansiblePlaybookCmd = builder.ansiblePlaybookCmd;
        ansibleLogPath = builder.ansibleLogPath;
        ansibleLibrary = builder.ansibleLibrary;

        if (builder.ansibleInventoryContent != null) {
            inventoryType = InventoryFileType.TEMPORARY;
            ansibleInventoryPath = null;
            ansibleInventoryContent = builder.ansibleInventoryContent;
        } else if (builder.ansibleInventoryPath != null) {
            inventoryType = InventoryFileType.REGULAR;
            ansibleInventoryPath = builder.ansibleInventoryPath;
            ansibleInventoryContent = null;
        } else {
            inventoryType = InventoryFileType.NONE;
            ansibleInventoryPath = null;
            ansibleInventoryContent = null;
        }
        executionTimeout = builder.executionTimeout;
    }

    /**
     * Получить тип inventory-файла
     */
    @Nonnull
    public InventoryFileType getInventoryFileType() {
        return inventoryType;
    }

    /**
     * Получить путь к заданному inventory-файлу
     *
     * @return путь к файлу
     * @throws IllegalStateException если путь не задан в конфигурации
     */
    @SuppressWarnings("ConstantConditions")
    @Nonnull
    public String getAnsibleInventoryPath() {
        if (inventoryType == InventoryFileType.REGULAR) {
            return ansibleInventoryPath;
        } else {
            throw new IllegalStateException("Configuration doesn't contain inventory file");
        }
    }

    /**
     * Получить содержимое для создания временного inventory-файла
     *
     * @return содержимое inventory-файла
     * @throws IllegalStateException если содержимое не задано в конфигурации
     */
    @SuppressWarnings("ConstantConditions")
    @Nonnull
    public String getAnsibleInventoryContent() {
        if (inventoryType == InventoryFileType.TEMPORARY) {
            return ansibleInventoryContent;
        } else {
            throw new IllegalStateException("Configuration doesn't contain inventory content");
        }
    }

    /**
     * Получить путь к исполняему файлу ansible-playbook
     *
     * @return путь к файлу или {@code null}, если он не был задан в конфигурации
     */
    @Nullable
    public String getAnsiblePlaybookCmd() {
        return ansiblePlaybookCmd;
    }

    /**
     * Получить путь для записи логов (переменной окружения ANSIBLE_LOG_PATH)
     *
     * @return путь для записи логов или {@code null}, если он не был задан в конфигурации
     */
    @Nullable
    public String getAnsibleLogPath() {
        return ansibleLogPath;
    }

    /**
     * Получить путь для задания библиотеки модулей ansible (переменной окружения ANSIBLE_LIBRARY)
     *
     * @return путь к директории или {@code null}, если он не был задан в конфигурации
     */
    @Nullable
    public String getAnsibleLibrary() {
        return ansibleLibrary;
    }

    /**
     * Получить таймаут на выполнение ansible-playbook
     *
     * @return величина таймаута или {@code null}, если таймаут не был задан в конфигурации
     */
    @Nullable
    public Duration getExecutionTimeout() {
        return executionTimeout;
    }

    /**
     * Тип inventory файла
     */
    enum InventoryFileType {
        /**
         * inventory не задан
         */
        NONE,
        /**
         * использовать существующий файл
         */
        REGULAR,
        /**
         * использовать временный файл с заданным содержимым
         */
        TEMPORARY,
    }

    /**
     * Класс для создания {@link AnsibleWrapperConfiguration}
     */
    public static class Builder {
        private String ansiblePlaybookCmd;
        private String ansibleLogPath;
        private String ansibleInventoryPath;
        private String ansibleInventoryContent;
        private String ansibleLibrary;
        private Duration executionTimeout;

        /**
         * @param ansiblePlaybookCmd полный путь до утилиты ansible-playbook
         * @return текущий билдер
         */
        public Builder withAnsiblePlaybookCmd(String ansiblePlaybookCmd) {
            this.ansiblePlaybookCmd = ansiblePlaybookCmd;
            return this;
        }

        /**
         * Задать путь для использования встроенным логированием ansible-playbook
         * Соответствует переменной окружения ANSIBLE_LOG_PATH
         *
         * @param ansibleLogPath путь к лог-файлу
         * @return текущий билдер
         */
        public Builder withAnsibleLogPath(String ansibleLogPath) {
            this.ansibleLogPath = ansibleLogPath;
            return this;
        }

        /**
         * Задать <a href="http://docs.ansible.com/ansible/intro_inventory.html">inventory</a>-файл
         * Соответствует переменной окружения ANSIBLE_INVENTORY
         *
         * @param ansibleInventoryPath путь к inventory-файлу
         * @return текущий билдер
         */
        public Builder withInventoryFile(String ansibleInventoryPath) {
            this.ansibleInventoryPath = ansibleInventoryPath;
            return this;
        }

        /**
         * Задать содержимое для записи во временный inventory-файл
         *
         * @param ansibleInventoryContent содержимое inventory
         * @return текущий билдер
         */
        public Builder withInventoryContent(String ansibleInventoryContent) {
            this.ansibleInventoryContent = ansibleInventoryContent;
            return this;
        }

        /**
         * Задать путь для поиска модулей ansible-playbook
         * Соответствует переменной окружения ANSIBLE_LIBRARY
         *
         * @param ansibleLibrary путь к директории с модулями ansible
         * @return текущий билдер
         */
        public Builder withAnsibleLibrary(String ansibleLibrary) {
            this.ansibleLibrary = ansibleLibrary;
            return this;
        }

        /**
         * Задать таймаут выполнения
         *
         * @param executionTimeout таймаут на выполнение ansible-playbook
         * @return текущий билдер
         */
        public Builder withExecutionTimeout(Duration executionTimeout) {
            this.executionTimeout = executionTimeout;
            return this;
        }

        /**
         * Создать конфигурацию с текущими настройками.
         * <p>
         * InventoryContent имеет приоритет над InventoryFile, при задании обоих значениями, отличными от {@code null}
         * в конфигурации будет доступен только content.
         *
         * @return новый экземпляр конфигурации
         */
        public AnsibleWrapperConfiguration build() {
            return new AnsibleWrapperConfiguration(this);
        }
    }
}
