package ru.yandex.direct.ansiblejuggler;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.ansiblejuggler.model.JugglerPlay;
import ru.yandex.direct.ansiblejuggler.model.JugglerPlaybook;
import ru.yandex.direct.ansiblejuggler.model.checks.AllEventsFromAllHostsCheck;
import ru.yandex.direct.ansiblejuggler.model.checks.JugglerCheck;
import ru.yandex.direct.ansiblejuggler.model.checks.JugglerChild;
import ru.yandex.direct.ansiblejuggler.model.checks.SingleEventFromAnyHostCheck;
import ru.yandex.direct.ansiblejuggler.model.tasks.JugglerCheckTask;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.ansiblejuggler.PlaybookUtils.checkChildrenHost;
import static ru.yandex.direct.ansiblejuggler.PlaybookUtils.checkHost;
import static ru.yandex.direct.ansiblejuggler.PlaybookUtils.checkJcheckMark;
import static ru.yandex.direct.ansiblejuggler.PlaybookUtils.checkName;
import static ru.yandex.direct.ansiblejuggler.PlaybookUtils.checkServices;
import static ru.yandex.direct.ansiblejuggler.PlaybookUtils.checkTtl;

/**
 * Генератор плейбуков с juggler-проверками в формате ansible-juggler.
 * <p>
 * В текущей реализации покрывает собой типовые кейсы проверок в директе:
 * <ul>
 * <li>события (event) с одним именем - должны приходить в течение TTL с любого хоста</li>
 * <li>набор событий с разными именами - должны приходить (для каждого) в течение TTL с любого хоста,
 * это условие должно выполняться для всех имен - результатом будет одна проверка</li>
 * </ul>
 *
 * @see JugglerPlay
 */
@ParametersAreNonnullByDefault
public class PlaybookBuilder {
    /**
     * Значение TTL данных, сгенерированных промежуточными проверками
     */
    public static final Duration INTERMEDIATE_CHECKS_TTL = Duration.ofMinutes(6);

    private final JugglerPlay jugglerPlay;
    private final String namespace;
    private final String jcheckMark;
    private final String intermediateHost;
    private final String targetHost;
    private final List<String> sourceHostOrCGroup;

    /**
     * Создать конструктор проверок с указанными параметрами
     *
     * @param jugglerApiUrl      juggler API url, например {@literal http://dimon.yandex.ru:8998/api}
     * @param namespace          имя пространста имен проверки для ограничения доступа
     * @param jcheckMark         метка для проверок (при очистке удаляются проверки со всех упомянутых
     *                           в плейбуке хостов, у которых метка совпадает с указанной
     *                           и при этом они не были описаны в плейбуке)
     *                           должен проходить проверку {@link PlaybookUtils#checkJcheckMark(String)}
     * @param withCleanup        добавлять ли в плейбук задачу по очистке проверок, отсутствующих в плейбуке
     *                           (по значению {@link #jcheckMark})
     * @param intermediateHost   хост на который будут заведены промежуточные проверки
     *                           должен проходить проверку {@link PlaybookUtils#checkHost(String)}
     * @param targetHost         хост на который будут заведены результирующие проверки
     *                           должен проходить проверку {@link PlaybookUtils#checkHost(String)}
     * @param sourceHostOrCGroup хост (или кондукторная группа, например {@literal CGROUP%direct_java_scripts}
     *                           которая будет подставлена в качестве исходной (источника сигналов) для проверок
     *                           должен проходить проверку {@link PlaybookUtils#checkChildrenHost(String)}
     * @throws NullPointerException     если любой из переданных параметров равен {@code null}
     * @throws IllegalArgumentException при невалидном значении любого из переданных параметров
     */
    public PlaybookBuilder(String jugglerApiUrl, String namespace, String jcheckMark, boolean withCleanup,
                           String intermediateHost, String targetHost, String sourceHostOrCGroup) {
        checkNotNull(jugglerApiUrl);
        checkJcheckMark(jcheckMark);
        checkHost(intermediateHost);
        checkHost(targetHost);
        checkChildrenHost(sourceHostOrCGroup);

        jugglerPlay = new JugglerPlay(jugglerApiUrl);
        this.namespace = namespace;
        this.jcheckMark = jcheckMark;
        if (withCleanup) {
            jugglerPlay.withCleanup(jcheckMark);
        }
        this.intermediateHost = intermediateHost;
        this.targetHost = targetHost;
        this.sourceHostOrCGroup = Collections.singletonList(sourceHostOrCGroup);
    }

    /**
     * Добавить в плейбук проверку на хост {@link #targetHost} с именем {@code service}, проверяющую что
     * "Событие {@code service} приходят с любого из {@link #sourceHostOrCGroup} в течение {@code ttl}"
     *
     * @param service имя исходного и результирующего событий, валидируется {@link PlaybookUtils#checkName(String)}
     * @param ttl     TTL проверки, валидируется {@link PlaybookUtils#checkTtl(Duration)}
     * @return инстанс добавленной проверки
     * @throws NullPointerException     в случае если любой из параметров равен {@code null}
     * @throws IllegalArgumentException при невалидных значениях параметров
     */
    public JugglerCheck addSingleServiceCheck(String service, Duration ttl) {
        return addSingleServiceCheck(targetHost, service, ttl);
    }

    /**
     * Добавить в плейбук проверку на хост {@code host} с именем {@code service}, проверяющую что
     * "Событие {@code service} приходят с любого из {@link #sourceHostOrCGroup} в течение {@code ttl}"
     *
     * @param host    имя целевого хоста проверки
     * @param service имя исходного и результирующего событий, валидируется {@link PlaybookUtils#checkName(String)}
     * @param ttl     TTL проверки, валидируется {@link PlaybookUtils#checkTtl(Duration)}
     * @return инстанс добавленной проверки
     * @throws NullPointerException     в случае если любой из параметров равен {@code null}
     * @throws IllegalArgumentException при невалидных значениях параметров
     */
    public JugglerCheck addSingleServiceCheck(String host, String service, Duration ttl) {
        checkName(service);
        checkTtl(ttl);

        JugglerCheck checkParams =
                new SingleEventFromAnyHostCheck(host, service, sourceHostOrCGroup, ttl, namespace, jcheckMark);
        jugglerPlay.withTask(new JugglerCheckTask(String.format("%s (aggregated check - auto)", service), checkParams));

        return checkParams;
    }

    /**
     * Добавляет в плейбук проверку на хост {@code host} с именем {@code targetService}, проверяющую что
     * "с любого {@link #sourceHostOrCGroup} по каждому {@code sourceService} приходят события в течение {@code ttl}"
     * <p>
     * Кроме того, на хост {@link #intermediateHost} добавляются промежуточные проверки по числу {@code sourceService}
     *
     * @param host           имя целевого хоста проверки
     * @param sourceServices список исходных событий для проверки
     *                       валидируется {@link PlaybookUtils#checkServices(List)}
     * @param targetService  имя результирующей проверки
     *                       валидируется {@link PlaybookUtils#checkName(String)}
     * @param ttl            TTL проверки
     *                       валидируется {@link PlaybookUtils#checkTtl(Duration)}
     * @return инстанс добавленной "аггрегирующей" (не промежуточных) проверки
     * @throws NullPointerException     в случае если любой из параметров равен {@code null}
     * @throws IllegalArgumentException при невалидных значениях параметров
     */
    public JugglerCheck addMultiServicesCheck(String host, List<String> sourceServices, String targetService,
                                              Duration ttl) {
        checkServices(sourceServices);
        checkName(targetService);
        checkTtl(ttl);

        JugglerCheck checkParams =
                new AllEventsFromAllHostsCheck(host, targetService, INTERMEDIATE_CHECKS_TTL, namespace, jcheckMark);

        for (String service : sourceServices) {
            JugglerCheck subcheckParams =
                    new SingleEventFromAnyHostCheck(intermediateHost, service, sourceHostOrCGroup, ttl, namespace,
                            jcheckMark);
            jugglerPlay.withTask(new JugglerCheckTask(String.format("%s (subcheck)", service), subcheckParams));
            checkParams.withChild(new JugglerChild(intermediateHost, service));
        }

        jugglerPlay.withTask(
                new JugglerCheckTask(String.format("%s (aggregated check - auto)", targetService), checkParams));

        return checkParams;
    }

    public JugglerCheck addTopLevelCheck(List<String> services, String topLevelService) {
        checkServices(services);
        checkName(topLevelService);

        JugglerCheck check = new AllEventsFromAllHostsCheck(targetHost, topLevelService,
                INTERMEDIATE_CHECKS_TTL, namespace, jcheckMark);

        for (String service : services) {
            check.withChild(new JugglerChild(targetHost, service));
        }

        jugglerPlay.withTask(
                new JugglerCheckTask(String.format("%s (top level check - auto)", topLevelService), check));

        return check;
    }

    /**
     * Добавляет в плейбук проверку на хост {@code host} с именем {@code targetService}, проверяющую что
     * "с любого {@code host} по каждому {@code sourceService} приходят события в течение {@code ttl}"
     *
     * @param host           имя целевого хоста проверки
     * @param sourceServices список исходных событий для проверки
     *                       валидируется {@link PlaybookUtils#checkServices(List)}
     * @param targetService  имя результирующей проверки
     *                       валидируется {@link PlaybookUtils#checkName(String)}
     * @param ttl            TTL проверки
     *                       валидируется {@link PlaybookUtils#checkTtl(Duration)}
     * @return инстанс добавленной "аггрегирующей" (не промежуточных) проверки
     * @throws NullPointerException     в случае если любой из параметров равен {@code null}
     * @throws IllegalArgumentException при невалидных значениях параметров
     */
    public JugglerCheck addNumericMultiServicesCheck(String host, List<String> sourceServices, String targetService,
                                                     Duration ttl) {
        checkServices(sourceServices);
        checkName(targetService);
        checkTtl(ttl);

        JugglerCheck checkParams =
                new AllEventsFromAllHostsCheck(host, targetService, ttl, namespace, jcheckMark);

        if (sourceServices.size() == 1 && sourceServices.get(0).equals(targetService)) {
            // особый сценарий, когда проверка нужна ровно такая же, как сырое событие.
            // чтобы исключить циклическую зависимость - не добавляем детей, это неявно сделает сам juggler
            checkParams.withNoChild();
        } else {
            for (String service : sourceServices) {
                checkParams.withChild(new JugglerChild(host, service));
            }
        }

        jugglerPlay.withTask(
                new JugglerCheckTask(String.format("%s (aggregated check on %s)", targetService, host), checkParams));

        return checkParams;
    }

    /**
     * Добавляет в плейбук проверку на хост {@link #targetHost} с именем {@code targetService}, проверяющую что
     * "с любого {@link #sourceHostOrCGroup} по каждому {@code sourceService} приходят события в течение {@code ttl}"
     * <p>
     * Кроме того, на хост {@link #intermediateHost} добавляются промежуточные проверки по числу {@code sourceService}
     *
     * @param sourceServices список исходных событий для проверки
     *                       валидируется {@link PlaybookUtils#checkServices(List)}
     * @param targetService  имя результирующей проверки
     *                       валидируется {@link PlaybookUtils#checkName(String)}
     * @param ttl            TTL проверки
     *                       валидируется {@link PlaybookUtils#checkTtl(Duration)}
     * @return инстанс добавленной "аггрегирующей" (не промежуточных) проверки
     * @throws NullPointerException     в случае если любой из параметров равен {@code null}
     * @throws IllegalArgumentException при невалидных значениях параметров
     */
    public JugglerCheck addMultiServicesCheck(List<String> sourceServices, String targetService, Duration ttl) {
        return addMultiServicesCheck(targetHost, sourceServices, targetService, ttl);
    }

    /**
     * Добавляет в плейбук проверку на хост {@link #targetHost} с именем {@code targetService} проверяющую,
     * что "каждое дочерние событие приходит в течение {@code ttl}"
     *
     * @param targetService имя результирующей проверки
     *                      валидируется {@link PlaybookUtils#checkName(String)}
     * @param children      дочерние события
     * @param ttl           TTL проверки
     *                      валидируется {@link PlaybookUtils#checkTtl(Duration)}
     * @return инстанс добавленной "аггрегирующей" проверки
     * @throws NullPointerException     в случае если любой из параметров равен {@code null}
     * @throws IllegalArgumentException при невалидных значениях параметров
     */
    public JugglerCheck addAllEventsFromAllHostsCheck(String targetService,
                                                      Collection<JugglerChild> children,
                                                      Duration ttl) {
        String taskName = String.format("%s (AllEventsFromAllHostsCheck check on %s)", targetService, targetHost);
        JugglerCheck check = new AllEventsFromAllHostsCheck(targetHost, targetService, ttl, namespace, jcheckMark)
                .withChildren(children);
        jugglerPlay.withTask(new JugglerCheckTask(taskName, check));
        return check;
    }

    /**
     * Добавляет в плейбук проверку на хост {@link #targetHost} с именем {@code targetService} проверяющую,
     * что "с любого хоста из {@code sourceHostsOrCGroups} событие приходит в течение {@code ttl}"
     *
     * @param targetService имя результирующей проверки
     * @param children      дочерние события
     * @param ttl           TTL проверки
     *                      валидируется {@link PlaybookUtils#checkTtl(Duration)}
     * @return инстанс добавленной проверки
     * * @throws NullPointerException     в случае если любой из параметров равен {@code null}
     * * @throws IllegalArgumentException при невалидных значениях параметров
     */
    public JugglerCheck addSingleEventFromAnyHostCheck(String targetService,
                                                       Collection<JugglerChild> children,
                                                       Duration ttl) {
        String taskName = String.format("%s (SingleEventFromAnyHostCheck check on %s)", targetService, targetHost);
        JugglerCheck check = new SingleEventFromAnyHostCheck(targetHost, targetService, children, ttl, namespace,
                jcheckMark);

        jugglerPlay.withTask(new JugglerCheckTask(taskName, check));
        return check;
    }

    /**
     * Сформировать плейбук
     *
     * @return новый {@link JugglerPlaybook}
     */
    public JugglerPlaybook build() {
        return jugglerPlay.asPlaybook();
    }
}
