package ru.yandex.direct.ansiblejuggler;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;

import static com.fasterxml.jackson.dataformat.yaml.YAMLGenerator.Feature.MINIMIZE_QUOTES;
import static com.fasterxml.jackson.dataformat.yaml.YAMLGenerator.Feature.WRITE_DOC_START_MARKER;
import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

@ParametersAreNonnullByDefault
public class PlaybookUtils {
    private static final Pattern JCHECK_PATTERN = Pattern.compile("[a-zA-Z0-9_]+");
    private static final Pattern NAME_OR_HOST_PATTERN = Pattern.compile("[a-zA-Z0-9_.-]+");
    private static final Pattern HOST_OR_CGROUP_PATTERN = Pattern.compile("(?:(CGROUP|DEPLOY)%)?[a-zA-Z0-9_.\\-@=%]+");

    private static final Duration MIN_TTL = Duration.of(1, ChronoUnit.MINUTES);
    /**
     * Максимальный TTL. это ограничение со стороны API juggler
     */
    private static final Duration MAX_TTL = Duration.of(10, ChronoUnit.DAYS);

    private static final int MAX_NAME_SIZE = 128;

    private PlaybookUtils() {
    }

    public static ObjectMapper getMapper() {
        return new ObjectMapper(new YAMLFactory().enable(MINIMIZE_QUOTES).enable(WRITE_DOC_START_MARKER));
    }

    public static void checkJcheckMark(String mark) {
        checkNotNull(mark, "jcheckMark cannot be null");
        checkArgument(JCHECK_PATTERN.matcher(mark).matches(), "Incorrect jcheckMark '%s'", mark);
    }

    public static void checkName(String name) {
        checkNotNull(name, "name cannot be null");
        checkArgument(NAME_OR_HOST_PATTERN.matcher(name).matches(), "Incorrect name '%s'. Allowed characters are %s",
                name, NAME_OR_HOST_PATTERN);
        checkArgument(name.length() < MAX_NAME_SIZE,
                "Incorrect name size '%s'. Allowed only %s characters", name, MAX_NAME_SIZE);
    }

    public static void checkServices(List<String> services) {
        checkNotNull(services, "List of services cannot be null");
        checkArgument(!services.isEmpty(), "List of services cannot be empty");
        services.forEach(PlaybookUtils::checkName);
    }

    public static void checkHost(String host) {
        checkNotNull(host, "host cannot be null");
        checkArgument(NAME_OR_HOST_PATTERN.matcher(host).matches(), "Incorrect host '%s'. Allowed characters are %s",
                host, NAME_OR_HOST_PATTERN);
    }

    public static void checkChildrenHost(String hostOrCGroup) {
        checkNotNull(hostOrCGroup, "host or CGroup cannot be null");
        checkArgument(HOST_OR_CGROUP_PATTERN.matcher(hostOrCGroup).matches(),
                "Incorrect host or CGroup '%s'. Allowed characters are %s", hostOrCGroup, HOST_OR_CGROUP_PATTERN);
    }

    public static void checkChildrenHost(List<String> hostOrCGroup) {
        checkNotNull(hostOrCGroup, "List of hosts or CGroups cannot be null");
        checkArgument(!hostOrCGroup.isEmpty(), "List of hosts or CGroups cannot be empty");
        hostOrCGroup.forEach(PlaybookUtils::checkChildrenHost);
    }

    public static void checkTtl(Duration ttl) {
        checkNotNull(ttl, "ttl cannot be null");
        checkArgument(ttl.compareTo(MIN_TTL) > 0, "ttl should be greater than %s", MIN_TTL);
        checkArgument(ttl.compareTo(MAX_TTL) <= 0, "ttl should be less than or equal to %s", MAX_TTL);
    }
}
